# Bongo Asset Package

Centralized frontend asset package for the Bongo Laravel ecosystem. Provides compiled JavaScript, CSS, and Vue.js components for admin interfaces across 30+ Bongo packages.

## Features

- **Vue 2 Components**: 50+ reusable components for admin UIs
- **TailwindCSS Theme**: Custom brand colors and utilities
- **Global Utilities**: Filters, directives, and event bus
- **WYSIWYG Editor**: Redactor with plugins
- **Build System**: Laravel Mix with code splitting
- **Consistent UI**: Shared assets across all Bongo packages

## Requirements

- PHP >= 8.2
- Laravel >= 10.0
- Node.js (for building assets)
- npm or yarn

## Installation

### 1. Install via Composer

```bash
composer require bongo/asset
```

### 2. Publish Assets

```bash
php artisan vendor:publish --tag=bongo:assets
```

This publishes compiled assets to your Laravel application's `public/` directory:
- `public/css/backend.css`
- `public/js/backend.js`
- `public/images/*`

### 3. Include in Blade Templates

```blade
<!DOCTYPE html>
<html>
<head>
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <meta name="api-token" content="{{ auth()->user()?->api_token }}">
    <link rel="stylesheet" href="{{ asset('css/backend.css') }}">
</head>
<body>
    <div id="app">
        <v-container>
            <v-header>
                <v-header-menu></v-header-menu>
            </v-header>
            <v-sidebar></v-sidebar>
            <v-main>
                <!-- Your content here -->
            </v-main>
        </v-container>
    </div>
    <script src="{{ asset('js/backend.js') }}"></script>
</body>
</html>
```

## Usage

### Available Components

#### Layout Components
```vue
<v-container>
<v-header>
<v-header-menu>
<v-sidebar>
<v-main>
```

#### UI Components
```vue
<v-alert type="success">Message</v-alert>
<v-button link="/url">Click</v-button>
<v-card>Content</v-card>
<v-combobox :options="[]" v-model="selected"></v-combobox>
<v-colorpicker v-model="color"></v-colorpicker>
<v-datepicker v-model="date"></v-datepicker>
<v-error :errors="errors" field="name"></v-error>
<v-label>Label</v-label>
```

#### Feature Components (Builders)
```vue
<data-table endpoint="/api/items"></data-table>
<form-builder endpoint="/api/forms"></form-builder>
<menu-builder endpoint="/api/menus"></menu-builder>
<gallery-builder endpoint="/api/galleries"></gallery-builder>
<image-manager endpoint="/api/images"></image-manager>
<document-manager endpoint="/api/documents"></document-manager>
```

#### Editor Components
```vue
<editor v-model="content"></editor>
<css-editor v-model="css"></css-editor>
<js-editor v-model="js"></js-editor>
<schema-editor v-model="schema"></schema-editor>
<file-uploader endpoint="/api/upload"></file-uploader>
```

### Global Filters

```vue
{{ date | date }}                    <!-- "Jan 19th 2026" -->
{{ 0.15 | percent }}                 <!-- "15%" -->
{{ 1234.56 | currency }}             <!-- "1,234.56" -->
{{ text | truncate(50, '...') }}     <!-- Truncated text -->
{{ text | uppercase }}               <!-- UPPERCASE -->
{{ text | ucwords }}                 <!-- Title Case -->
```

### Global Directives

```vue
<input v-uppercase>                  <!-- Force uppercase -->
<input v-alpha>                      <!-- Letters only -->
<input v-numeric>                    <!-- Numbers/decimals -->
<input v-alphanumeric>               <!-- Letters, numbers, !, *, % -->
<input v-currency>                   <!-- Currency format -->
<input v-date>                       <!-- MM/DD/YYYY -->
<input v-maxnumber="100">            <!-- Max value: 100 -->
<input v-maxchars="50">              <!-- Max length: 50 -->
```

## Development

### Building Assets

Navigate to the `src/` directory:

```bash
cd vendor/bongo/asset/src

# Install dependencies
npm install

# Development build
npm run dev

# Watch mode
npm run watch

# Production build
npm run production
```

### Modifying Source Files

1. Edit files in `src/resources/js/` or `src/resources/sass/`
2. Run `npm run watch` to rebuild on changes
3. Refresh browser to see changes

### Publishing Updated Assets

After rebuilding:

```bash
php artisan vendor:publish --tag=bongo:assets --force
```

## Architecture

### Directory Structure

```
src/
├── AssetServiceProvider.php     # Service provider
├── package.json                 # npm dependencies
├── webpack.mix.js               # Build configuration
├── tailwind.config.js           # Tailwind theme
├── public/                      # Compiled assets
│   ├── css/backend.css
│   ├── js/backend.js
│   └── images/
└── resources/                   # Source files
    ├── js/
    │   ├── backend.js           # Vue app entry
    │   ├── layout/              # Layout components
    │   ├── components/          # Feature components
    │   └── vendor/              # Third-party JS
    ├── sass/
    │   ├── backend.scss         # Sass entry
    │   ├── components/          # Custom styles
    │   └── vendor/              # Third-party overrides
    └── images/                  # Source images
```

### Technology Stack

- **Vue 2.5.17**: Component framework
- **TailwindCSS 1.4.6**: Utility-first CSS
- **Laravel Mix 5.x**: Build tool
- **Redactor**: WYSIWYG editor
- **Axios**: HTTP client
- **SweetAlert2**: Alert dialogs
- **vuedraggable**: Drag and drop

## Documentation

- **[ARCHITECTURE.md](ARCHITECTURE.md)**: Detailed architecture documentation with diagrams
- **[CLAUDE.md](CLAUDE.md)**: Quick reference guide for Claude Code
- **[.cursorrules](.cursorrules)**: Development guidelines for Cursor AI
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)**: Code templates for GitHub Copilot

## Testing

```bash
# PHP tests
vendor/bin/phpunit

# Code style
vendor/bin/pint
vendor/bin/pint --test

# Static analysis
vendor/bin/phpstan analyse
```

## Browser Compatibility

- Modern browsers with ES6+ support
- Mobile-responsive design
- VTooltip disabled on viewports ≤ 768px

## Contributing

This package is part of the Bongo monorepo at `/Users/stuart/Packages/bongo/`. Each package is a separate git repository.

### Workflow

1. Make changes in this package
2. Build assets: `npm run production`
3. Test in consuming application
4. Commit changes
5. Publish to consuming apps: `php artisan vendor:publish --tag=bongo:assets --force`

## License

MIT License. See [LICENSE](LICENSE) for details.

## Support

For issues or questions:
- Review the [ARCHITECTURE.md](ARCHITECTURE.md) documentation
- Check the troubleshooting section in [CLAUDE.md](CLAUDE.md)
- Contact: stuart.elliott@bespokeuk.com

## Related Packages

This package provides assets for:
- `bongo/framework` - Core framework
- `cms/*` - 12 CMS packages (builder, page, menu, etc.)
- `default/*` - 14 default packages (blade, user, dashboard, etc.)
- `custom/*` - 3 custom packages

All Bongo packages use these assets for consistent admin UI.
