# GitHub Copilot Instructions - Bongo Blade Package

## Project Overview

This Laravel package provides reusable Blade components and custom directives for form inputs, buttons, icons, and UI elements. All components use Tailwind CSS and integrate with Laravel's validation system.

## Key Classes and Relationships

### BladeServiceProvider
- **Location**: `src/BladeServiceProvider.php`
- **Extends**: `Bongo\Framework\Providers\AbstractServiceProvider`
- **Purpose**: Registers the BladeManager service and bootstraps the package
- **Key Properties**: `protected string $module = 'blade'`
- **Methods**:
  - `register()`: Binds `BladeManager` to container as `blade_manager`
  - `boot()`: Calls `BladeManager::register()` to set up all directives

### BladeManager
- **Location**: `src/Services/BladeManager.php`
- **Purpose**: Central registration point for all Blade directives
- **Key Method**: `register()` - Orchestrates all directive registrations
- **Registration Methods**:
  - `registerFormGroupDirective()`: Custom `@formgroup` / `@endformgroup` structural directives
  - `register{Component}Directive()`: Uses `Blade::include()` to map directives to views

**Registration Pattern**:
```php
protected function registerInputDirective(): void
{
    Blade::include('blade::input', 'input');
    // Creates @input directive that renders blade::input view
}
```

## Component Categories

### Form Components
| Directive | View File | Purpose |
|-----------|-----------|---------|
| `@input` | `input.blade.php` | Text/number/email inputs with validation |
| `@select` | `select.blade.php` | Dropdown select with options |
| `@multiSelect` | `multiselect.blade.php` | Multi-select dropdown |
| `@checkbox` | `checkbox.blade.php` | Checkbox with hidden input for unchecked state |
| `@textarea` | `textarea.blade.php` | Multi-line text input |
| `@wysiwyg` | `wysiwyg.blade.php` | WYSIWYG editor integration |
| `@label` | `label.blade.php` | Form label with auto-generation |
| `@static` | `static.blade.php` | Read-only field display |

### Button Components
| Directive | View File | Icon | Default Text |
|-----------|-----------|------|--------------|
| `@createButton` | `button_create.blade.php` | Plus icon | "Create" |
| `@editButton` | `button_edit.blade.php` | Pencil icon | "Edit" |
| `@showButton` | `button_show.blade.php` | Eye icon | "View" |
| `@saveButton` | `button_save.blade.php` | Check icon | "Save" |
| `@deleteButton` | `button_delete.blade.php` | Trash icon | "Delete" |
| `@backButton` | `button_back.blade.php` | Arrow left | "Back" |
| `@linkButton` | `button_link.blade.php` | Link icon | "Link" |

### UI Components
- **Card**: `card.blade.php` - Container with optional header/footer
- **Breadcrumbs**: `breadcrumbs.blade.php` - Navigation breadcrumb trail
- **Message**: `message.blade.php` - Flash message display (success/warning/danger/info)
- **Error**: `error.blade.php` - Validation error display for individual fields

### Icon Components
18 SVG icons in `src/Views/icons/`:
`dashboard`, `user`, `page`, `post`, `menu`, `gallery`, `form`, `image`, `component`, `layout`, `document`, `estimate`, `project`, `review`, `design`, `setting`, `lock`, `redirect`

## Code Style Templates

### Adding a New Form Component

```php
// 1. Create src/Views/custom_input.blade.php
<div class="form-group">
    @include('blade::label')

    <div class="mt-1 relative rounded-sm shadow-sm">
        <input
            id="{{ str($id ?? $name)->slug() }}"
            type="text"
            name="{{ $name }}"
            value="{{ old($name, $value ?? null) }}"
            class="form-input block w-full rounded-sm sm:text-sm sm:leading-5 {{ $errors->has($name) ? 'border-red-300 text-red-900' : '' }}"
            {{ isset($required) ? 'required' : '' }}
        />
        @include('blade::error_icon')
    </div>

    @include('blade::error')
</div>

// 2. Register in BladeManager
protected function registerCustomInputDirective(): void
{
    Blade::include('blade::custom_input', 'customInput');
}

// 3. Add to register() method
public function register(): void
{
    // ... existing registrations
    $this->registerCustomInputDirective();
}
```

### Adding a New Button

```php
// src/Views/button_archive.blade.php
<span class="inline-flex rounded-sm shadow-sm {{ $class ?? '' }}">
    <a class="inline-flex items-center px-4 py-2 border border-transparent text-sm leading-5 font-medium rounded-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none transition duration-150 ease-in-out"
       href="{{ $url }}"
       title="{{ $name ?? 'Archive' }}"
       aria-label="{{ $name ?? 'Archive' }}"
       v-tooltip='"Archive this item"'
    >
        <svg class="-ml-1 mr-2 h-5 w-5" fill="currentColor" viewBox="0 0 20 20">
            <!-- SVG path here -->
        </svg>
        {{ $name ?? 'Archive' }}
    </a>
</span>

// Register in BladeManager
protected function registerArchiveButtonDirective(): void
{
    Blade::include('blade::button_archive', 'archiveButton');
}
```

### Using Components in Views

```blade
{{-- Form inputs --}}
@input([
    'label' => 'Email Address',
    'name' => 'email',
    'type' => 'email',
    'value' => $user->email,
    'required' => true,
    'placeholder' => 'user@example.com'
])

@select([
    'label' => 'Department',
    'name' => 'department_id',
    'value' => $user->department_id,
    'options' => $departments->pluck('name', 'id')->toArray(),
    'placeholder' => 'Select a department',
    'required' => true
])

@checkbox([
    'label' => 'Active User',
    'name' => 'is_active',
    'value' => $user->is_active
])

@textarea([
    'label' => 'Biography',
    'name' => 'bio',
    'value' => $user->bio,
    'rows' => 5
])

{{-- Buttons --}}
@createButton(['name' => 'New User', 'url' => route('users.create')])
@editButton(['url' => route('users.edit', $user)])
@deleteButton(['url' => route('users.destroy', $user)])

{{-- Card component --}}
<x-blade::card header="User Profile" footer="Last updated: {{ $user->updated_at }}">
    User content here
</x-blade::card>

{{-- Breadcrumbs --}}
@include('blade::breadcrumbs', ['breadcrumbs' => [
    ['url' => route('home'), 'label' => 'Home'],
    ['url' => route('users.index'), 'label' => 'Users'],
    'Edit User'
]])

{{-- Flash messages --}}
@include('blade::message')
```

## Common Patterns

### Parameter Handling
All components follow these conventions:
- Parameters passed as associative arrays
- `$name` is typically required for form fields
- `$label` auto-generates from `$name` if not provided (converts snake_case to Title Case)
- `$value` integrates with `old($name, $value ?? null)` for form repopulation
- `$errors` integration for validation error display
- `$class` for additional CSS classes
- `$id` defaults to slugified version of `$name`

### Validation Error States
Components automatically apply error styling when `$errors->has($name)`:
```blade
class="{{ $errors->has($name) ? 'border-red-300 text-red-900 placeholder-red-300 focus:border-red-300' : '' }}"
```

### Label Auto-Generation
```blade
{{ $label ?? ucwords(str_replace(']', '', str_replace('[', '', str_replace('_', ' ', $name)))) }}
```
Converts `user_first_name` → "User First Name"

### Required Field Indicators
```blade
@if ($required ?? false)
    <span class="text-red-500">&ast;</span>
@endif
```

### Hidden Checkbox Pattern
Checkboxes include hidden input to ensure "0" is submitted when unchecked:
```blade
<input type="hidden" name="{{ $name }}" value="0">
<input type="checkbox" name="{{ $name }}" value="1" {{ old($name, $value == 1) ? 'checked' : '' }} />
```

## Key Dependencies

- `bongo/framework` ^3.0 - Provides `AbstractServiceProvider` base class
- Tailwind CSS - All components use Tailwind utility classes
- Laravel Validation - `$errors` view bag integration
- Laravel Session - Flash messages via `session('message')` and `session('message_type')`

## Testing Patterns

```php
use Bongo\Blade\Tests\TestCase;

class ComponentTest extends TestCase
{
    public function test_input_directive_renders_correctly()
    {
        $view = $this->blade(
            '@input(["name" => "email", "label" => "Email", "value" => "test@example.com"])'
        );

        $this->assertStringContainsString('name="email"', $view);
        $this->assertStringContainsString('value="test@example.com"', $view);
        $this->assertStringContainsString('Email', $view);
    }
}
```

## Important Conventions

- All form components include `@include('blade::error')` for field-level error display
- Buttons use `inline-flex` for consistent alignment
- SVG icons use `currentColor` for theming
- All components support error states via `$errors->has($name)`
- ID attributes are auto-generated but can be overridden
- Components use `{{ }}` for escaped output, `{!! !!}` only when HTML rendering is required
