# Bongo Blade Package - Architecture Documentation

## Overview

The Bongo Blade package is a Laravel package that provides reusable Blade components and custom directives for building consistent, accessible forms and UI elements. It leverages Laravel's Blade templating system and integrates seamlessly with Laravel's validation and form handling.

## Directory Structure

```
blade/
├── src/
│   ├── BladeServiceProvider.php           # Service provider, extends AbstractServiceProvider
│   ├── Services/
│   │   └── BladeManager.php               # Central directive registration manager
│   └── Views/                             # Blade component templates (blade:: namespace)
│       ├── Form Components
│       │   ├── input.blade.php            # Text/number/email inputs
│       │   ├── select.blade.php           # Single-select dropdown
│       │   ├── multiselect.blade.php      # Multi-select dropdown
│       │   ├── checkbox.blade.php         # Checkbox with hidden fallback
│       │   ├── textarea.blade.php         # Multi-line text input
│       │   ├── wysiwyg.blade.php          # WYSIWYG editor integration
│       │   ├── label.blade.php            # Form labels with auto-generation
│       │   └── static.blade.php           # Read-only field display
│       ├── Button Components
│       │   ├── button_create.blade.php    # Create action button
│       │   ├── button_edit.blade.php      # Edit action button
│       │   ├── button_show.blade.php      # View/show action button
│       │   ├── button_save.blade.php      # Save/submit button
│       │   ├── button_delete.blade.php    # Delete action button
│       │   ├── button_back.blade.php      # Back/cancel button
│       │   └── button_link.blade.php      # Generic link button
│       ├── UI Components
│       │   ├── card.blade.php             # Container with header/footer
│       │   ├── breadcrumbs.blade.php      # Breadcrumb navigation
│       │   ├── message.blade.php          # Flash message display
│       │   ├── error.blade.php            # Field validation error
│       │   ├── error_icon.blade.php       # Error state icon overlay
│       │   ├── content_header.blade.php   # Page content header
│       │   └── content_body.blade.php     # Page content body wrapper
│       └── icons/                         # SVG icon components
│           ├── dashboard.blade.php
│           ├── user.blade.php
│           ├── page.blade.php
│           ├── post.blade.php
│           ├── menu.blade.php
│           ├── gallery.blade.php
│           ├── form.blade.php
│           ├── image.blade.php
│           ├── component.blade.php
│           ├── layout.blade.php
│           ├── document.blade.php
│           ├── estimate.blade.php
│           ├── project.blade.php
│           ├── review.blade.php
│           ├── design.blade.php
│           ├── setting.blade.php
│           ├── lock.blade.php
│           └── redirect.blade.php
├── tests/
│   ├── TestCase.php                       # Orchestra Testbench base test case
│   ├── Unit/.gitkeep
│   └── Feature/.gitkeep
├── composer.json                          # Package dependencies and autoloading
├── phpunit.xml                            # PHPUnit configuration
└── .php_cs                                # PHP-CS-Fixer configuration (legacy)
```

## Class Diagrams

### Core Service Architecture

```
┌─────────────────────────────────────────┐
│  AbstractServiceProvider                │
│  (bongo/framework)                      │
├─────────────────────────────────────────┤
│  + boot(): void                         │
│  + register(): void                     │
│  # registerViews(): void                │
│  # $module: string                      │
└─────────────────────────────────────────┘
                    ▲
                    │ extends
                    │
┌─────────────────────────────────────────┐
│  BladeServiceProvider                   │
├─────────────────────────────────────────┤
│  # $module = 'blade'                    │
├─────────────────────────────────────────┤
│  + register(): void                     │
│  + boot(): void                         │
└─────────────────────────────────────────┘
                    │
                    │ binds & resolves
                    ▼
┌─────────────────────────────────────────┐
│  BladeManager                           │
├─────────────────────────────────────────┤
│  + register(): void                     │
│  # registerFormGroupDirective(): void   │
│  # registerLabelDirective(): void       │
│  # registerStaticDirective(): void      │
│  # registerCheckboxDirective(): void    │
│  # registerInputDirective(): void       │
│  # registerSelectDirective(): void      │
│  # registerMultiSelectDirective(): void │
│  # registerTextareaDirective(): void    │
│  # registerWysiwygDirective(): void     │
│  # registerCreateButtonDirective(): void│
│  # registerEditButtonDirective(): void  │
│  # registerShowButtonDirective(): void  │
│  # registerSaveButtonDirective(): void  │
│  # registerDeleteButtonDirective(): void│
│  # registerBackButtonDirective(): void  │
│  # registerLinkButtonDirective(): void  │
└─────────────────────────────────────────┘
                    │
                    │ registers directives via
                    │ Blade::directive() & Blade::include()
                    ▼
┌─────────────────────────────────────────┐
│  Laravel Blade Compiler                 │
└─────────────────────────────────────────┘
```

### View Component Hierarchy

```
blade:: namespace (src/Views/)
│
├── Form Components
│   ├── input.blade.php ──────► includes: label, error_icon, error
│   ├── select.blade.php ─────► includes: label, error
│   ├── multiselect.blade.php ► includes: label, error
│   ├── checkbox.blade.php ───► includes: error
│   ├── textarea.blade.php ───► includes: label, error
│   ├── wysiwyg.blade.php ────► includes: error
│   ├── static.blade.php ─────► includes: label
│   └── label.blade.php (shared)
│
├── Button Components (standalone)
│   ├── button_create.blade.php
│   ├── button_edit.blade.php
│   ├── button_show.blade.php
│   ├── button_save.blade.php
│   ├── button_delete.blade.php
│   ├── button_back.blade.php
│   └── button_link.blade.php
│
├── UI Components
│   ├── card.blade.php (uses slots)
│   ├── breadcrumbs.blade.php
│   ├── message.blade.php (session flash messages)
│   ├── error.blade.php (field errors)
│   ├── error_icon.blade.php (shared)
│   ├── content_header.blade.php
│   └── content_body.blade.php
│
└── icons/ (SVG partials, 18 icons)
    └── *.blade.php
```

## Lifecycle and Flow Diagrams

### Package Bootstrap Lifecycle

```
┌──────────────────────────────────────────────┐
│  Laravel Application Boot                    │
└──────────────────────────────────────────────┘
                    │
                    ▼
┌──────────────────────────────────────────────┐
│  BladeServiceProvider::register()            │
│  - Calls parent::register()                  │
│  - Binds BladeManager to container           │
│    as 'blade_manager'                        │
└──────────────────────────────────────────────┘
                    │
                    ▼
┌──────────────────────────────────────────────┐
│  AbstractServiceProvider::boot()             │
│  - Auto-registers views from src/Views/      │
│    under 'blade::' namespace                 │
└──────────────────────────────────────────────┘
                    │
                    ▼
┌──────────────────────────────────────────────┐
│  BladeServiceProvider::boot()                │
│  - Resolves 'blade_manager' from container   │
│  - Calls BladeManager::register()            │
└──────────────────────────────────────────────┘
                    │
                    ▼
┌──────────────────────────────────────────────┐
│  BladeManager::register()                    │
│  - Registers @formgroup directive            │
│  - Registers all component includes:         │
│    * Blade::include('blade::input', 'input') │
│    * Blade::include('blade::select', 'select'│
│    * ... (15+ directives)                    │
└──────────────────────────────────────────────┘
                    │
                    ▼
┌──────────────────────────────────────────────┐
│  Directives Available in Views               │
│  @input(['name' => 'email'])                 │
│  @select(['name' => 'role'])                 │
│  @createButton(['url' => '/create'])         │
└──────────────────────────────────────────────┘
```

### Component Rendering Flow

```
User's Blade View
     │
     │ @input(['name' => 'email', 'label' => 'Email', 'value' => 'test@example.com'])
     │
     ▼
Blade Compiler
     │
     │ Resolves 'input' directive → blade::input view
     │
     ▼
blade::input.blade.php
     │
     ├──► @include('blade::label') ──► Renders label with auto-generation
     │                                  │
     │                                  └─► Checks $required, adds asterisk
     │
     ├──► <input> tag ──────────────► Applies classes, error states
     │                                  │
     │                                  ├─► Checks $errors->has($name)
     │                                  ├─► Sets value with old($name, $value)
     │                                  └─► Applies $required, $disabled, etc.
     │
     ├──► @include('blade::error_icon') ──► Shows error icon if $errors->has($name)
     │
     └──► @include('blade::error') ──► Displays validation error message
                                        │
                                        └─► Renders $errors->first($name)
```

### Validation Error Handling Flow

```
Form Submission
     │
     ▼
Laravel Controller Validation
     │
     │ $request->validate([...])
     │ Validation fails
     │
     ▼
Redirect Back with Errors
     │
     │ Session flash: $errors (ViewErrorBag)
     │
     ▼
Form View Re-render
     │
     ├──► @input(['name' => 'email'])
     │         │
     │         ├──► Checks $errors->has('email') → true
     │         ├──► Applies error classes (border-red-300, text-red-900)
     │         ├──► Shows error_icon.blade.php
     │         └──► Shows error.blade.php with $errors->first('email')
     │
     └──► @include('blade::message')
               │
               └──► Checks session('message') and session('message_type')
                    └──► Renders success/warning/danger/info notification
```

## Component Reference Tables

### Form Component Parameters

| Component | Required Params | Optional Params | Special Features |
|-----------|----------------|-----------------|------------------|
| `@input` | `name` | `label`, `value`, `type`, `placeholder`, `maxlength`, `step`, `min`, `required`, `disabled`, `class`, `id`, `custom` | Supports text, number, email, etc. via `type` |
| `@select` | `name`, `options` | `label`, `value`, `placeholder`, `default`, `multiple`, `required`, `class`, `id`, `attributes` | Options as `key => value` array |
| `@multiSelect` | `name`, `options`, `value` (array) | `label`, `placeholder`, `default`, `required`, `class`, `id`, `size`, `attributes` | Fixed `multiple` attribute |
| `@checkbox` | `name` | `label`, `value` | Hidden input with value="0" for unchecked state |
| `@textarea` | `name` | `label`, `value`, `placeholder`, `rows`, `required`, `disabled`, `class` | Standard textarea |
| `@wysiwyg` | `name` | `value`, `placeholder`, `rows`, `height`, `class`, `resizable`, `disabled`, `required` | WYSIWYG editor integration, `data-url` for image uploads |
| `@label` | `name` | `label`, `id`, `required`, `label_class`, `hideLabel` | Auto-generates label text from name |
| `@static` | `name` | `label`, `value`, `class`, `style`, `link`, `target`, `isHtml` | Read-only display, supports links and HTML |

### Button Component Parameters

| Component | Required Params | Optional Params | Default Text | Icon |
|-----------|----------------|-----------------|--------------|------|
| `@createButton` | `url` | `name`, `class` | "Create" | Plus |
| `@editButton` | `url` | `name`, `class` | "Edit" | Pencil |
| `@showButton` | `url` | `name`, `class` | "View" | Eye |
| `@saveButton` | - | `name`, `class` | "Save" | Check |
| `@deleteButton` | `url` | `name`, `class` | "Delete" | Trash |
| `@backButton` | `url` | `name`, `class` | "Back" | Arrow Left |
| `@linkButton` | `url` | `name`, `class` | "Link" | Link |

### UI Component Parameters

| Component | Parameters | Description |
|-----------|-----------|-------------|
| Card | `header` (slot), `slot`, `footer` (slot), `class`, `cardClass` | Container component with optional header/footer |
| Breadcrumbs | `breadcrumbs` (array) | Array format: `[['url' => '...', 'label' => '...'], 'Final Item']` |
| Message | None (uses session data) | Displays session flash messages: `message` and `message_type` (success/warning/danger/info) |
| Error | `name` | Displays `$errors->first($name)` |
| Error Icon | `name` | Shows SVG error icon if `$errors->has($name)` |

### Available Icons (18 Total)

| Icon | Usage |
|------|-------|
| `blade::icons.dashboard` | Dashboard/home icon |
| `blade::icons.user` | User/profile icon |
| `blade::icons.page` | Page/document icon |
| `blade::icons.post` | Blog post icon |
| `blade::icons.menu` | Menu/navigation icon |
| `blade::icons.gallery` | Image gallery icon |
| `blade::icons.form` | Form/input icon |
| `blade::icons.image` | Single image icon |
| `blade::icons.component` | Component/module icon |
| `blade::icons.layout` | Layout/template icon |
| `blade::icons.document` | Document/file icon |
| `blade::icons.estimate` | Estimate/quote icon |
| `blade::icons.project` | Project icon |
| `blade::icons.review` | Review/rating icon |
| `blade::icons.design` | Design/creative icon |
| `blade::icons.setting` | Settings/gear icon |
| `blade::icons.lock` | Lock/security icon |
| `blade::icons.redirect` | Redirect/link icon |

## Extension Points

### Adding New Form Components

1. **Create View File**: `src/Views/custom_field.blade.php`
   - Include common partials: `@include('blade::label')`, `@include('blade::error')`
   - Apply error state classes: `{{ $errors->has($name) ? 'border-red-300...' : '' }}`
   - Support `old()` helper: `value="{{ old($name, $value ?? null) }}"`
   - Handle required/disabled states

2. **Register Directive** in `BladeManager`:
   ```php
   protected function registerCustomFieldDirective(): void
   {
       Blade::include('blade::custom_field', 'customField');
   }
   ```

3. **Add to `register()` method**:
   ```php
   public function register(): void
   {
       // Existing registrations...
       $this->registerCustomFieldDirective();
   }
   ```

### Adding New Buttons

1. **Create Button View**: `src/Views/button_action.blade.php`
   - Structure: `<span class="inline-flex..."><a href="{{ $url }}">SVG + Text</a></span>`
   - Use Tailwind button classes: `bg-{color}-600`, `hover:bg-{color}-700`
   - Include SVG icon with `class="-ml-1 mr-2 h-5 w-5"`
   - Add `v-tooltip` attribute for Vue.js tooltip integration
   - Support `$url`, `$name`, `$class` parameters

2. **Register in `BladeManager`**:
   ```php
   protected function registerActionButtonDirective(): void
   {
       Blade::include('blade::button_action', 'actionButton');
   }
   ```

### Adding New Icons

1. **Create Icon View**: `src/Views/icons/custom_icon.blade.php`
   - Use SVG with `viewBox="0 0 24 24"` or `viewBox="0 0 20 20"`
   - Set `stroke="currentColor"` or `fill="currentColor"` for theming
   - Include standard SVG attributes: `fill="none"`, `stroke-linecap="round"`, etc.

2. **Include in Components**: `@include('blade::icons.custom_icon')`

### Extending with Custom Directives

For directives that need logic (not just includes):

```php
// In BladeManager
protected function registerCustomDirective(): void
{
    Blade::directive('customDirective', function ($expression) {
        return "<?php echo customLogic({$expression}); ?>";
    });

    Blade::directive('endcustomDirective', function () {
        return '<?php echo endCustomLogic(); ?>';
    });
}
```

## How to Add New Features

### Adding a Date Picker Component

```blade
<!-- src/Views/datepicker.blade.php -->
<div class="form-group">
    @include('blade::label')

    <div class="mt-1 relative rounded-sm shadow-sm">
        <input
            id="{{ str($id ?? $name)->slug() }}"
            type="text"
            name="{{ $name }}"
            value="{{ old($name, $value ?? null) }}"
            class="form-input datepicker block w-full rounded-sm sm:text-sm sm:leading-5 {{ $errors->has($name) ? 'border-red-300 text-red-900' : '' }}"
            placeholder="{{ $placeholder ?? 'DD/MM/YYYY' }}"
            autocomplete="off"
            {{ isset($required) ? 'required' : '' }}
            {{ isset($disabled) ? 'disabled' : '' }}
        />
        @include('blade::error_icon')
    </div>

    @include('blade::error')
</div>
```

```php
// In BladeManager
protected function registerDatePickerDirective(): void
{
    Blade::include('blade::datepicker', 'datePicker');
}

public function register(): void
{
    // Existing registrations...
    $this->registerDatePickerDirective();
}
```

Usage:
```blade
@datePicker(['label' => 'Start Date', 'name' => 'start_date', 'value' => $project->start_date])
```

### Adding a Toggle Switch Component

```blade
<!-- src/Views/toggle.blade.php -->
<div class="form-group">
    <label class="flex items-center">
        <input type="hidden" name="{{ $name }}" value="0">
        <input
            type="checkbox"
            name="{{ $name }}"
            value="1"
            class="toggle-switch"
            {{ old($name, $value == 1) ? 'checked' : '' }}
        />
        <span class="ml-2 text-sm text-gray-700">{{ $label ?? ucwords(str_replace('_', ' ', $name)) }}</span>
    </label>
    @include('blade::error')
</div>
```

```php
// In BladeManager
protected function registerToggleDirective(): void
{
    Blade::include('blade::toggle', 'toggle');
}
```

## Testing Strategy

### Unit Tests
- Test directive registration in `BladeManager`
- Test service provider bindings

### Feature Tests
- Test component rendering with various parameter combinations
- Test error state rendering
- Test `old()` helper integration
- Test validation error display

### Example Test:
```php
namespace Bongo\Blade\Tests\Feature;

use Bongo\Blade\Tests\TestCase;

class InputComponentTest extends TestCase
{
    public function test_input_renders_with_label()
    {
        $html = $this->blade('@input(["name" => "email", "label" => "Email Address"])');

        $this->assertStringContainsString('Email Address', $html);
        $this->assertStringContainsString('name="email"', $html);
    }

    public function test_input_shows_error_state()
    {
        $this->withViewErrors(['email' => 'Email is required']);
        $html = $this->blade('@input(["name" => "email"])');

        $this->assertStringContainsString('border-red-300', $html);
        $this->assertStringContainsString('Email is required', $html);
    }
}
```

## Design Patterns Used

- **Service Provider Pattern**: Bootstraps package via Laravel's service container
- **Facade Pattern**: `Blade::include()` and `Blade::directive()` provide simplified interface
- **Template Method Pattern**: `AbstractServiceProvider` defines boot/register lifecycle
- **Component Pattern**: Reusable Blade view components with parameter binding
- **Decorator Pattern**: Components wrap standard HTML with validation, styling, and error handling
