# CLAUDE.md - Bongo Blade Package

## Overview

This Laravel package provides reusable Blade components and custom directives for forms, buttons, icons, and UI elements. All components use Tailwind CSS and integrate with Laravel's validation system.

**For detailed information, see**:
- `.cursorrules` - Cursor AI development guidelines
- `ARCHITECTURE.md` - Detailed architecture and component reference
- `.github/copilot-instructions.md` - GitHub Copilot code patterns
- `README.md` - User-facing package documentation

## Commands

From the package root directory:

```bash
# Testing
vendor/bin/phpunit                    # Run test suite

# Code Quality
vendor/bin/pint                       # Fix code style issues
vendor/bin/pint --test                # Check code style without fixing
vendor/bin/phpstan analyse            # Run static analysis

# Dependencies
composer install                      # Install dependencies
composer update -W                    # Update dependencies with dependencies
rm -f composer.lock && composer update -W  # Fresh dependency update
```

## Quick Architecture Reference

### Package Structure

```
blade/
├── src/
│   ├── BladeServiceProvider.php       # Extends AbstractServiceProvider ($module = 'blade')
│   ├── Services/
│   │   └── BladeManager.php           # Registers all Blade directives
│   └── Views/                         # Component templates (blade:: namespace)
│       ├── Form components: input, select, multiselect, checkbox, textarea, wysiwyg, label, static
│       ├── Buttons: button_create, button_edit, button_show, button_save, button_delete, button_back, button_link
│       ├── UI: card, breadcrumbs, message, error, error_icon
│       └── icons/ (18 SVG icons: dashboard, user, page, post, menu, gallery, etc.)
└── tests/
    ├── TestCase.php                   # Orchestra Testbench setup
    ├── Unit/.gitkeep
    └── Feature/.gitkeep
```

### Key Classes

**BladeServiceProvider** (`src/BladeServiceProvider.php`)
- Extends: `Bongo\Framework\Providers\AbstractServiceProvider`
- Property: `protected string $module = 'blade'`
- Binds: `BladeManager` as `blade_manager` singleton
- Boot: Calls `BladeManager::register()` to set up directives

**BladeManager** (`src/Services/BladeManager.php`)
- Method: `register()` - Orchestrates all directive registrations
- Pattern: Uses `Blade::include()` to map directives to views
- Example: `Blade::include('blade::input', 'input')` creates `@input` directive

### Directive Registration Pattern

```php
// In BladeManager
protected function registerInputDirective(): void
{
    Blade::include('blade::input', 'input');
    // Creates @input directive that renders src/Views/input.blade.php
}
```

### Component Usage Examples

```blade
{{-- Form Components --}}
@input(['label' => 'Email', 'name' => 'email', 'type' => 'email', 'required' => true])
@select(['label' => 'Role', 'name' => 'role_id', 'options' => $roles, 'value' => $user->role_id])
@checkbox(['label' => 'Active', 'name' => 'is_active', 'value' => $user->is_active])
@textarea(['label' => 'Bio', 'name' => 'bio', 'value' => $user->bio, 'rows' => 5])

{{-- Buttons --}}
@createButton(['name' => 'New User', 'url' => route('users.create')])
@editButton(['url' => route('users.edit', $user)])
@saveButton(['name' => 'Save Changes'])
@deleteButton(['url' => route('users.destroy', $user)])

{{-- UI Components --}}
@include('blade::message')  {{-- Flash messages --}}
@include('blade::breadcrumbs', ['breadcrumbs' => [...]])
<x-blade::card header="Title">Content</x-blade::card>
```

## Key Files

| File | Purpose | Lines |
|------|---------|-------|
| `src/BladeServiceProvider.php` | Service provider, registers BladeManager | ~30 |
| `src/Services/BladeManager.php` | Central directive registration | ~120 |
| `src/Views/input.blade.php` | Text input component with validation | ~26 |
| `src/Views/select.blade.php` | Dropdown select component | ~44 |
| `src/Views/checkbox.blade.php` | Checkbox with hidden fallback | ~25 |
| `src/Views/button_create.blade.php` | Create action button | ~16 |
| `src/Views/message.blade.php` | Flash message display | ~217 |
| `src/Views/card.blade.php` | Card container component | ~16 |
| `tests/TestCase.php` | Orchestra Testbench base class | ~27 |

## Code Style

### Conventions
- **PHP**: 8.2+ features, type hints, visibility modifiers
- **Style**: Laravel Pint (see `.php_cs` for rules)
- **Naming**: camelCase directives, snake_case view files, PascalCase classes
- **Blade**: `{{ }}` for escaped output, `{!! !!}` only when HTML needed

### Component Parameter Patterns
All components accept parameter arrays with common keys:
- `name` (required for form fields): Field name
- `label` (optional): Display label (auto-generated from `name` if omitted)
- `value` (optional): Field value (integrates with `old()` helper)
- `class` (optional): Additional CSS classes
- `required`, `disabled` (optional): Field states
- `id` (optional): Custom field ID (defaults to slugified `name`)

### Validation Integration
- All form components check `$errors->has($name)` for error states
- Error styling applied automatically: `border-red-300`, `text-red-900`
- Include `@include('blade::error')` to display error messages
- Include `@include('blade::error_icon')` for error state icons

### Auto-Generated Labels
Labels are auto-generated from field names if not provided:
```php
// Converts 'user_first_name' → "User First Name"
{{ $label ?? ucwords(str_replace(']', '', str_replace('[', '', str_replace('_', ' ', $name)))) }}
```

## Dependencies

- **Runtime**: `bongo/framework` ^3.0, Laravel 10+, PHP 8.2+
- **Dev**: `orchestra/testbench` ^8.0, `laravel/pint` ^1.0, `nunomaduro/larastan` ^2.0.1

## Framework Integration

This package extends `Bongo\Framework\Providers\AbstractServiceProvider`:
- Views auto-registered from `src/Views/` under `blade::` namespace
- No routes, migrations, config, or middleware needed
- Service provider `$module` property set to `'blade'`

## Common Tasks

### Add New Form Component
1. Create `src/Views/component_name.blade.php` with label, error, and field
2. Register in `BladeManager::register()`:
   ```php
   protected function registerComponentNameDirective(): void
   {
       Blade::include('blade::component_name', 'componentName');
   }
   ```
3. Call in `register()` method

### Add New Button
1. Create `src/Views/button_type.blade.php` with SVG icon and Tailwind styles
2. Register in `BladeManager`:
   ```php
   protected function registerTypeButtonDirective(): void
   {
       Blade::include('blade::button_type', 'typeButton');
   }
   ```

### Add New Icon
1. Create `src/Views/icons/icon_name.blade.php` with SVG (use `currentColor`)
2. Include in views: `@include('blade::icons.icon_name')`

## Testing

Tests use Orchestra Testbench for Laravel package testing:
```php
use Bongo\Blade\Tests\TestCase;

class ComponentTest extends TestCase
{
    public function test_input_renders()
    {
        $html = $this->blade('@input(["name" => "email"])');
        $this->assertStringContainsString('name="email"', $html);
    }
}
```

## Important Notes

- No configuration file needed
- No routes or migrations
- Pure view component package
- All components use Tailwind CSS
- Validation errors handled via `$errors` view variable
- Form repopulation via `old()` helper
- Flash messages use `session('message')` and `session('message_type')`
- Available icon set: 18 SVG icons in `src/Views/icons/`
