# Bongo Blade Package

Provides custom Blade components and directives for building consistent, accessible forms and UI elements in Laravel applications. All components integrate seamlessly with Laravel's validation system and use Tailwind CSS for styling.

## Features

- **Form Components**: Input, select, multiselect, checkbox, textarea, WYSIWYG editor
- **Button Components**: Create, edit, show, save, delete, back, and link buttons
- **UI Components**: Cards, breadcrumbs, flash messages, and validation error displays
- **Icon Library**: 18 SVG icons for common CMS operations
- **Validation Integration**: Automatic error state styling and error message display
- **Accessibility**: Proper labels, ARIA attributes, and semantic HTML
- **Form Repopulation**: Automatic integration with Laravel's `old()` helper

## Requirements

- PHP 8.2+
- Laravel 10+
- Composer
- Tailwind CSS (for styling)

## Installation

### Via Composer

```bash
composer require bongo/blade
```

### Laravel Configuration

**Laravel 11+**: Service provider auto-discovery handles registration automatically.

**Laravel 10**: If needed, add to `config/app.php`:
```php
'providers' => [
    // Other providers...
    Bongo\Blade\BladeServiceProvider::class,
],
```

## Usage

### Form Components

#### Input Field

```blade
@input([
    'label' => 'Email Address',
    'name' => 'email',
    'type' => 'email',
    'value' => $user->email,
    'required' => true,
    'placeholder' => 'user@example.com'
])
```

**Parameters**:
- `name` (required): Field name
- `label` (optional): Display label (auto-generated from name if omitted)
- `type` (optional): Input type (text, email, number, password, etc.) - defaults to 'text'
- `value` (optional): Field value
- `placeholder` (optional): Placeholder text
- `required` (optional): Marks field as required
- `disabled` (optional): Disables the field
- `maxlength` (optional): Maximum length
- `step`, `min` (optional): For number inputs
- `class` (optional): Additional CSS classes
- `id` (optional): Custom field ID (defaults to slugified name)

#### Select Dropdown

```blade
@select([
    'label' => 'Department',
    'name' => 'department_id',
    'value' => $user->department_id,
    'options' => $departments->pluck('name', 'id')->toArray(),
    'placeholder' => 'Select a department',
    'required' => true
])
```

**Parameters**:
- `name` (required): Field name
- `options` (required): Associative array of key => value pairs
- `label` (optional): Display label
- `value` (optional): Selected value
- `placeholder` (optional): Placeholder option
- `default` (optional): Default option as `[key => value]`
- `multiple` (optional): Enable multiple selection
- `required` (optional): Marks field as required
- `class` (optional): Additional CSS classes

#### Multi-Select

```blade
@multiSelect([
    'label' => 'Tags',
    'name' => 'tags[]',
    'value' => $post->tags->pluck('id')->toArray(),
    'options' => $tags->pluck('name', 'id')->toArray(),
    'size' => 5
])
```

**Parameters**:
- Same as `@select` but requires `value` to be an array
- `size` (optional): Visible options (defaults to 3)

#### Checkbox

```blade
@checkbox([
    'label' => 'Active User',
    'name' => 'is_active',
    'value' => $user->is_active
])
```

**Parameters**:
- `name` (required): Field name
- `label` (optional): Display label
- `value` (optional): Checked state (1/true = checked)

**Note**: Includes hidden input with value="0" to ensure unchecked state is submitted.

#### Textarea

```blade
@textarea([
    'label' => 'Biography',
    'name' => 'bio',
    'value' => $user->bio,
    'rows' => 5,
    'placeholder' => 'Tell us about yourself...'
])
```

**Parameters**:
- `name` (required): Field name
- `label` (optional): Display label
- `value` (optional): Field value
- `rows` (optional): Number of visible rows
- `placeholder` (optional): Placeholder text
- `required` (optional): Marks field as required
- `disabled` (optional): Disables the field

#### WYSIWYG Editor

```blade
@wysiwyg([
    'name' => 'content',
    'value' => $post->content,
    'height' => 400,
    'placeholder' => 'Enter content...'
])
```

**Parameters**:
- `name` (required): Field name
- `value` (optional): Editor content
- `height` (optional): Editor height in pixels (defaults to 350)
- `rows` (optional): Textarea rows (defaults to 20)
- `placeholder` (optional): Placeholder text
- `resizable` (optional): Enable/disable resizing (defaults to true)
- `required`, `disabled` (optional): Field states

**Note**: Image upload URL is automatically set to `/admin/images`.

#### Static Display (Read-Only)

```blade
@static([
    'label' => 'Created At',
    'name' => 'created_at',
    'value' => $user->created_at->format('d/m/Y H:i')
])

{{-- With link --}}
@static([
    'label' => 'Website',
    'name' => 'website',
    'value' => $user->website,
    'link' => $user->website,
    'target' => '_blank'
])
```

**Parameters**:
- `name` (required): Field identifier
- `label` (optional): Display label
- `value` (optional): Display value (shows "-- --" if empty)
- `link` (optional): Makes value clickable
- `target` (optional): Link target (defaults to 'self')
- `isHtml` (optional): Render value as HTML
- `class`, `style` (optional): Additional styling

**Note**: Automatically formats Carbon date instances.

#### Label

```blade
@label([
    'name' => 'first_name',
    'label' => 'First Name',
    'required' => true
])
```

**Parameters**:
- `name` (required): Field name (used for `for` attribute)
- `label` (optional): Label text (auto-generated if omitted)
- `id` (optional): Custom field ID
- `required` (optional): Adds red asterisk
- `label_class` (optional): Additional CSS classes
- `hideLabel` (optional): Hide label but maintain structure

### Button Components

#### Create Button

```blade
@createButton(['name' => 'Create User', 'url' => route('users.create')])
```

#### Edit Button

```blade
@editButton(['name' => 'Edit User', 'url' => route('users.edit', $user)])
```

#### Show/View Button

```blade
@showButton(['name' => 'View Details', 'url' => route('users.show', $user)])
```

#### Save Button

```blade
@saveButton(['name' => 'Save Changes'])
```

#### Delete Button

```blade
@deleteButton(['name' => 'Delete User', 'url' => route('users.destroy', $user)])
```

#### Back Button

```blade
@backButton(['name' => 'Back to List', 'url' => route('users.index')])
```

#### Link Button

```blade
@linkButton(['name' => 'External Link', 'url' => 'https://example.com'])
```

**Common Button Parameters**:
- `url` (required for most): Target URL
- `name` (optional): Button text (defaults vary by button type)
- `class` (optional): Additional CSS classes

### UI Components

#### Card Container

```blade
<x-blade::card>
    <x-slot name="header">
        <h3>User Profile</h3>
    </x-slot>

    <p>Card content goes here</p>

    <x-slot name="footer">
        Last updated: {{ $user->updated_at }}
    </x-slot>
</x-blade::card>
```

**Parameters**:
- `header` (slot, optional): Card header content
- `slot` (required): Main card content
- `footer` (slot, optional): Card footer content
- `class` (optional): Additional CSS for content area
- `cardClass` (optional): Additional CSS for card wrapper

#### Breadcrumbs

```blade
@include('blade::breadcrumbs', ['breadcrumbs' => [
    ['url' => route('home'), 'label' => 'Home'],
    ['url' => route('users.index'), 'label' => 'Users'],
    'Edit User'  // Final item (no link)
]])
```

**Format**: Array of arrays with `url` and `label`, or plain strings for final item.

#### Flash Messages

```blade
{{-- In your layout --}}
@include('blade::message')
```

Displays session flash messages. Set in controller:
```php
return redirect()->route('users.index')
    ->with('message', 'User created successfully')
    ->with('message_type', 'success');  // success, warning, danger, info
```

**Message Types**:
- `success`: Green notification
- `warning`: Yellow notification
- `danger`: Red notification
- `info`: Blue notification

Also displays:
- Validation errors (automatic)
- No JavaScript warning
- Legacy browser (IE) warning

### Form Group Directives

```blade
@formgroup
    {{-- Form content --}}
@endformgroup
```

Creates a `<div class="form-group">` wrapper.

### Icon Components

18 available icons in `src/Views/icons/`:

```blade
@include('blade::icons.dashboard')
@include('blade::icons.user')
@include('blade::icons.page')
@include('blade::icons.post')
@include('blade::icons.menu')
@include('blade::icons.gallery')
@include('blade::icons.form')
@include('blade::icons.image')
@include('blade::icons.component')
@include('blade::icons.layout')
@include('blade::icons.document')
@include('blade::icons.estimate')
@include('blade::icons.project')
@include('blade::icons.review')
@include('blade::icons.design')
@include('blade::icons.setting')
@include('blade::icons.lock')
@include('blade::icons.redirect')
```

All icons use `currentColor` for easy theming.

## Validation Integration

All form components automatically:
- Display validation errors via `$errors->has($name)`
- Apply error state styling (red borders, text colours)
- Show error messages via `@include('blade::error')`
- Display error icons via `@include('blade::error_icon')`
- Repopulate values using `old($name, $value)`

Example controller validation:
```php
$request->validate([
    'email' => 'required|email',
    'name' => 'required|string|max:255',
]);
```

Errors display automatically when using components.

## Styling

All components use **Tailwind CSS** utility classes:
- Form inputs: `form-input`, `form-checkbox`
- Error states: `border-red-300`, `text-red-900`
- Buttons: `bg-gray-600`, `hover:bg-gray-700`, `text-white`
- Cards: `bg-white`, `shadow`, `rounded-sm`

Ensure Tailwind CSS is properly configured in your Laravel application.

## Testing

```bash
# Run tests
vendor/bin/phpunit

# Check code style
vendor/bin/pint --test

# Fix code style
vendor/bin/pint

# Static analysis
vendor/bin/phpstan analyse
```

## Development

### Adding Custom Components

1. Create view file in `src/Views/component_name.blade.php`
2. Register in `src/Services/BladeManager.php`:
   ```php
   protected function registerComponentNameDirective(): void
   {
       Blade::include('blade::component_name', 'componentName');
   }
   ```
3. Add registration call in `register()` method

See `ARCHITECTURE.md` for detailed development guidelines.

## Package Structure

```
blade/
├── src/
│   ├── BladeServiceProvider.php       # Service provider
│   ├── Services/
│   │   └── BladeManager.php           # Directive registration
│   └── Views/                         # Blade components
│       ├── Form components (input, select, checkbox, etc.)
│       ├── Button components (button_*.blade.php)
│       ├── UI components (card, breadcrumbs, message, etc.)
│       └── icons/ (18 SVG icons)
├── tests/                             # PHPUnit tests
├── ARCHITECTURE.md                    # Detailed architecture docs
├── CLAUDE.md                          # AI development guide
└── README.md                          # This file
```

## License

MIT Licence - See LICENCE file for details

## Credits

- **Author**: Stuart Elliott <stuart.elliott@bespokeuk.com>
- **Homepage**: https://bespokeuk.com
- **Repository**: https://bitbucket.org/designtec/blade

## Support

For issues, questions, or contributions, please contact the package maintainer.
