# GitHub Copilot Instructions - Builder Package

## Project Overview

**bongo/builder** is a Laravel visual page builder providing 764+ pre-designed HTML/CSS blocks organized into components, designs, and layouts. Users can manage blocks through an admin interface and integrate them into pages via API or shortcodes.

**Architecture**: File-based storage with vendor/resource override pattern, single polymorphic model, facade-based access.

**Tech Stack**: Laravel 10+, PHP 8.2+, ContentBuilder JS, Laravel Mix, SASS

## Key Classes and Relationships

### BuilderItem (Model)

**Location**: `src/Models/BuilderItem.php`

The core model handling all three types (component, design, layout):

```php
class BuilderItem
{
    public string $type;              // 'component', 'design', or 'layout'
    public ?string $name;             // Slugified name (e.g., 'header-02')
    public ?string $key;              // Unique key for lookup
    public ?string $category;         // Slugified category
    public ?string $html;             // Rendered Blade template
    public ?string $thumbnail;        // Base64-encoded PNG

    // Path properties for vendor and resource locations
    public ?string $relativePath;
    public ?string $absolutePath;
    // ... (more path properties)

    public function all(): Collection              // Get all items (cached 10 min)
    public function allByCategory(): array         // Items grouped by category
    public function allAsJson(): false|string      // JSON for API
    public function find(string $key): mixed       // Find by key
    public function findByCategory(string $category): Collection
    public function save(): void                   // Save to resource path
    public function delete(): void                 // Remove from resource path
    public function createFromFile(SplFileInfo $file): BuilderItem
}
```

**Storage Pattern**: Files at `{category}/{name}/index.blade.php` + `preview.png`

**Dual Locations**:
- Vendor: `vendor/bongo/builder/src/Views/frontend/{type}/`
- Resource: `resources/views/vendor/builder/{type}/`

### Facades

**Location**: `src/Facades/`

Three facades for convenient access:

```php
Component::all()
Component::find('header-02')
Component::findByCategory('headline')
Component::categories()

Design::all()
Design::findByCategory('banners')

Layout::all()
Layout::findByCategory('headers')
```

Registered as global aliases: `Component`, `Design`, `Layout`

### Controllers

#### Backend Controllers (Admin CRUD)

**Location**: `src/Http/Controllers/Backend/`

- `ComponentController` - Manage components
- `DesignController` - Manage designs
- `LayoutController` - Manage layouts
- `IconController` - View FontAwesome icons

**Standard CRUD Pattern**:
```php
index()    // List all items by category
create()   // Show create form
store()    // Save new item
show($key) // Display single item
edit($key) // Show edit form
update($key) // Update item
destroy($key) // Delete item
```

#### API Controllers (Frontend Integration)

**Location**: `src/Http/Controllers/Api/`

21 controllers providing JSON/HTML endpoints:

**Block Controllers**:
- `ComponentController::index()` - All components as JSON
- `DesignController::index()` - All designs as JSON
- `SnippetController::index()` - Code snippets
- `ModuleController::index($module)` - Dynamic modules

**Asset Controllers**:
- `IconController::index()` - FontAwesome library
- `ImageController::index()` - Image library
- `FileController::index()` - File manager
- `FontController::index()` - Web fonts
- `ButtonController::index()` - Button styles
- `StyleController::index()` - CSS styles
- `SymbolController::index()` - Special symbols

**Dynamic Content Controllers**:
- `PostController` - Blog post listings
- `ReviewController` - Review listings
- `GalleryController` - Gallery builder
- `FormController` - Form builder
- `CarouselController` - Carousel content
- `SliderController` - Slider content
- `MenuController` - Menu builder

### Middleware

**HasShortCodes** (`src/Http/Middleware/HasShortCodes.php`):

Processes shortcodes in responses:

```php
public function handle($request, Closure $next)
{
    $response = $next($request);
    $content = ShortCode::parse($response->content());
    $response->setContent($content);
    return $response;
}
```

Allows: `[component key="header-02"]` in templates

## Code Style Templates

### Creating a Controller

```php
namespace Bongo\Builder\Http\Controllers\Api;

use Bongo\Framework\Http\Controllers\Api\AbstractApiController;

class NewController extends AbstractApiController
{
    public function index()
    {
        // Return view or JSON
        return view('builder::api.template');
    }
}
```

### Adding a Route

**API Routes** (`src/Routes/api.php`):
```php
Route::prefix('builder')
    ->as('builder.')
    ->group(function () {
        Route::get('endpoint', [NewController::class, 'index'])
            ->name('endpoint.index');
    });
```

**Backend Routes** (`src/Routes/backend.php`):
```php
Route::as('resource.')
    ->prefix(config('builder.resource_prefix'))
    ->group(function () {
        Route::get('/', [ResourceController::class, 'index'])->name('index');
        Route::get('create', [ResourceController::class, 'create'])->name('create');
        Route::post('store', [ResourceController::class, 'store'])->name('store');
        Route::prefix('{key}')->group(function () {
            Route::get('/', [ResourceController::class, 'show'])->name('show');
            Route::get('edit', [ResourceController::class, 'edit'])->name('edit');
            Route::post('update', [ResourceController::class, 'update'])->name('update');
            Route::delete('delete', [ResourceController::class, 'destroy'])->name('destroy');
        });
    });
```

### Working with BuilderItem

```php
// Get all items
$items = Component::all();

// Find specific item
$item = Component::find('header-02');
if ($item) {
    echo $item->getName();
    echo $item->getCategory();
    echo $item->getHtml();
    echo $item->getThumbnail();
}

// Filter by category
$headlines = Component::findByCategory('headline');

// Create new item
$component = new BuilderItem('component');
$component->setName('my-block');
$component->setKey('my-block');
$component->setCategory('headline');
$component->setHtml('<div class="block">Content</div>');
$component->setThumbnail(file_get_contents($imagePath));
$component->save();

// Delete item (removes from resource path only)
$component->delete();
```

### Exception Handling

```php
use Bongo\Builder\Exceptions\ComponentException;

try {
    $item = Component::findOrFail($key);
} catch (Exception $e) {
    log_exception($e);
    throw new ComponentException("Component not found: {$key}");
}
```

### View Templates

**Backend Views** (use Blade):
```blade
@extends('backend::layouts.app')

@section('content')
    <div class="container">
        @foreach($items as $category => $categoryItems)
            <h2>{{ ucwords(str_replace('-', ' ', $category)) }}</h2>
            @foreach($categoryItems as $item)
                <div class="block">
                    <img src="{{ $item->getThumbnail() }}" alt="{{ $item->getName() }}">
                    <h3>{{ $item->getName() }}</h3>
                </div>
            @endforeach
        @endforeach
    </div>
@endsection
```

**API Views** (return JSON):
```blade
@php
    echo Component::allAsJson();
@endphp
```

## Common Patterns

### Retrieving Items

```php
// All items
Component::all()                           // Returns Collection

// By category
Component::findByCategory('headline')      // Returns Collection

// Single item
Component::find('header-02')               // Returns BuilderItem|null
Component::findOrFail('header-02')         // Returns BuilderItem or 404

// As JSON
Component::allAsJson()                     // Returns JSON string

// Categories
Component::categories()                    // Returns ['slug' => 'Name']
Component::categoriesAsJson()              // Returns JSON string
```

### CRUD Operations

```php
// Create
$item = new BuilderItem('component');
$item->setName('new-block');
$item->setCategory('headline');
$item->setHtml($html);
$item->setThumbnail($imageData);
$item->save();

// Update
$item = Component::find('header-02');
$item->setHtml($newHtml);
$item->save();

// Delete
$item = Component::find('header-02');
$item->delete();  // Only removes from resource path
```

### File Path Operations

```php
$item->getBasePath()           // Vendor path (original)
$item->getResourcePath()       // Resource path (customizations)
$item->getAllFiles()           // Merged (resource overrides vendor)
$item->getVendorFiles()        // Only vendor files
$item->getResourceFiles()      // Only resource files
```

### Rendering Templates

```php
$item->getHtml()               // Rendered Blade template
$item->getRawHtml()            // Raw file contents
$item->getThumbnail()          // Base64-encoded image
$item->getRawThumbnail()       // Raw image data
```

## Frontend Asset Development

### Build Commands

```bash
npm install                    # Install dependencies
npm run dev                    # Development build
npm run watch                  # Watch for changes
npm run production             # Production build (minified)
```

### Asset Structure

**Backend Editor**:
- Source: `resources/backend/sass/editor.scss`, `resources/backend/js/editor.js`
- Compiled: `public/backend/editor.css`, `public/backend/editor.js`
- Includes ContentBuilder JS library

**Frontend**:
- Source: `resources/frontend/sass/box.scss`, `resources/frontend/js/box.js`
- Compiled: `public/frontend/box.css`, `public/frontend/box.js`
- Includes vendor libraries (jQuery Appear, Skrollr, AOS)

## Configuration Reference

**Key Config Options** (`config/builder.php`):

```php
'component_prefix' => 'components'       // /admin/components
'design_prefix' => 'sections'            // /admin/sections
'layout_prefix' => 'layouts'             // /admin/layouts

'vendor_path' => 'vendor/bongo/builder/src/Views/'
'resource_path' => 'resources/views/vendor/builder/'

'component_categories' => [...]          // Component categories
'design_categories' => [...]             // Design categories
'layout_categories' => [...]             // Layout categories

'regular_icons' => [...]                 // 160 FontAwesome regular icons
'solid_icons' => [...]                   // 1,400 FontAwesome solid icons
'brand_icons' => [...]                   // 490 FontAwesome brand icons
```

## Testing

```bash
vendor/bin/phpunit                       # Run tests
vendor/bin/pint                          # Fix code style
vendor/bin/pint --test                   # Check code style
```

## Important Notes

- All names and categories are auto-slugified (lowercase, hyphenated)
- Preview images are base64-encoded for display
- Delete operations only remove from resource path (vendor remains)
- Cache is cleared automatically on save/delete
- Blocks must have both `index.blade.php` and `preview.png`
- Use `DIRECTORY_SEPARATOR` constant for cross-platform compatibility
