<?php

namespace Bongo\Builder\Http\Controllers\Backend;

use Bongo\Builder\Facades\DesignFacade;
use Bongo\Builder\Models\BuilderItem;
use Bongo\Framework\Http\Controllers\AbstractController;
use Exception;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\File;
use Illuminate\View\View;

class DesignController extends AbstractController
{
    public function index(Request $request): View
    {
        if (config('settings.memory_limit') && ! empty(config('settings.memory_limit'))) {
            ini_set('memory_limit', config('settings.memory_limit'));
        }

        $categories = Arr::except(DesignFacade::categories(), ['buttons', 'custom']);

        $category = $request->get('category', 'achievements');
        $designs = DesignFacade::findByCategory($category);

        return view('builder::backend.design.index', compact('categories', 'designs'));
    }

    public function create(): View
    {
        $design = new BuilderItem('design');

        return view('builder::backend.design.create', compact('design'));
    }

    public function show(string $key): View
    {
        $design = DesignFacade::findOrFail($key);

        return view('builder::backend.design.show', compact('design'));
    }

    /**
     * @throws Exception
     */
    public function store(Request $request): RedirectResponse
    {
        // Create a new design instance
        $design = new BuilderItem('design');
        $thumbnail = $this->getFileContents($request) ?? $design->getRawThumbnail();

        // Save the changes
        $design->setName($request->get('name'));
        $design->setKey($request->get('name'));
        $design->setCategory($request->get('category'));
        $design->setThumbnail($thumbnail);
        $design->setHtml($request->get('html'));
        $design->save();

        return redirect()
            ->route('backend.design.show', $design->key)
            ->success(trans('builder::backend.design_store_success'));
    }

    public function edit(string $key): View
    {
        $design = DesignFacade::findOrFail($key);

        return view('builder::backend.design.edit', compact('design'));
    }

    public function update(Request $request, string $key): RedirectResponse
    {
        // Find the existing design
        $design = DesignFacade::find($key);
        $thumbnail = $this->getFileContents($request) ?? $design->getRawThumbnail();

        // Delete the old one
        $design->delete();

        // Save the changes
        $design->setName($request->get('name'));
        $design->setKey($request->get('name'));
        $design->setCategory($request->get('category'));
        $design->setThumbnail($thumbnail);
        $design->setHtml($request->get('html'));
        $design->save();

        return redirect()
            ->route('backend.design.show', $design->key)
            ->success(trans('builder::backend.design_update_success'));
    }

    public function destroy(string $key): RedirectResponse
    {
        $design = DesignFacade::find($key);
        $design->delete();

        return redirect()
            ->route('backend.design.index')
            ->success(trans('builder::backend.design_delete_success'));
    }

    private function getFileContents(Request $request): ?string
    {
        if ($file = $request->file('thumbnail')) {
            return File::get($file->getRealPath());
        }

        return null;
    }
}
