<?php

namespace Bongo\Builder\Http\Controllers\Backend;

use Bongo\Builder\Facades\ComponentFacade;
use Bongo\Builder\Models\BuilderItem;
use Bongo\Framework\Http\Controllers\AbstractController;
use Exception;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Illuminate\View\View;

class ComponentController extends AbstractController
{
    public function index(Request $request): View
    {
        if (config('settings.memory_limit') && ! empty(config('settings.memory_limit'))) {
            ini_set('memory_limit', config('settings.memory_limit'));
        }

        $categories = ComponentFacade::categories();

        $category = $request->get('category', 'achievements');
        $components = ComponentFacade::findByCategory($category);

        return view('builder::backend.component.index', compact('categories', 'components'));
    }

    public function create(): View
    {
        $component = new BuilderItem('component');

        return view('builder::backend.component.create', compact('component'));
    }

    public function show(string $key): View
    {
        $component = ComponentFacade::findOrFail($key);

        return view('builder::backend.component.show', compact('component'));
    }

    /**
     * @throws Exception
     */
    public function store(Request $request): RedirectResponse
    {
        // Create a new component instance
        $component = new BuilderItem('component');
        $thumbnail = $this->getFileContents($request) ?? $component->getRawThumbnail();

        // Save the changes
        $component->setName($request->get('name'));
        $component->setKey($request->get('name'));
        $component->setCategory($request->get('category'));
        $component->setThumbnail($thumbnail);
        $component->setHtml($request->get('html'));
        $component->save();

        return redirect()
            ->route('backend.component.show', $component->key)
            ->success(trans('builder::backend.component_store_success'));
    }

    public function edit(string $key): View
    {
        $component = ComponentFacade::findOrFail($key);

        return view('builder::backend.component.edit', compact('component'));
    }

    public function update(Request $request, string $key): RedirectResponse
    {
        // Find the existing component
        $component = ComponentFacade::find($key);
        $thumbnail = $this->getFileContents($request) ?? $component->getRawThumbnail();

        // Delete the old one
        $component->delete();

        // Save the changes
        $component->setName($request->get('name'));
        $component->setKey($request->get('name'));
        $component->setCategory($request->get('category'));
        $component->setThumbnail($thumbnail);
        $component->setHtml($request->get('html'));
        $component->save();

        return redirect()
            ->route('backend.component.show', $component->key)
            ->success(trans('builder::backend.component_update_success'));
    }

    public function destroy(string $key): RedirectResponse
    {
        $component = ComponentFacade::find($key);
        $component->delete();

        return redirect()
            ->route('backend.component.index')
            ->success(trans('builder::backend.component_delete_success'));
    }

    /**
     * @throws FileNotFoundException
     */
    private function getFileContents(Request $request): ?string
    {
        if ($file = $request->file('thumbnail')) {
            return File::get($file->getRealPath());
        }

        return null;
    }
}
