/**
 * Restore the saved settings
 *
 * This page is opened when you click settings icon on the snippet. In this page, we will modify custom variables.
 * Custom variables are data that be used for any purpose. For example, on a slider snippet, custom variables
 * are used to store the url of the images and also other slider settings that user can change.
 * Ini this example, we have a checkbox and an input text as example on how to interact with custom variables.
 * That's why custom variables are stored (in snippet) in a data attribute named data-settings,
 * because custom variables can control the snippet content.
 * Get custom variables by reading data-settings attribute defined on snippet => see minimalist-blocks/content.js
 */
function restoreSavedSettings() {
    let settings = getSettings();

    if (settings !== 'undefined') {

        let json = JSON.parse(settings);
        document.querySelector("#ref").value = json.ref;

        if (document.querySelector("#columns")) {
            document.querySelector("#columns").value = json.columns || 4;
        }
        if (document.querySelector("#padding")) {
            document.querySelector("#padding").value = json.padding || 0;
        }
        if (document.querySelector("#preset")) {
            document.querySelector("#preset").value = json.preset || 'medium';
        }
        if (document.querySelector("#quality")) {
            document.querySelector("#quality").value = json.quality || 80;
        }
        if (document.querySelector("#lazy")) {
            document.querySelector("#lazy").checked = json.lazy_load;
        }
        if (document.querySelector("#lightbox")) {
            document.querySelector("#lightbox").checked = json.lightbox;
        }
        if (document.querySelector("#class-list")) {
            document.querySelector("#class-list").value = json.class_list;
        }
    }
}

/**
 * Register events
 *
 * When user make any change to the value, call updateSnippetHtml() function
 * updateSnippetHtml() function generates HTML content to be displayed on the snippet.
 */
function registerFormEvents() {

    // Gallery dropdown
    document.querySelector('#ref')
        .addEventListener('change', function (e) {
            updateSnippetHtml();
        });

    // Columns
    if (document.querySelector("#columns")) {
        document.querySelector('#columns')
            .addEventListener('change', function (e) {
                updateSnippetHtml();
            });
    }

    // Padding
    if (document.querySelector("#padding")) {
        document.querySelector('#padding')
            .addEventListener('change', function (e) {
                updateSnippetHtml();
            });
    }

    // Preset
    if (document.querySelector("#preset")) {
        document.querySelector('#preset')
            .addEventListener('change', function (e) {
                updateSnippetHtml();
            });
    }

    // Quality
    if (document.querySelector("#quality")) {
        document.querySelector('#quality')
            .addEventListener('change', function (e) {
                updateSnippetHtml();
            });
    }

    // lazy load
    if (document.querySelector("#lazy")) {
        document.querySelector('#lazy')
            .addEventListener('change', function (e) {
                updateSnippetHtml();
            });
    }

    // lightbox
    if (document.querySelector("#lightbox")) {
        document.querySelector('#lightbox')
            .addEventListener('change', function (e) {
                updateSnippetHtml();
            });
    }

    // Custom classes
    if (document.querySelector("#class-list")) {
        document.querySelector('#class-list')
            .addEventListener('change', function (e) {
                updateSnippetHtml();
            });
    }
}

/**
 * Generate html content.
 *
 * Here you can use the custom variables.
 * Also you can embed custom javascript.
 * Here you can also create a unique ID {id} so that multiple snippets won;t be a problem.
 */
function updateSnippetHtml() {

    let id = makeId();
    let html = '';

    let ref = document.querySelector("#ref").value;

    let columns = document.querySelector("#columns")
        ? document.querySelector("#columns").value
        : null;

    let padding = document.querySelector("#padding")
        ? document.querySelector("#padding").value
        : null;

    let preset = document.querySelector("#preset")
        ? document.querySelector("#preset").value
        : null;

    let quality = document.querySelector("#quality")
        ? document.querySelector("#quality").value
        : null;

    let lazy = document.querySelector("#lazy")
        ? document.querySelector("#lazy").checked
        : null;

    let lightbox = document.querySelector("#lightbox")
        ? document.querySelector("#lightbox").checked
        : null;

    let classList = document.querySelector("#class-list")
        ? document.querySelector("#class-list").value
        : null;

    let endpoint = `/api/builder/gallery-builder?ref=${ref}`;

    if (columns) {
        endpoint += `&columns=${columns}`;
    }
    if (padding) {
        endpoint += `&padding=${padding}`;
    }
    if (preset) {
        endpoint += `&preset=${preset}`;
    }
    if (quality) {
        endpoint += `&quality=${quality}`;
    }
    if (lazy) {
        endpoint += `&lazy=${lazy}`;
    }
    if (lightbox) {
        endpoint += `&lightbox=${lightbox}`;
    }

    // The ajax container
    html += `<div id="${id}" class="gallery-builder ${classList}"></div>`;

    // Append the javascript
    html += '<scr' + 'ipt>';
    html += getDocumentReadyScript();
    html += `
        docReady(function () {

            var xmlhttp = new XMLHttpRequest();
            xmlhttp.onreadystatechange = function() {
                if (xmlhttp.readyState === XMLHttpRequest.DONE) {
                
                    if (xmlhttp.status === 200) {
                        let galleryContainer = document.getElementById("${id}");
                        galleryContainer.innerHTML = xmlhttp.responseText;

                        if (typeof lazySizes !== 'undefined') {
                            lazySizes.init();

                            if (navigator 
                                && navigator.userAgent 
                                && navigator.userAgent.indexOf('Safari') != -1 
                                && navigator.userAgent.indexOf('Chrome') == -1
                            ) {
                                const images = document.querySelectorAll('img[data-src]');
                                for (let img of images) {
                                    img.src = img.getAttribute('data-src');
                                }
                            }
                        }
                        
                        if (typeof lightboxManager !== 'undefined') {
                            lightboxManager.init();
                        }                        
                    }
                }
            };
            xmlhttp.open("GET", "${endpoint}", true);
            xmlhttp.send();
        });
    `;
    html += '</scr' + 'ipt>';

    // After generate the html, put the html on the snippet by calling setHtml() method. This will render the content.
    setHtml(html);

    // We also put the custom variables on the snippet by calling setSettings() method.
    let settings = `{
        "ref": "${ref}",
        "columns": "${columns}",
        "padding": "${padding}",
        "preset": "${preset}",
        "quality": "${quality}",
        "lazy_load": ${lazy},
        "lightbox": ${lightbox},
        "class_list": "${classList}"
    }`;

    setSettings(settings);
}

/**
 * Get document ready script
 */
function getDocumentReadyScript() {
    return `
        var docReady = function (fn) {
            var stateCheck = setInterval(function () {
                if (document.readyState !== "complete") return;
                clearInterval(stateCheck);
                try { fn() } catch (e) { }
            }, 1);
        };
    `;
}

/**
 * Get the active module
 */
function getActiveModule() {
    return parent.document.querySelector("[data-module-active]");
}

/**
 *
 * @returns {string}
 */
function getSettings() {
    let activeModule = getActiveModule(); //get active module
    return decodeURIComponent(activeModule.getAttribute('data-settings'));
}

/**
 *
 * @param html
 */
function setHtml(html) {
    parent.document.querySelector('#hidContentModuleCode').value = html;
}

/**
 *
 * @param settings
 */
function setSettings(settings) {
    parent.document.querySelector('#hidContentModuleSettings').value = settings;
}

/**
 * http://stackoverflow.com/questions/1349404/generate-a-string-of-5-random-characters-in-javascript
 * @returns {string}
 */
function makeId() {
    let text = "";
    let possible = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
    for (let i = 0; i < 2; i++)
        text += possible.charAt(Math.floor(Math.random() * possible.length));

    let text2 = "";
    let possible2 = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";
    for (let i = 0; i < 5; i++)
        text2 += possible2.charAt(Math.floor(Math.random() * possible2.length));

    return text + text2;
}

// restore any saved settings
restoreSavedSettings();

// Register the events
registerFormEvents();

// Update the html for the snippet
updateSnippetHtml();