# Bongo Captcha

[![Latest Version on Packagist](https://img.shields.io/packagist/v/bongo/captcha.svg?style=flat-square)](https://packagist.org/packages/bongo/captcha)
[![Total Downloads](https://img.shields.io/packagist/dt/bongo/captcha.svg?style=flat-square)](https://packagist.org/packages/bongo/captcha)

A Laravel package providing CAPTCHA verification services for Bongo framework applications. Supports multiple CAPTCHA providers (Google reCAPTCHA v3, Cloudflare Turnstile) through a flexible driver-based architecture.

## Features

- **Multi-Provider Support**: Google reCAPTCHA v3 and Cloudflare Turnstile via driver pattern
- **Laravel Integration**: First-class validation rule and form request support
- **Score-Based Validation**: Configurable score thresholds for different security levels
- **Action-Based Tokens**: Granular control with action identifiers
- **Multiple Forms**: Support for multiple CAPTCHA instances on the same page
- **Fluent API**: Chainable setters for easy configuration
- **Testing Friendly**: Automatic skip in testing environment
- **JavaScript Integration**: Frontend handling for Google reCAPTCHA v3
- **Facade & Helper**: Multiple access methods (manager, facade, helper function)

## Requirements

- PHP 8.2+
- Laravel 10.x or 11.x
- bongo/framework ^3.0

## Installation

### Step 1: Install via Composer

```bash
composer require bongo/captcha
```

### Step 2: Publish Assets

```bash
# Publish JavaScript assets
php artisan vendor:publish --tag=captcha:assets

# Publish configuration (optional)
php artisan vendor:publish --provider="Bongo\Captcha\CaptchaServiceProvider" --tag=config
```

### Step 3: Configure Environment

Add your CAPTCHA credentials to `.env`:

```env
# Default driver (google or cloudflare)
CAPTCHA_DRIVER=google

# Google reCAPTCHA v3
CAPTCHA_GOOGLE_KEY=your-site-key
CAPTCHA_GOOGLE_SECRET=your-secret-key

# Cloudflare Turnstile (optional)
CAPTCHA_CLOUDFLARE_KEY=your-site-key
CAPTCHA_CLOUDFLARE_SECRET=your-secret-key
```

### Step 4: Get API Keys

#### Google reCAPTCHA v3

1. Visit [Google reCAPTCHA Admin](https://www.google.com/recaptcha/admin)
2. Register a new site
3. Select **reCAPTCHA v3**
4. Add your domain(s)
5. Copy the **Site Key** and **Secret Key**

#### Cloudflare Turnstile

1. Visit [Cloudflare Dashboard](https://dash.cloudflare.com/)
2. Navigate to Turnstile
3. Create a new site
4. Copy the **Site Key** and **Secret Key**

## Usage

### Basic Usage in Forms

#### 1. Add CAPTCHA to Blade Template

```blade
<form method="POST" action="{{ route('contact.store') }}">
    @csrf

    {{-- Load Google reCAPTCHA API (place in <head> or before </body>) --}}
    {!! captcha()->init() !!}

    <div class="form-group">
        <label for="name">Name</label>
        <input type="text" name="name" id="name" class="form-control" required>
    </div>

    <div class="form-group">
        <label for="email">Email</label>
        <input type="email" name="email" id="email" class="form-control" required>
    </div>

    <div class="form-group">
        <label for="message">Message</label>
        <textarea name="message" id="message" class="form-control" required></textarea>
    </div>

    {{-- Hidden CAPTCHA field (must be inside form) --}}
    {!! captcha()->field('contact_form') !!}

    <button type="submit" class="btn btn-primary">Send Message</button>
</form>

{{-- CAPTCHA JavaScript initialization (place before </body>) --}}
{!! captcha()->script('contact_form') !!}

@if ($errors->has('captcha-response'))
    <div class="alert alert-danger">
        {{ $errors->first('captcha-response') }}
    </div>
@endif
```

#### 2. Add Validation Rule to Form Request

```php
<?php

namespace App\Http\Requests;

use Bongo\Captcha\Rules\Captcha;
use Illuminate\Foundation\Http\FormRequest;

class ContactFormRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'email'],
            'message' => ['required', 'string'],
            'captcha-response' => [
                'required',
                new Captcha(
                    action: 'contact_form',    // Must match field() action
                    minScore: 0.5,              // Minimum score (0.0 - 1.0)
                    enabled: true               // Enable/disable verification
                ),
            ],
        ];
    }

    public function messages(): array
    {
        return [
            'captcha-response.required' => 'Please complete the CAPTCHA verification.',
        ];
    }
}
```

#### 3. Handle Form Submission in Controller

```php
<?php

namespace App\Http\Controllers;

use App\Http\Requests\ContactFormRequest;

class ContactController extends Controller
{
    public function create()
    {
        return view('contact.form');
    }

    public function store(ContactFormRequest $request)
    {
        // CAPTCHA already validated by form request
        $validated = $request->validated();

        // Process form data...

        return redirect()->route('contact.success')
            ->with('message', 'Your message has been sent!');
    }
}
```

### Manual Verification

If you prefer to verify CAPTCHA manually without using the validation rule:

```php
<?php

namespace App\Http\Controllers;

use Bongo\Captcha\Facades\Captcha;
use Illuminate\Http\Request;

class LoginController extends Controller
{
    public function login(Request $request)
    {
        // Get CAPTCHA token from request
        $token = $request->input('captcha-response');

        // Verify token
        $result = Captcha::verify($token, 'login');

        // Check verification result
        if (!$result['success']) {
            return back()->withErrors([
                'captcha' => $result['error'] ?? 'CAPTCHA verification failed.'
            ]);
        }

        // Check score threshold
        if ($result['score'] < 0.5) {
            return back()->withErrors([
                'captcha' => 'CAPTCHA score too low. Please try again.'
            ]);
        }

        // CAPTCHA passed, continue with login...
    }
}
```

### Multiple Forms on Same Page

You can have multiple CAPTCHA-protected forms on the same page by using different actions:

```blade
{{-- Initialize CAPTCHA API once --}}
{!! captcha()->init() !!}

{{-- Login Form --}}
<form method="POST" action="{{ route('login') }}" id="login-form">
    @csrf
    <input type="email" name="email" required>
    <input type="password" name="password" required>

    {!! captcha()->field('login') !!}

    <button type="submit">Login</button>
</form>

{{-- Register Form --}}
<form method="POST" action="{{ route('register') }}" id="register-form">
    @csrf
    <input type="text" name="name" required>
    <input type="email" name="email" required>
    <input type="password" name="password" required>

    {!! captcha()->field('register') !!}

    <button type="submit">Register</button>
</form>

{{-- Initialize both forms --}}
{!! captcha()->script('login') !!}
{!! captcha()->script('register') !!}
```

### Switching CAPTCHA Providers

#### Via Environment Variable

```env
# Use Cloudflare instead of Google
CAPTCHA_DRIVER=cloudflare
```

#### At Runtime

```php
// Use Google reCAPTCHA
$result = captcha()->driver('google')->verify($token, 'login');

// Use Cloudflare Turnstile
$result = captcha()->driver('cloudflare')->verify($token, 'login');
```

### Conditional CAPTCHA Verification

You can conditionally enable CAPTCHA based on user status or other criteria:

```php
public function rules(): array
{
    // Only require CAPTCHA for guests
    $requireCaptcha = !auth()->check();

    return [
        'email' => ['required', 'email'],
        'password' => ['required', 'string'],
        'captcha-response' => [
            $requireCaptcha ? 'required' : 'nullable',
            new Captcha(
                action: 'login',
                minScore: 0.5,
                enabled: $requireCaptcha
            ),
        ],
    ];
}
```

### Adjusting Score Thresholds

Different actions can have different score thresholds based on their sensitivity:

```php
// Low security (public forms like newsletter signup)
new Captcha('newsletter', minScore: 0.3)

// Medium security (contact forms)
new Captcha('contact', minScore: 0.5)

// High security (password reset, account changes)
new Captcha('password_reset', minScore: 0.7)

// Very high security (payments, sensitive operations)
new Captcha('payment', minScore: 0.9)
```

**Score Guidelines:**
- `0.0 - 0.3`: Likely bot
- `0.3 - 0.5`: Suspicious
- `0.5 - 0.7`: Neutral
- `0.7 - 0.9`: Likely human
- `0.9 - 1.0`: Very likely human

## Advanced Usage

### Using the Facade

```php
use Bongo\Captcha\Facades\Captcha;

// Verify token
$result = Captcha::verify($token, 'login');

// Switch driver
$result = Captcha::driver('cloudflare')->verify($token, 'login');

// Get site key
$key = Captcha::key();

// Generate HTML
$init = Captcha::init();
$field = Captcha->field('action');
$script = Captcha::script('action');
```

### Using the Helper Function

```php
// Get manager instance
$captcha = captcha();

// Verify token
$result = captcha()->verify($token, 'login');

// Switch driver
$result = captcha()->driver('cloudflare')->verify($token, 'login');

// Generate HTML
echo captcha()->init();
echo captcha()->field('login');
echo captcha()->script('login');
```

### Using Dependency Injection

```php
use Bongo\Captcha\Managers\Captcha as CaptchaManager;

class LoginController extends Controller
{
    public function __construct(
        protected CaptchaManager $captcha
    ) {}

    public function login(Request $request)
    {
        $result = $this->captcha->verify(
            $request->input('captcha-response'),
            'login'
        );

        // Process result...
    }
}
```

### Customizing Field Name

By default, the CAPTCHA field is named `captcha-response`. You can customize this:

```php
// Set custom field name
$captcha = captcha()->setReference('g-recaptcha-response');

// Generate field with custom name
echo $captcha->field('login');
// Output: <input name="g-recaptcha-response" ...>

// Update validation rule to match
public function rules(): array
{
    return [
        'g-recaptcha-response' => [
            'required',
            new Captcha('login', 0.5),
        ],
    ];
}
```

### Runtime Configuration Override

You can override configuration at runtime:

```php
$captcha = captcha()
    ->setDomain('https://custom-domain.com')
    ->setEndpoint('/custom-endpoint')
    ->setKey('custom-key')
    ->setSecret('custom-secret')
    ->setLocale('es'); // Spanish

$result = $captcha->verify($token, 'login');
```

## Testing

### Automatic Skip in Tests

CAPTCHA verification is automatically skipped when `APP_ENV=testing`:

```php
<?php

namespace Tests\Feature;

use Tests\TestCase;

class ContactFormTest extends TestCase
{
    public function test_contact_form_submission()
    {
        // CAPTCHA automatically skipped in testing environment
        $response = $this->post(route('contact.store'), [
            'name' => 'John Doe',
            'email' => 'john@example.com',
            'message' => 'Test message',
            'captcha-response' => 'any-value', // Any value works in testing
        ]);

        $response->assertRedirect(route('contact.success'));
    }
}
```

### Running Package Tests

```bash
# Run tests
composer test

# Run tests with coverage
composer test:coverage

# Run static analysis
composer analyse

# Fix code style
composer format
```

## Configuration

The package configuration is located at `config/captcha.php` after publishing:

```php
<?php

return [
    /**
     * Default CAPTCHA Driver
     *
     * Supported: "google", "cloudflare"
     */
    'driver' => env('CAPTCHA_DRIVER', 'google'),

    /**
     * CAPTCHA Service Configuration
     *
     * Configure credentials and endpoints for each CAPTCHA provider.
     */
    'services' => [
        'google' => [
            'domain' => env('CAPTCHA_GOOGLE_DOMAIN', 'https://www.google.com'),
            'endpoint' => env('CAPTCHA_GOOGLE_ENDPOINT', '/recaptcha'),
            'key' => env('CAPTCHA_GOOGLE_KEY'),
            'secret' => env('CAPTCHA_GOOGLE_SECRET'),
        ],
        'cloudflare' => [
            'domain' => env('CAPTCHA_CLOUDFLARE_DOMAIN', 'https://www.cloudflare.com'),
            'endpoint' => env('CAPTCHA_CLOUDFLARE_ENDPOINT', '/captcha'),
            'key' => env('CAPTCHA_CLOUDFLARE_KEY'),
            'secret' => env('CAPTCHA_CLOUDFLARE_SECRET'),
        ],
    ],
];
```

### Environment Variables

All configuration values can be set via environment variables:

```env
# Driver Selection
CAPTCHA_DRIVER=google

# Google reCAPTCHA v3
CAPTCHA_GOOGLE_DOMAIN=https://www.google.com
CAPTCHA_GOOGLE_ENDPOINT=/recaptcha
CAPTCHA_GOOGLE_KEY=your-site-key
CAPTCHA_GOOGLE_SECRET=your-secret-key

# Cloudflare Turnstile
CAPTCHA_CLOUDFLARE_DOMAIN=https://www.cloudflare.com
CAPTCHA_CLOUDFLARE_ENDPOINT=/captcha
CAPTCHA_CLOUDFLARE_KEY=your-site-key
CAPTCHA_CLOUDFLARE_SECRET=your-secret-key
```

## Extending the Package

### Adding a New CAPTCHA Provider

You can add support for additional CAPTCHA providers by following these steps:

#### 1. Create a Service Class

```php
<?php

declare(strict_types=1);

namespace App\Captcha\Services;

use Bongo\Captcha\Contracts\Captcha as CaptchaContract;
use Bongo\Captcha\Services\Captcha;
use Illuminate\Support\Facades\Http;

class HcaptchaCaptcha extends Captcha implements CaptchaContract
{
    protected ?string $name = 'hcaptcha';

    public function verify(string $token, ?string $action = null): array
    {
        $this->setAction($action);
        $this->validateRequest();

        $response = Http::asForm()
            ->post($this->getVerifyUrl(), [
                'secret' => $this->secret,
                'response' => $token,
                'remoteip' => $this->request->getClientIp(),
            ])
            ->json();

        return [
            'success' => $response['success'] ?? false,
            'score' => $response['score'] ?? 1.0,
            'action' => $action,
            'hostname' => $response['hostname'] ?? null,
        ];
    }

    public function getVerifyUrl(): string
    {
        return "{$this->getBaseUrl()}/siteverify";
    }

    public function init(): string
    {
        $apiJs = "{$this->getBaseUrl()}/api.js";
        return "<script src=\"{$apiJs}\" async defer></script>";
    }

    public function field(?string $action = null): string
    {
        $this->setAction($action);
        return "<input id=\"{$this->uniqueId}\" type=\"hidden\" name=\"{$this->reference}\" data-key=\"{$this->key}\" data-action=\"{$this->action}\">";
    }

    public function script(?string $action = null): string
    {
        $this->setAction($action);
        $captchaJs = asset('js/hcaptcha-captcha.js');
        return "<script src=\"{$captchaJs}\"></script><script>new HcaptchaCaptcha().init('{$this->uniqueId}');</script>";
    }
}
```

#### 2. Register the Driver

Extend the `CaptchaManager` and add a factory method:

```php
<?php

namespace App\Captcha\Managers;

use App\Captcha\Services\HcaptchaCaptcha;
use Bongo\Captcha\Contracts\Captcha as CaptchaContract;
use Bongo\Captcha\Managers\Captcha as BaseCaptchaManager;

class CaptchaManager extends BaseCaptchaManager
{
    public function createHcaptchaDriver(): CaptchaContract
    {
        return $this->container->make(HcaptchaCaptcha::class);
    }
}
```

#### 3. Add Configuration

```php
// config/captcha.php
'services' => [
    'hcaptcha' => [
        'domain' => env('CAPTCHA_HCAPTCHA_DOMAIN', 'https://hcaptcha.com'),
        'endpoint' => env('CAPTCHA_HCAPTCHA_ENDPOINT', '/api'),
        'key' => env('CAPTCHA_HCAPTCHA_KEY'),
        'secret' => env('CAPTCHA_HCAPTCHA_SECRET'),
    ],
],
```

#### 4. Use the New Driver

```env
CAPTCHA_DRIVER=hcaptcha
CAPTCHA_HCAPTCHA_KEY=your-site-key
CAPTCHA_HCAPTCHA_SECRET=your-secret-key
```

## Troubleshooting

### Common Issues

#### CAPTCHA Not Loading

**Problem:** Google reCAPTCHA script doesn't load or forms don't submit.

**Solutions:**
1. Ensure `captcha()->init()` is called before the form
2. Check browser console for JavaScript errors
3. Verify `window.grecaptcha` is defined after page load
4. Confirm JavaScript assets are published: `php artisan vendor:publish --tag=captcha:assets`

#### "timeout-or-duplicate" Error

**Problem:** Token has already been used or expired.

**Cause:** CAPTCHA tokens are single-use. Multiple validation calls would fail.

**Solution:** The package handles this automatically via the `$hasValidated` flag in the validation rule. If you see this error, it may indicate custom validation logic calling verify() multiple times.

#### "missing-input-secret" Error

**Problem:** Secret key not configured.

**Solutions:**
1. Add `CAPTCHA_GOOGLE_SECRET` to `.env`
2. Clear configuration cache: `php artisan config:clear`
3. Verify environment variable is set: `echo $CAPTCHA_GOOGLE_SECRET`

#### Low or Zero Score

**Problem:** CAPTCHA always returns score of 0.0 or fails validation.

**Causes:**
1. Using reCAPTCHA v2 keys with v3 code
2. Wrong site key in frontend vs. secret key in backend
3. Domain not whitelisted in reCAPTCHA admin

**Solutions:**
1. Ensure you're using reCAPTCHA **v3** keys
2. Verify keys match in Google admin console
3. Add your domain to the allowed domains list

#### Multiple Forms Not Working

**Problem:** Multiple forms on same page interfere with each other.

**Solution:** Use different actions for each form:

```blade
{!! captcha()->field('form1') !!}
{!! captcha()->script('form1') !!}

{!! captcha()->field('form2') !!}
{!! captcha()->script('form2') !!}
```

#### Firefox Compatibility Issues

**Problem:** Form doesn't submit in Firefox.

**Solution:** The package handles Firefox compatibility automatically by binding both button clicks and form submit events. Ensure you're using the published JavaScript assets.

## API Reference

### Captcha Manager

```php
use Bongo\Captcha\Managers\Captcha;

// Get default driver
$captcha = app(Captcha::class);

// Get specific driver
$captcha = app(Captcha::class)->driver('google');

// Verify token
$result = $captcha->verify(string $token, ?string $action = null): array;

// Get site key
$key = $captcha->key(): string;

// Generate initialization script
$script = $captcha->init(): string;

// Generate hidden field
$field = $captcha->field(?string $action = null): string;

// Generate JavaScript initialization
$script = $captcha->script(?string $action = null): string;

// Get field name (default: 'captcha-response')
$name = $captcha->reference(): string;
```

### Validation Rule

```php
use Bongo\Captcha\Rules\Captcha;

new Captcha(
    string $action,         // Action identifier (e.g., 'login')
    float $minScore = 0.5,  // Minimum score threshold (0.0 - 1.0)
    bool $enabled = true    // Enable/disable verification
)
```

### Verification Response

```php
// Success response
[
    'success' => true,
    'score' => 0.9,           // Float: 0.0 (bot) to 1.0 (human)
    'action' => 'login',      // String: Action identifier
    'hostname' => 'example.com', // String: Request hostname
]

// Failure response
[
    'success' => false,
    'error' => 'Error message', // String: Error description
]
```

## Security Considerations

1. **Never expose secret keys**: Keep `CAPTCHA_*_SECRET` in `.env` file, never commit to version control
2. **Use HTTPS**: Always use HTTPS in production to protect token transmission
3. **Validate scores**: Set appropriate score thresholds based on action sensitivity
4. **Verify actions**: The package automatically validates action matches to prevent token reuse
5. **Rate limiting**: Consider adding rate limiting to forms as an additional security layer
6. **IP logging**: Client IP addresses are sent to CAPTCHA providers for fraud detection

## Performance Considerations

1. **Singleton manager**: CaptchaManager is registered as singleton for optimal performance
2. **Lazy driver resolution**: Drivers are only instantiated when needed
3. **Asset publishing**: JavaScript is published to application's public directory to avoid package lookups
4. **Duplicate prevention**: The validation rule prevents multiple API calls via `$hasValidated` flag
5. **Caching**: Consider caching verification results for a short period if needed

## Changelog

Please see [CHANGELOG](CHANGELOG.md) for more information on recent changes.

## Contributing

Contributions are welcome! Please feel free to submit a Pull Request.

## Security Vulnerabilities

If you discover a security vulnerability, please email [stuart.elliott@bespokeuk.com](mailto:stuart.elliott@bespokeuk.com).

## Credits

- [Stuart Elliott](https://github.com/stuartjameselliot)
- [All Contributors](../../contributors)

## License

The MIT License (MIT). Please see [License File](LICENSE) for more information.

## Support

For support, please:
- Open an issue on [BitBucket](https://bitbucket.org/designtec/captcha/issues)
- Email [stuart.elliott@bespokeuk.com](mailto:stuart.elliott@bespokeuk.com)

## Related Packages

- [bongo/framework](https://designtecpackages.co.uk/packages/bongo/framework) - Base framework package
- [bongo/user](https://designtecpackages.co.uk/packages/bongo/user) - User management package
- [bongo/asset](https://designtecpackages.co.uk/packages/bongo/asset) - Asset management package

## Further Documentation

- **[CLAUDE.md](CLAUDE.md)** - Quick reference guide for Claude Code
- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Detailed architecture documentation with diagrams
- **[.cursorrules](.cursorrules)** - Development guidelines for Cursor AI
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)** - GitHub Copilot instructions
