function GoogleCaptcha() {
    this.field = null;
    this.key = null;
    this.action = null;
    this.browser = null;

    this.init = function (uniqueId) {
        this.checkForUniqueId(uniqueId);
        this.checkForRecaptchaScript();
        this.setBrowser();
        this.setField(uniqueId);
        this.setKey();
        this.setAction();
        this.setForm();
        this.setSubmitEventHandler();
    }

    this.checkForUniqueId = function (uniqueId) {
        if (! uniqueId) {
            throw 'Unique ID is required for Google reCAPTCHA.';
        }
    }

    this.checkForRecaptchaScript = function () {
        if (typeof window.grecaptcha === 'undefined') {
            throw 'Google reCAPTCHA script not loaded.';
        }
    }

    this.setBrowser = function () {
        this.browser = navigator.userAgent;
    }

    this.isFireFox = function () {
        return this.browser.includes('Firefox');
    }

    this.setField = function (uniqueId) {
        this.field = document.getElementById(uniqueId);
        if (! this.field) {
            throw `Element with ID ${uniqueId} not found.`;
        }
    }

    this.setKey = function () {
        this.key = this.field.getAttribute('data-key');
        if (! this.key) {
            throw 'Data-key attribute is required for Google reCAPTCHA.';
        }
    }

    this.setAction = function () {
        this.action = this.field.getAttribute('data-action') || null;
    }

    this.setForm = function () {
        this.form = this.field.closest('form');
        if (! this.form) {
            throw 'Google reCAPTCHA field must be inside a form.';
        }
    }

    this.setSubmitEventHandler = function () {

        // Bind the submit buttons for firefox compatibility
        let submitButtons = this.form.querySelectorAll('button[type="submit"], input[type="submit"]');
        submitButtons.forEach((button) => {
            button.addEventListener('click', (event) => {
                event.preventDefault();
                this.generateToken();
            });
        });

        // Bind the form submit event for other browsers
        this.form.addEventListener('submit', (event) => {
            event.preventDefault();
            this.generateToken();
        });
    }

    this.generateToken = function () {
        window.grecaptcha.ready(() => {
            window.grecaptcha
                .execute(this.key, {
                    action: this.action
                })
                .then((token) => {
                    this.field.value = token;
                    this.form.submit();
                })
                .catch((error) => {
                    console.error('Recaptcha error:', error);
                });
        });
    }
}
