<?php

declare(strict_types=1);

namespace Bongo\Captcha\Services;

use Bongo\Captcha\Exceptions\CaptchaException;
use Illuminate\Http\Request;

abstract class Captcha
{
    protected Request $request;

    protected ?string $name = null;

    protected ?string $action = null;

    protected ?string $reference = null;

    protected ?string $uniqueId = null;

    protected ?string $domain = null;

    protected ?string $endpoint = null;

    protected ?string $key = null;

    protected ?string $secret = null;

    protected ?string $locale = null;

    /** @throws CaptchaException */
    public function __construct(Request $request)
    {
        if (empty($this->name)) {
            throw CaptchaException::invalidProperty(static::class, 'name');
        }

        $this->boot($request);
    }

    public function boot(Request $request): void
    {
        $this->setRequest($request)
            ->setReference()
            ->setDomain()
            ->setEndpoint()
            ->setKey()
            ->setSecret()
            ->setLocale();
    }

    public function key(): string
    {
        return $this->key;
    }

    public function getBaseUrl(): string
    {
        return "{$this->domain}/{$this->endpoint}";
    }

    public function setName(string $name): self
    {
        $this->name = $name;

        return $this;
    }

    public function setAction(?string $action = null): self
    {
        $this->action = $action;

        if (empty($this->uniqueId) || $this->uniqueId !== $this->makeUniqueId()) {
            $this->setUniqueId();
        }

        return $this;
    }

    public function reference(): string
    {
        return $this->reference;
    }

    public function setReference(?string $reference = null): self
    {
        $this->reference = $reference ?? 'captcha-response';

        return $this;
    }

    public function makeUniqueId(): string
    {
        return base64_encode("{$this->reference}-{$this->action}-");
    }

    public function setUniqueId(): self
    {
        $this->uniqueId = $this->makeUniqueId();

        return $this;
    }

    public function setRequest(Request $request): self
    {
        $this->request = $request;

        return $this;
    }

    public function setDomain(?string $domain = null): self
    {
        $this->domain = trim($domain ?? config("captcha.services.{$this->name}.domain"), '/');

        return $this;
    }

    public function setEndpoint(?string $endpoint = null): self
    {
        $this->endpoint = trim($endpoint ?? config("captcha.services.{$this->name}.endpoint"), '/');

        return $this;
    }

    public function setKey(?string $key = null): self
    {
        $this->key = $key ?? config("captcha.services.{$this->name}.key");

        return $this;
    }

    public function setSecret(?string $secret = null): self
    {
        $this->secret = $secret ?? config("captcha.services.{$this->name}.secret");

        return $this;
    }

    public function setLocale(?string $locale = null): self
    {
        $this->locale = $locale ?? app()->getLocale();

        return $this;
    }

    /** @throws CaptchaException */
    protected function validateRequest(): void
    {
        if (empty($this->domain)) {
            throw CaptchaException::invalidProperty(static::class, 'domain');
        }

        if (empty($this->endpoint)) {
            throw CaptchaException::invalidProperty(static::class, 'endpoint');
        }

        if (empty($this->key)) {
            throw CaptchaException::invalidProperty(static::class, 'key');
        }

        if (empty($this->secret)) {
            throw CaptchaException::invalidProperty(static::class, 'secret');
        }
    }
}
