<?php

declare(strict_types=1);

namespace Bongo\Captcha\Services;

use Bongo\Captcha\Contracts\Captcha as CaptchaContract;
use Bongo\Captcha\Exceptions\CaptchaException;
use Bongo\Captcha\Exceptions\InvalidActionException;
use Bongo\Captcha\Exceptions\InvalidRequestException;
use Bongo\Captcha\Exceptions\InvalidResponseException;
use Bongo\Captcha\Exceptions\InvalidSecretException;
use Bongo\Captcha\Exceptions\InvalidTokenException;
use Bongo\Captcha\Exceptions\MissingSecretException;
use Bongo\Captcha\Exceptions\MissingTokenException;
use Bongo\Captcha\Exceptions\TimeOutOrDuplicateException;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class GoogleCaptcha extends Captcha implements CaptchaContract
{
    protected ?string $name = 'google';

    /** @throws CaptchaException */
    public function verify(string $token, ?string $action = null): array
    {
        $this->setAction($action);
        $this->validateRequest();

        try {
            // Make the request to the Google reCAPTCHA API to verify the token
            $response = Http::asForm()
                ->post($this->getVerifyUrl(), [
                    'secret' => $this->secret,
                    'response' => $token,
                    'remoteip' => $this->request->getClientIp(),
                ])
                ->json();

            // Validate the response from the API
            $this->validateResponse($response);

            // Otherwise return the response data
            return [
                'success' => true,
                'score' => $response['score'] ?? null,
                'action' => $response['action'] ?? null,
                'hostname' => $response['hostname'] ?? null,
            ];

        } catch (CaptchaException $e) {
            Log::error($e->getMessage(), [
                'captcha' => $this->name,
                'token' => $token,
                'action' => $this->action,
                'response' => isset($response) ? json_encode($response) : null,
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function getVerifyUrl(): string
    {
        return "{$this->getBaseUrl()}/api/siteverify";
    }

    /** @throws CaptchaException */
    public function init(): string
    {
        if (empty($this->key)) {
            throw CaptchaException::invalidProperty(static::class, 'key');
        }

        // Generate the script tag for the Google reCAPTCHA API
        $apiJs = "{$this->getBaseUrl()}/api.js?hl={$this->locale}&render={$this->key}";

        return "<script src=\"{$apiJs}\" async defer></script>";
    }

    /** @throws CaptchaException */
    public function field(?string $action = null): string
    {
        if (empty($this->key)) {
            throw CaptchaException::invalidProperty(static::class, 'key');
        }

        $this->setAction($action);

        return "
            <input
                id=\"{$this->uniqueId}\"
                class=\"g-recaptcha\"
                type=\"hidden\"
                name=\"{$this->reference}\"
                data-key=\"{$this->key}\"
                data-action=\"{$this->action}\"
            >
        ";
    }

    /** @throws CaptchaException */
    public function script(?string $action = null): string
    {
        if (empty($this->key)) {
            throw CaptchaException::invalidProperty(static::class, 'key');
        }

        $this->setAction($action);

        // Generate the script tag for our javascript functionality
        $captchaJs = asset('js/google-captcha.js');
        $html = "<script src=\"{$captchaJs}\" async defer></script>";

        // Add the initialization script for the Google Captcha
        $html .= "
            <script data-action=\"{$this->action}\" type=\"text/javascript\">
                window.addEventListener('load', () => {
                    if (typeof GoogleCaptcha === 'undefined') {
                        console.log('Google Captcha not loaded');

                        return;
                    }
                    try {
                        new GoogleCaptcha().init('{$this->uniqueId}');
                    } catch (error) {
                        console.error(error);
                    }
                });
            </script>
        ";

        return $html;
    }

    /** @throws CaptchaException */
    protected function validateResponse(?array $response = null): void
    {
        if (empty($response)) {
            throw new InvalidResponseException(trans('captcha::messages.invalid_response'));
        }

        // If the response was successful, then bail
        if (isset($response['success']) && (bool) $response['success'] === true) {
            return;
        }

        // If there are no error codes, then bail
        if (! isset($response['error-codes'])) {
            return;
        }

        // Validate the secret
        if (in_array('missing-input-secret', $response['error-codes'], true)) {
            throw new MissingSecretException(trans('captcha::messages.missing_secret'));
        }
        if (in_array('invalid-input-secret', $response['error-codes'], true)) {
            throw new InvalidSecretException(trans('captcha::messages.invalid_secret'));
        }

        // Validate the token
        if (in_array('missing-input-response', $response['error-codes'], true)) {
            throw new MissingTokenException(trans('captcha::messages.missing_token'));
        }
        if (in_array('invalid-input-response', $response['error-codes'], true)) {
            throw new InvalidTokenException(trans('captcha::messages.invalid_token'));
        }

        // Validate the request
        if (in_array('bad-request', $response['error-codes'], true)) {
            throw new InvalidRequestException(trans('captcha::messages.invalid_request'));
        }
        if (in_array('timeout-or-duplicate', $response['error-codes'], true)) {
            throw new TimeOutOrDuplicateException(trans('captcha::messages.timeout_or_duplicate'));
        }

        // Validate the action
        if ($this->action && (! isset($response['action']) || $this->action !== $response['action'])) {
            throw new InvalidActionException(trans('captcha::messages.invalid_action'));
        }
    }
}
