# GitHub Copilot Instructions - Bongo Captcha

## Project Overview

Laravel package providing CAPTCHA verification services for Bongo framework applications. Supports Google reCAPTCHA v3 and Cloudflare Turnstile through a driver-based architecture using Laravel's Manager pattern.

**Core Functionality:**
- Multi-provider CAPTCHA support
- Score-based validation with configurable thresholds
- Action-based tokens for granular control
- Laravel validation rule integration
- JavaScript frontend integration
- Helper functions and facades

## Key Classes and Relationships

### 1. Service Provider

**`Bongo\Captcha\CaptchaServiceProvider`**

Entry point extending `Bongo\Framework\Providers\AbstractServiceProvider`.

```php
class CaptchaServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'captcha';

    public function register(): void
    {
        // Binds CaptchaManager as singleton
        $this->app->singleton(CaptchaManager::class, function ($app) {
            return new CaptchaManager($app);
        });
    }

    public function boot(): void
    {
        parent::boot(); // Auto-loads config, translations
        $this->bootAssets(); // Publishes JS to public/js/
        $this->app->booted(fn() => include __DIR__.'/helpers.php');
    }
}
```

### 2. Manager (Driver Resolver)

**`Bongo\Captcha\Managers\Captcha`**

Extends `Illuminate\Support\Manager` to resolve CAPTCHA drivers.

```php
class Captcha extends Manager
{
    // Returns config('captcha.driver') or 'google'
    public function getDefaultDriver()

    // Factory methods for drivers
    public function createCloudflareDriver(): CaptchaContract
    public function createGoogleDriver(): CaptchaContract
}
```

**Usage:**
```php
// Resolve default driver
$captcha = app(CaptchaManager::class);

// Resolve specific driver
$captcha = app(CaptchaManager::class)->driver('google');
```

### 3. Contract (Interface)

**`Bongo\Captcha\Contracts\Captcha`**

Interface all CAPTCHA services must implement.

```php
interface Captcha
{
    // Verify token with optional action
    public function verify(string $token, ?string $action = null): array;

    // Get site key
    public function key(): string;

    // Generate script tag to load CAPTCHA API
    public function init(): string;

    // Generate hidden field HTML
    public function field(?string $action = null): string;

    // Generate initialization JavaScript
    public function script(?string $action = null): string;

    // Get field reference name
    public function reference(): string;

    // Fluent setters (return AbstractCaptcha)
    public function setRequest(Request $request): AbstractCaptcha;
    public function setReference(?string $reference = null): AbstractCaptcha;
    public function setAction(?string $action = null): AbstractCaptcha;
    public function setDomain(?string $domain = null): AbstractCaptcha;
    public function setEndpoint(?string $endpoint = null): AbstractCaptcha;
    public function setKey(?string $key = null): AbstractCaptcha;
    public function setSecret(?string $secret = null): AbstractCaptcha;
    public function setLocale(?string $locale = null): AbstractCaptcha;
}
```

### 4. Abstract Service

**`Bongo\Captcha\Services\Captcha`**

Base class providing common functionality.

```php
abstract class Captcha
{
    protected ?string $name = null;       // Driver name (e.g., 'google')
    protected ?string $action = null;     // Action identifier
    protected ?string $reference = null;  // Field name
    protected ?string $uniqueId = null;   // Unique field ID
    protected ?string $domain = null;     // API domain
    protected ?string $endpoint = null;   // API endpoint
    protected ?string $key = null;        // Site key
    protected ?string $secret = null;     // Secret key
    protected ?string $locale = null;     // Locale code

    public function boot(Request $request): void
    {
        // Initializes all properties from config/request
        $this->setRequest($request)
            ->setReference()
            ->setDomain()
            ->setEndpoint()
            ->setKey()
            ->setSecret()
            ->setLocale();
    }

    // Generate unique ID for multiple instances
    public function makeUniqueId(): string
    {
        return base64_encode("{$this->reference}-{$this->action}-");
    }

    // Validate required properties are set
    protected function validateRequest(): void
}
```

### 5. Concrete Services

**`Bongo\Captcha\Services\GoogleCaptcha`**

Google reCAPTCHA v3 implementation.

```php
class GoogleCaptcha extends Captcha implements CaptchaContract
{
    protected ?string $name = 'google';

    public function verify(string $token, ?string $action = null): array
    {
        // POST to /recaptcha/api/siteverify
        // Returns: ['success' => bool, 'score' => float, 'action' => string, 'hostname' => string]
    }

    public function init(): string
    {
        // Returns: <script src="https://www.google.com/recaptcha/api.js?hl={locale}&render={key}"></script>
    }

    public function field(?string $action = null): string
    {
        // Returns: <input type="hidden" name="{reference}" data-key="{key}" data-action="{action}">
    }

    public function script(?string $action = null): string
    {
        // Returns: <script src="/js/google-captcha.js"></script> + initialization code
    }

    protected function validateResponse(?array $response = null): void
    {
        // Throws specific exceptions based on error-codes array
    }
}
```

**`Bongo\Captcha\Services\CloudflareCaptcha`**

Cloudflare Turnstile implementation (placeholder).

```php
class CloudflareCaptcha extends Captcha implements CaptchaContract
{
    protected ?string $name = 'cloudflare';

    // Currently returns success without verification
    public function verify(string $token, ?string $action = null): array
    {
        return ['success' => true, 'score' => 1.0, 'action' => $action, 'hostname' => $this->request->getHost()];
    }

    // Empty implementations (to be implemented)
    public function init(): string { return ''; }
    public function field(?string $action = null): string { return ''; }
    public function script(?string $action = null): string { return ''; }
}
```

### 6. Validation Rule

**`Bongo\Captcha\Rules\Captcha`**

Laravel ValidationRule implementation.

```php
class Captcha implements ValidationRule
{
    protected bool $hasValidated = false; // Prevents duplicate validation

    public function __construct(
        string $action,         // Action to verify (e.g., 'login')
        float $minScore = 0.5,  // Minimum score threshold
        bool $enabled = true    // Enable/disable flag
    ) {}

    public function validate(string $attribute, mixed $value, Closure $fail): void
    {
        // Skips if already validated (prevents duplicate API calls)
        if ($this->hasValidated) return;
        $this->hasValidated = true;

        // Skips in testing environment
        if (app()->environment('testing')) return;

        // Skips if disabled
        if (!$this->enabled) return;

        // Verifies token
        $result = app(CaptchaManager::class)->verify($value, $this->action);

        // Fails if unsuccessful or score too low
        if (!$result['success'] || !isset($result['score'])) {
            $fail('captcha::validation.no_score')->translate([...]);
        }

        if ($result['score'] < $this->minScore) {
            $fail('captcha::validation.min_score')->translate([...]);
        }
    }
}
```

### 7. Facade

**`Bongo\Captcha\Facades\Captcha`**

Facade for `CaptchaManager`.

```php
class Captcha extends Facade
{
    protected static function getFacadeAccessor(): string
    {
        return CaptchaManager::class;
    }
}
```

**Usage:**
```php
use Bongo\Captcha\Facades\Captcha;

$result = Captcha::verify($token, 'login');
$result = Captcha::driver('cloudflare')->verify($token);
```

### 8. Helper Function

**`captcha()`** - Global helper returning `CaptchaManager` instance.

```php
function captcha(): CaptchaManager
{
    return app(CaptchaManager::class);
}
```

**Usage:**
```php
echo captcha()->init();
echo captcha()->field('contact');
echo captcha()->script('contact');
```

## Class Hierarchy

```
Illuminate\Support\Manager
└── Bongo\Captcha\Managers\Captcha

Bongo\Captcha\Services\Captcha (abstract)
├── Bongo\Captcha\Services\GoogleCaptcha
└── Bongo\Captcha\Services\CloudflareCaptcha

Bongo\Captcha\Contracts\Captcha (interface)
├── Implemented by: GoogleCaptcha
└── Implemented by: CloudflareCaptcha

Exception
└── Bongo\Captcha\Exceptions\CaptchaException
    ├── InvalidActionException
    ├── InvalidRequestException
    ├── InvalidResponseException
    ├── InvalidSecretException
    ├── InvalidTokenException
    ├── MissingSecretException
    ├── MissingTokenException
    └── TimeOutOrDuplicateException
```

## Code Style Templates

### Adding a New CAPTCHA Service

```php
<?php

declare(strict_types=1);

namespace Bongo\Captcha\Services;

use Bongo\Captcha\Contracts\Captcha as CaptchaContract;
use Bongo\Captcha\Exceptions\CaptchaException;
use Illuminate\Support\Facades\Http;

class HcaptchaCaptcha extends Captcha implements CaptchaContract
{
    protected ?string $name = 'hcaptcha';

    /** @throws CaptchaException */
    public function verify(string $token, ?string $action = null): array
    {
        $this->setAction($action);
        $this->validateRequest();

        $response = Http::asForm()
            ->post($this->getVerifyUrl(), [
                'secret' => $this->secret,
                'response' => $token,
                'remoteip' => $this->request->getClientIp(),
            ])
            ->json();

        return [
            'success' => $response['success'] ?? false,
            'score' => $response['score'] ?? null,
            'action' => $action,
            'hostname' => $response['hostname'] ?? null,
        ];
    }

    public function getVerifyUrl(): string
    {
        return "{$this->getBaseUrl()}/siteverify";
    }

    /** @throws CaptchaException */
    public function init(): string
    {
        if (empty($this->key)) {
            throw CaptchaException::invalidProperty(static::class, 'key');
        }

        $apiJs = "{$this->getBaseUrl()}/api.js?hl={$this->locale}";
        return "<script src=\"{$apiJs}\" async defer></script>";
    }

    /** @throws CaptchaException */
    public function field(?string $action = null): string
    {
        if (empty($this->key)) {
            throw CaptchaException::invalidProperty(static::class, 'key');
        }

        $this->setAction($action);

        return "<input id=\"{$this->uniqueId}\" class=\"h-captcha\" type=\"hidden\" name=\"{$this->reference}\" data-key=\"{$this->key}\" data-action=\"{$this->action}\">";
    }

    /** @throws CaptchaException */
    public function script(?string $action = null): string
    {
        if (empty($this->key)) {
            throw CaptchaException::invalidProperty(static::class, 'key');
        }

        $this->setAction($action);

        $captchaJs = asset('js/hcaptcha-captcha.js');
        return "<script src=\"{$captchaJs}\" async defer></script>
        <script>
            window.addEventListener('load', () => {
                new HcaptchaCaptcha().init('{$this->uniqueId}');
            });
        </script>";
    }
}
```

### Using CAPTCHA in Form Requests

```php
<?php

declare(strict_types=1);

namespace App\Http\Requests;

use Bongo\Captcha\Rules\Captcha;
use Illuminate\Foundation\Http\FormRequest;

class ContactFormRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'email'],
            'message' => ['required', 'string'],
            'captcha-response' => [
                'required',
                new Captcha(
                    action: 'contact_form',
                    minScore: 0.5,
                    enabled: config('captcha.enabled', true)
                ),
            ],
        ];
    }
}
```

### Controller Usage Pattern

```php
<?php

declare(strict_types=1);

namespace App\Http\Controllers;

use App\Http\Requests\ContactFormRequest;
use Bongo\Captcha\Facades\Captcha;

class ContactController extends Controller
{
    public function create()
    {
        // Blade will use these helper methods
        return view('contact.form');
    }

    // Using Form Request (recommended)
    public function store(ContactFormRequest $request)
    {
        // CAPTCHA already validated by form request
        $validated = $request->validated();

        // Process form...

        return redirect()->route('contact.success');
    }

    // Manual verification (alternative)
    public function storeManual(Request $request)
    {
        $result = Captcha::verify(
            $request->input('captcha-response'),
            'contact_form'
        );

        if (!$result['success'] || $result['score'] < 0.5) {
            return back()->withErrors(['captcha' => 'CAPTCHA verification failed.']);
        }

        // Process form...

        return redirect()->route('contact.success');
    }
}
```

### Blade Template Pattern

```blade
<form method="POST" action="{{ route('contact.store') }}">
    @csrf

    {{-- Initialize CAPTCHA API (place in <head> or before closing </body>) --}}
    {!! captcha()->init() !!}

    <div class="form-group">
        <label for="name">Name</label>
        <input type="text" name="name" id="name" class="form-control" required>
    </div>

    <div class="form-group">
        <label for="email">Email</label>
        <input type="email" name="email" id="email" class="form-control" required>
    </div>

    <div class="form-group">
        <label for="message">Message</label>
        <textarea name="message" id="message" class="form-control" required></textarea>
    </div>

    {{-- Hidden CAPTCHA field (must be inside form) --}}
    {!! captcha()->field('contact_form') !!}

    <button type="submit" class="btn btn-primary">Send Message</button>
</form>

{{-- CAPTCHA JavaScript (place before closing </body>) --}}
{!! captcha()->script('contact_form') !!}

@if ($errors->has('captcha-response'))
    <div class="alert alert-danger">{{ $errors->first('captcha-response') }}</div>
@endif
```

## Common Patterns

### 1. Driver Switching

```php
// Switch driver at runtime
$googleCaptcha = captcha()->driver('google');
$cloudflareCaptcha = captcha()->driver('cloudflare');

// Use specific driver
$result = captcha()->driver('google')->verify($token, 'login');
```

### 2. Multiple Forms on Same Page

```php
// Login form
{!! captcha()->field('login') !!}
{!! captcha()->script('login') !!}

// Register form
{!! captcha()->field('register') !!}
{!! captcha()->script('register') !!}

// Each generates unique ID: base64_encode('captcha-response-login-')
```

### 3. Conditional CAPTCHA

```php
public function rules(): array
{
    return [
        'captcha-response' => [
            'required_if:needs_verification,true',
            new Captcha(
                action: 'protected_action',
                minScore: 0.7,
                enabled: $this->shouldVerifyCaptcha()
            ),
        ],
    ];
}

private function shouldVerifyCaptcha(): bool
{
    // Disable for trusted users, admins, etc.
    return !auth()->check() || !auth()->user()->isTrusted();
}
```

### 4. Custom Error Messages

```php
public function messages(): array
{
    return [
        'captcha-response.required' => 'Please complete the CAPTCHA verification.',
    ];
}

public function attributes(): array
{
    return [
        'captcha-response' => 'CAPTCHA',
    ];
}
```

### 5. Testing Pattern

```php
// tests/Feature/ContactFormTest.php
public function test_contact_form_submission(): void
{
    // CAPTCHA automatically skipped in testing environment
    $response = $this->post(route('contact.store'), [
        'name' => 'John Doe',
        'email' => 'john@example.com',
        'message' => 'Test message',
        'captcha-response' => 'any-value-works-in-testing',
    ]);

    $response->assertRedirect(route('contact.success'));
}
```

## Configuration Reference

```php
// config/captcha.php
return [
    'driver' => env('CAPTCHA_DRIVER', 'google'),

    'services' => [
        'google' => [
            'domain' => env('CAPTCHA_GOOGLE_DOMAIN', 'https://www.google.com'),
            'endpoint' => env('CAPTCHA_GOOGLE_ENDPOINT', '/recaptcha'),
            'key' => env('CAPTCHA_GOOGLE_KEY'),
            'secret' => env('CAPTCHA_GOOGLE_SECRET'),
        ],
        'cloudflare' => [
            'domain' => env('CAPTCHA_CLOUDFLARE_DOMAIN', 'https://www.cloudflare.com'),
            'endpoint' => env('CAPTCHA_CLOUDFLARE_ENDPOINT', '/captcha'),
            'key' => env('CAPTCHA_CLOUDFLARE_KEY'),
            'secret' => env('CAPTCHA_CLOUDFLARE_SECRET'),
        ],
    ],
];
```

## Environment Variables

```env
# CAPTCHA Driver Selection
CAPTCHA_DRIVER=google

# Google reCAPTCHA v3
CAPTCHA_GOOGLE_KEY=6LdXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
CAPTCHA_GOOGLE_SECRET=6LdXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

# Cloudflare Turnstile
CAPTCHA_CLOUDFLARE_KEY=0x4XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
CAPTCHA_CLOUDFLARE_SECRET=0x4XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
```

## Key Files Reference

| File | Purpose |
|------|---------|
| `src/CaptchaServiceProvider.php` | Service provider, bootstraps package |
| `src/Managers/Captcha.php` | Driver manager, resolves CAPTCHA services |
| `src/Services/Captcha.php` | Abstract base service with common functionality |
| `src/Services/GoogleCaptcha.php` | Google reCAPTCHA v3 implementation |
| `src/Services/CloudflareCaptcha.php` | Cloudflare Turnstile (placeholder) |
| `src/Rules/Captcha.php` | Laravel validation rule |
| `src/Contracts/Captcha.php` | Interface for CAPTCHA services |
| `src/helpers.php` | Global helper function |
| `public/js/google-captcha.js` | Frontend JavaScript for Google reCAPTCHA |
