# Bongo Captcha - Architecture Documentation

## Table of Contents

1. [Overview](#overview)
2. [Directory Structure](#directory-structure)
3. [Class Architecture](#class-architecture)
4. [Verification Flow](#verification-flow)
5. [Driver System](#driver-system)
6. [Validation Rule System](#validation-rule-system)
7. [Frontend Integration](#frontend-integration)
8. [Exception Hierarchy](#exception-hierarchy)
9. [Configuration System](#configuration-system)
10. [Extension Points](#extension-points)

## Overview

The Bongo Captcha package provides a flexible, driver-based CAPTCHA verification system for Laravel applications within the Bongo framework. It implements a clean separation of concerns with support for multiple CAPTCHA providers through Laravel's Manager pattern.

**Design Principles:**
- **Driver Pattern**: Multiple CAPTCHA providers through Laravel Manager
- **Contract-Based**: All drivers implement `Contracts\Captcha` interface
- **Fluent API**: Chainable setters for configuration
- **Validation Integration**: First-class Laravel validation rule support
- **Frontend Abstraction**: Provider-agnostic JavaScript generation
- **Exception Specificity**: Detailed exception types for different failure scenarios

## Directory Structure

```
src/
├── Config/
│   └── captcha.php                      # Driver configuration
│
├── Contracts/
│   └── Captcha.php                      # CAPTCHA service interface
│
├── Exceptions/                          # Exception hierarchy
│   ├── CaptchaException.php             # Base exception with static factories
│   ├── InvalidActionException.php       # Action mismatch
│   ├── InvalidRequestException.php      # Malformed request
│   ├── InvalidResponseException.php     # Empty/invalid API response
│   ├── InvalidSecretException.php       # Invalid secret key
│   ├── InvalidTokenException.php        # Invalid/expired token
│   ├── MissingSecretException.php       # Secret key not configured
│   ├── MissingTokenException.php        # Token not provided
│   └── TimeOutOrDuplicateException.php  # Timeout/duplicate submission
│
├── Facades/
│   └── Captcha.php                      # Facade for CaptchaManager
│
├── Managers/
│   └── Captcha.php                      # Driver manager (extends Manager)
│
├── Rules/
│   └── Captcha.php                      # ValidationRule implementation
│
├── Services/
│   ├── Captcha.php                      # Abstract base service
│   ├── GoogleCaptcha.php                # Google reCAPTCHA v3 implementation
│   └── CloudflareCaptcha.php            # Cloudflare Turnstile (placeholder)
│
├── Translations/
│   └── en/
│       ├── messages.php                 # Exception messages
│       └── validation.php               # Validation error messages
│
├── CaptchaServiceProvider.php           # Service provider
└── helpers.php                          # Global helper: captcha()

public/
└── js/
    ├── google-captcha.js                # Google reCAPTCHA frontend
    └── cloudflare-captcha.js            # Cloudflare frontend (placeholder)

tests/
├── TestCase.php                         # Base test case
└── ExampleTest.php                      # Example tests
```

## Class Architecture

### High-Level Class Diagram

```
┌─────────────────────────────────────────────────────────────────┐
│                    Laravel Application                          │
└───────────────┬─────────────────────────────────────────────────┘
                │
                │ Uses
                ▼
┌───────────────────────────────────────────────────────────────┐
│               CaptchaServiceProvider                          │
│  extends: Bongo\Framework\Providers\AbstractServiceProvider  │
│                                                               │
│  - Registers CaptchaManager singleton                        │
│  - Publishes assets to public/js/                            │
│  - Loads helpers.php                                         │
└───────────────┬───────────────────────────────────────────────┘
                │ Registers
                ▼
┌────────────────────────────────────────────────────────────────┐
│         Bongo\Captcha\Managers\Captcha                        │
│         extends: Illuminate\Support\Manager                   │
│                                                               │
│  + getDefaultDriver(): string                                │
│  + createGoogleDriver(): CaptchaContract                     │
│  + createCloudflareDriver(): CaptchaContract                 │
└────────────────┬───────────────────────────────────────────────┘
                 │ Resolves
                 ▼
┌────────────────────────────────────────────────────────────────┐
│              Bongo\Captcha\Contracts\Captcha                  │
│              interface                                        │
│                                                               │
│  + verify(string $token, ?string $action): array             │
│  + key(): string                                             │
│  + init(): string                                            │
│  + field(?string $action): string                            │
│  + script(?string $action): string                           │
│  + reference(): string                                       │
│  + set*(...): AbstractCaptcha (fluent setters)               │
└────────────────┬───────────────────────────────────────────────┘
                 │ Implemented by
                 ▼
    ┌────────────┴────────────┐
    │                         │
    ▼                         ▼
┌─────────────────┐   ┌──────────────────┐
│ GoogleCaptcha   │   │CloudflareCaptcha │
│ extends Captcha │   │extends Captcha   │
│ implements      │   │implements        │
│ CaptchaContract │   │CaptchaContract   │
└─────────────────┘   └──────────────────┘
         │
         │ Uses
         ▼
┌─────────────────────────────────────────┐
│     Bongo\Captcha\Services\Captcha      │
│     abstract class                      │
│                                         │
│  # $name: ?string                       │
│  # $action: ?string                     │
│  # $reference: ?string                  │
│  # $uniqueId: ?string                   │
│  # $domain: ?string                     │
│  # $endpoint: ?string                   │
│  # $key: ?string                        │
│  # $secret: ?string                     │
│  # $locale: ?string                     │
│                                         │
│  + boot(Request): void                  │
│  + makeUniqueId(): string               │
│  + set*(...): self (fluent setters)     │
│  # validateRequest(): void              │
└─────────────────────────────────────────┘
```

### Facade and Helper

```
┌────────────────────────────────────────┐
│   Bongo\Captcha\Facades\Captcha       │
│   extends Illuminate\Support\Facade   │
│                                       │
│   + getFacadeAccessor(): string       │
│     returns: CaptchaManager::class    │
└────────────────────────────────────────┘
                │
                │ Proxies to
                ▼
┌────────────────────────────────────────┐
│      CaptchaManager (singleton)       │
└────────────────────────────────────────┘

┌────────────────────────────────────────┐
│       Global Helper Function          │
│                                       │
│   function captcha(): CaptchaManager  │
│   {                                   │
│       return app(CaptchaManager);     │
│   }                                   │
└────────────────────────────────────────┘
```

### Validation Rule Class

```
┌─────────────────────────────────────────────────────────┐
│          Bongo\Captcha\Rules\Captcha                   │
│          implements ValidationRule                     │
│                                                        │
│  # $hasValidated: bool = false                        │
│  # $action: string                                    │
│  # $minScore: float                                   │
│  # $enabled: bool                                     │
│                                                        │
│  + __construct(                                       │
│      string $action,                                  │
│      float $minScore = 0.5,                           │
│      bool $enabled = true                             │
│    )                                                  │
│                                                        │
│  + validate(                                          │
│      string $attribute,                               │
│      mixed $value,                                    │
│      Closure $fail                                    │
│    ): void                                            │
│                                                        │
│    Logic:                                             │
│    1. Skip if already validated (prevents duplicates) │
│    2. Skip in testing environment                     │
│    3. Skip if disabled                                │
│    4. Verify token via CaptchaManager                 │
│    5. Fail if score < minScore                        │
└─────────────────────────────────────────────────────────┘
```

## Verification Flow

### End-to-End Flow Diagram

```
┌──────────┐                                    ┌──────────────┐
│  User    │                                    │  Laravel App │
└────┬─────┘                                    └──────┬───────┘
     │                                                 │
     │ 1. Load Form                                    │
     ├────────────────────────────────────────────────>│
     │                                                 │
     │                                                 │ 2. Render View
     │                                                 ├──────────┐
     │                                                 │          │
     │                                                 │ captcha()->init()
     │                                                 │ captcha()->field('action')
     │                                                 │ captcha()->script('action')
     │                                                 │          │
     │                                                 │<─────────┘
     │                                                 │
     │ 3. HTML with CAPTCHA fields                     │
     │<────────────────────────────────────────────────┤
     │                                                 │
     │                                                 │
┌────┴─────────────────────────────────────────┐      │
│  Browser (JavaScript Execution)              │      │
│                                              │      │
│  4. Google reCAPTCHA API loads               │      │
│  5. new GoogleCaptcha().init(uniqueId)       │      │
│  6. Binds submit event handlers              │      │
└──────────────────────────────────────────────┘      │
     │                                                 │
     │ 7. User submits form                            │
     ├──────────────┐                                  │
     │              │                                  │
     │ 8. JS intercepts submit                         │
     │              │                                  │
     │              │                                  │
┌────▼──────────────▼──────────────────────────┐      │
│  9. grecaptcha.execute(key, {action})        │      │
│                                              │      │
│  10. Google API generates token              │      │
│                                              │      │
│  11. Sets token in hidden field              │      │
│                                              │      │
│  12. Submits form to server                  │      │
└──────────────────────────────────────────────┘      │
     │                                                 │
     │ 13. POST with token                             │
     ├────────────────────────────────────────────────>│
     │                                                 │
     │                                                 │ 14. FormRequest
     │                                                 │     validation
     │                                                 ├──────────┐
     │                                                 │          │
     │                                                 │  new Captcha('action', 0.5)
     │                                                 │  ->validate(...)
     │                                                 │          │
     │                                                 │          │
     │                                                 │  15. CaptchaManager
     │                                                 │      ->verify($token, 'action')
     │                                                 │          │
     │                                                 │          │
     │                                    ┌────────────▼──────────▼────┐
     │                                    │  GoogleCaptcha::verify()   │
     │                                    │                            │
     │                                    │  16. POST to Google API    │
     │  ┌─────────────────────────────────┤  /recaptcha/api/siteverify│
     │  │                                 │                            │
     │  │ 17. Response:                   │  Parameters:               │
     │  │     {                           │  - secret                  │
     │  │       "success": true,          │  - response (token)        │
     │  │       "score": 0.9,             │  - remoteip                │
     │  │       "action": "action",       │                            │
     │  │       "hostname": "example.com" │                            │
     │  │     }                           │                            │
     │  │                                 └────────────┬───────────────┘
     │  │                                              │
     │  └─────────────────────────────────────────────>│
     │                                                 │
     │                                                 │ 18. validateResponse()
     │                                                 │     checks error-codes
     │                                                 │          │
     │                                                 │          │
     │                                                 │ 19. Check score >= 0.5
     │                                                 │          │
     │                                                 │<─────────┘
     │                                                 │
     │                                                 │ 20. Validation passed
     │                                                 │
     │                                                 │ 21. Process request
     │                                                 │
     │ 22. Success response                            │
     │<────────────────────────────────────────────────┤
     │                                                 │
```

### Verification Sequence Detail

**Phase 1: Initialization**

```
1. ServiceProvider boots
   └─> Registers CaptchaManager singleton
   └─> Publishes public/js/google-captcha.js
   └─> Loads helpers.php

2. Controller renders view
   └─> Calls captcha()->init()
       └─> GoogleCaptcha::init()
           └─> Returns: <script src="https://www.google.com/recaptcha/api.js?hl=en&render={key}"></script>

   └─> Calls captcha()->field('login')
       └─> GoogleCaptcha::field('login')
           └─> setAction('login')
           └─> setUniqueId() → base64_encode('captcha-response-login-')
           └─> Returns: <input type="hidden" name="captcha-response" data-key="{key}" data-action="login">

   └─> Calls captcha()->script('login')
       └─> GoogleCaptcha::script('login')
           └─> Returns: <script src="/js/google-captcha.js"></script>
               + <script>new GoogleCaptcha().init('{uniqueId}');</script>
```

**Phase 2: User Interaction**

```
3. Browser loads page
   └─> Google reCAPTCHA API initializes
   └─> GoogleCaptcha.init(uniqueId) executes
       └─> Finds hidden field by uniqueId
       └─> Extracts data-key and data-action
       └─> Finds parent form
       └─> Binds submit event handlers

4. User clicks submit button
   └─> JavaScript intercepts event.preventDefault()
   └─> Calls grecaptcha.execute(key, {action: 'login'})
       └─> Google generates token
       └─> Sets token in hidden field
       └─> Submits form
```

**Phase 3: Server Verification**

```
5. FormRequest validation triggered
   └─> Captcha rule instantiated: new Captcha('login', 0.5, true)
   └─> validate() called
       │
       ├─> Check $hasValidated flag (prevent duplicates)
       ├─> Check testing environment (skip if true)
       ├─> Check enabled flag
       │
       └─> app(CaptchaManager::class)->verify($token, 'login')
           │
           └─> Manager resolves 'google' driver
               └─> GoogleCaptcha::verify($token, 'login')
                   │
                   ├─> setAction('login')
                   ├─> validateRequest() (checks domain, endpoint, key, secret)
                   │
                   ├─> Http::asForm()->post('https://www.google.com/recaptcha/api/siteverify', [
                   │       'secret' => $secret,
                   │       'response' => $token,
                   │       'remoteip' => $request->getClientIp(),
                   │   ])
                   │
                   ├─> validateResponse($response)
                   │   └─> Check error-codes array
                   │       ├─> missing-input-secret → throw MissingSecretException
                   │       ├─> invalid-input-secret → throw InvalidSecretException
                   │       ├─> missing-input-response → throw MissingTokenException
                   │       ├─> invalid-input-response → throw InvalidTokenException
                   │       ├─> bad-request → throw InvalidRequestException
                   │       ├─> timeout-or-duplicate → throw TimeOutOrDuplicateException
                   │       └─> Action mismatch → throw InvalidActionException
                   │
                   └─> Return: [
                           'success' => true,
                           'score' => 0.9,
                           'action' => 'login',
                           'hostname' => 'example.com',
                       ]

6. Validation rule checks score
   └─> if ($result['score'] < 0.5)
       └─> $fail('captcha::validation.min_score')

7. If validation passes
   └─> Controller processes request
```

## Driver System

### Driver Resolution Flow

```
┌────────────────────────────────────────────────────────────────┐
│                    Driver Resolution                           │
└────────────────────────────────────────────────────────────────┘

app(CaptchaManager::class)
    │
    └─> Illuminate\Support\Manager resolves driver
        │
        ├─> getDefaultDriver()
        │   └─> Returns: config('captcha.driver') ?? 'google'
        │
        ├─> driver($name = null)
        │   │
        │   └─> Creates driver via: create{Driver}Driver()
        │       │
        │       ├─> createGoogleDriver()
        │       │   └─> container->make(GoogleCaptcha::class)
        │       │       └─> __construct(Request $request)
        │       │           └─> boot($request)
        │       │               └─> Chains fluent setters:
        │       │                   setRequest($request)
        │       │                   setReference('captcha-response')
        │       │                   setDomain(config('captcha.services.google.domain'))
        │       │                   setEndpoint(config('captcha.services.google.endpoint'))
        │       │                   setKey(config('captcha.services.google.key'))
        │       │                   setSecret(config('captcha.services.google.secret'))
        │       │                   setLocale(app()->getLocale())
        │       │
        │       └─> createCloudflareDriver()
        │           └─> container->make(CloudflareCaptcha::class)
        │               └─> Same boot process
        │
        └─> Returns: CaptchaContract instance
```

### Adding a New Driver

**Step 1: Create Service Class**

```php
// src/Services/HcaptchaCaptcha.php
namespace Bongo\Captcha\Services;

use Bongo\Captcha\Contracts\Captcha as CaptchaContract;

class HcaptchaCaptcha extends Captcha implements CaptchaContract
{
    protected ?string $name = 'hcaptcha'; // Must match config key

    public function verify(string $token, ?string $action = null): array
    {
        // Implementation
    }

    public function init(): string { /* ... */ }
    public function field(?string $action = null): string { /* ... */ }
    public function script(?string $action = null): string { /* ... */ }
}
```

**Step 2: Add Factory Method**

```php
// src/Managers/Captcha.php
public function createHcaptchaDriver(): CaptchaContract
{
    return $this->container->make(HcaptchaCaptcha::class);
}
```

**Step 3: Add Configuration**

```php
// src/Config/captcha.php
'services' => [
    'hcaptcha' => [
        'domain' => env('CAPTCHA_HCAPTCHA_DOMAIN', 'https://hcaptcha.com'),
        'endpoint' => env('CAPTCHA_HCAPTCHA_ENDPOINT', '/siteverify'),
        'key' => env('CAPTCHA_HCAPTCHA_KEY'),
        'secret' => env('CAPTCHA_HCAPTCHA_SECRET'),
    ],
],
```

**Step 4: Create JavaScript**

```javascript
// public/js/hcaptcha-captcha.js
function HcaptchaCaptcha() {
    this.init = function(uniqueId) {
        // Initialize hCaptcha
    }
}
```

## Validation Rule System

### Duplicate Prevention Mechanism

The validation rule uses a flag to prevent duplicate API calls:

```
┌──────────────────────────────────────────────────────────────┐
│            Why Duplicate Prevention is Needed                │
└──────────────────────────────────────────────────────────────┘

FormRequest lifecycle:
1. validate() called           → Calls Captcha::validate() → API call
2. validated() retrieves data  → Calls Captcha::validate() AGAIN
3. $request->input() called    → Could trigger validation AGAIN

Problem: Each validate() call would make an API request to Google,
         causing "timeout-or-duplicate" errors.

Solution: $hasValidated flag prevents subsequent calls.

┌──────────────────────────────────────────────────────────────┐
│                    Implementation                            │
└──────────────────────────────────────────────────────────────┘

class Captcha implements ValidationRule
{
    protected bool $hasValidated = false;

    public function validate(string $attribute, mixed $value, Closure $fail): void
    {
        // First check: Skip if already validated
        if ($this->hasValidated) {
            return; // No API call made
        }

        // Mark as validated BEFORE making API call
        $this->hasValidated = true;

        // Now safe to make API call
        $result = app(CaptchaManager::class)->verify($value, $this->action);

        // Subsequent calls to validate() will return immediately
    }
}
```

### Testing Environment Behaviour

```
┌──────────────────────────────────────────────────────────────┐
│              Testing Environment Detection                   │
└──────────────────────────────────────────────────────────────┘

if (app()->environment('testing')) {
    return; // Skip CAPTCHA verification
}

Why:
- CAPTCHA APIs require real HTTP requests
- Tests would need to mock external services
- Tests would be slower and more brittle
- Real CAPTCHA verification not needed in tests

Usage in tests:
$this->post('/login', [
    'email' => 'user@example.com',
    'password' => 'password',
    'captcha-response' => 'any-value', // Ignored in testing
]);

assertValid('captcha-response'); // Always passes in testing
```

### Validation Flow Chart

```
                    ┌──────────────────────┐
                    │ Captcha::validate()  │
                    └──────────┬───────────┘
                               │
                    ┌──────────▼───────────┐
                    │ $hasValidated?       │
                    └──────────┬───────────┘
                               │
                      ┌────────┴────────┐
                      │                 │
                   Yes│                 │No
                      │                 │
              ┌───────▼──────┐   ┌──────▼──────────┐
              │ return       │   │ $hasValidated   │
              │ (skip)       │   │ = true          │
              └──────────────┘   └──────┬──────────┘
                                        │
                               ┌────────▼───────────┐
                               │ environment ==     │
                               │ 'testing'?         │
                               └────────┬───────────┘
                                        │
                               ┌────────┴────────┐
                               │                 │
                            Yes│                 │No
                               │                 │
                       ┌───────▼──────┐   ┌──────▼──────────┐
                       │ return       │   │ $enabled?       │
                       │ (skip)       │   └──────┬──────────┘
                       └──────────────┘          │
                                        ┌────────┴────────┐
                                        │                 │
                                     No │                 │Yes
                                        │                 │
                                ┌───────▼──────┐   ┌──────▼──────────────┐
                                │ return       │   │ CaptchaManager      │
                                │ (skip)       │   │ ->verify()          │
                                └──────────────┘   └──────┬──────────────┘
                                                          │
                                                 ┌────────▼────────┐
                                                 │ $result         │
                                                 │ ['success']?    │
                                                 └────────┬────────┘
                                                          │
                                                 ┌────────┴────────┐
                                                 │                 │
                                              No │                 │Yes
                                                 │                 │
                                         ┌───────▼──────┐   ┌──────▼──────────┐
                                         │ $fail(       │   │ $result         │
                                         │   'no_score')│   │ ['score'] >=    │
                                         └──────────────┘   │ $minScore?      │
                                                            └──────┬──────────┘
                                                                   │
                                                          ┌────────┴────────┐
                                                          │                 │
                                                       No │                 │Yes
                                                          │                 │
                                                  ┌───────▼──────┐   ┌──────▼──────┐
                                                  │ $fail(       │   │ Validation  │
                                                  │ 'min_score') │   │ passes      │
                                                  └──────────────┘   └─────────────┘
```

## Frontend Integration

### Google reCAPTCHA JavaScript Flow

```
┌──────────────────────────────────────────────────────────────────┐
│                  GoogleCaptcha JavaScript                        │
└──────────────────────────────────────────────────────────────────┘

window.addEventListener('load', () => {
    new GoogleCaptcha().init(uniqueId);
});

    │
    └─> init(uniqueId)
        │
        ├─> checkForUniqueId(uniqueId)
        │   └─> Validates uniqueId parameter exists
        │
        ├─> checkForRecaptchaScript()
        │   └─> Validates window.grecaptcha is loaded
        │
        ├─> setBrowser()
        │   └─> Stores navigator.userAgent
        │
        ├─> setField(uniqueId)
        │   └─> Finds: document.getElementById(uniqueId)
        │       └─> Validates field exists
        │
        ├─> setKey()
        │   └─> Extracts: field.getAttribute('data-key')
        │       └─> Validates key exists
        │
        ├─> setAction()
        │   └─> Extracts: field.getAttribute('data-action')
        │       └─> Optional, can be null
        │
        ├─> setForm()
        │   └─> Finds: field.closest('form')
        │       └─> Validates form exists
        │
        └─> setSubmitEventHandler()
            │
            ├─> Bind submit buttons (Firefox compatibility)
            │   └─> form.querySelectorAll('button[type="submit"], input[type="submit"]')
            │       └─> button.addEventListener('click', (e) => {
            │               e.preventDefault();
            │               this.generateToken();
            │           })
            │
            └─> Bind form submit (other browsers)
                └─> form.addEventListener('submit', (e) => {
                        e.preventDefault();
                        this.generateToken();
                    })

generateToken()
    │
    └─> grecaptcha.ready(() => {
            grecaptcha.execute(key, { action: action })
                .then((token) => {
                    field.value = token;
                    form.submit();
                })
                .catch((error) => {
                    console.error('Recaptcha error:', error);
                });
        })
```

### Multiple Forms Support

```
┌──────────────────────────────────────────────────────────────────┐
│              Multiple CAPTCHAs on Same Page                      │
└──────────────────────────────────────────────────────────────────┘

Page HTML:
<form id="login">
    {!! captcha()->field('login') !!}
    <!-- Generates: <input id="Y2FwdGNoYS1yZXNwb25zZS1sb2dpbi0=" ... data-action="login"> -->
</form>

<form id="register">
    {!! captcha()->field('register') !!}
    <!-- Generates: <input id="Y2FwdGNoYS1yZXNwb25zZS1yZWdpc3Rlci0=" ... data-action="register"> -->
</form>

{!! captcha()->script('login') !!}
<!-- new GoogleCaptcha().init('Y2FwdGNoYS1yZXNwb25zZS1sb2dpbi0=') -->

{!! captcha()->script('register') !!}
<!-- new GoogleCaptcha().init('Y2FwdGNoYS1yZXNwb25zZS1yZWdpc3Rlci0=') -->

Result:
- Two independent GoogleCaptcha instances
- Each binds to its own form
- Each has unique action ('login' vs 'register')
- Both use same Google API key
- Both submit to same field name ('captcha-response')
```

## Exception Hierarchy

```
Exception
│
└─> Bongo\Captcha\Exceptions\CaptchaException
    │
    │   Static Factory Method:
    │   + invalidProperty(string $class, string $property): self
    │
    ├─> InvalidActionException
    │   Thrown when: Action in response doesn't match expected action
    │   Google error: (custom validation)
    │
    ├─> InvalidRequestException
    │   Thrown when: Request to API is malformed
    │   Google error: 'bad-request'
    │
    ├─> InvalidResponseException
    │   Thrown when: API returns empty/null response
    │   Google error: (no response)
    │
    ├─> InvalidSecretException
    │   Thrown when: Secret key is invalid
    │   Google error: 'invalid-input-secret'
    │
    ├─> InvalidTokenException
    │   Thrown when: Token is invalid/expired
    │   Google error: 'invalid-input-response'
    │
    ├─> MissingSecretException
    │   Thrown when: Secret key not provided
    │   Google error: 'missing-input-secret'
    │
    ├─> MissingTokenException
    │   Thrown when: Token not provided in request
    │   Google error: 'missing-input-response'
    │
    └─> TimeOutOrDuplicateException
        Thrown when: Token already used or timeout
        Google error: 'timeout-or-duplicate'
```

### Exception Handling Strategy

```php
// src/Services/GoogleCaptcha.php

public function verify(string $token, ?string $action = null): array
{
    try {
        // Make API request
        $response = Http::asForm()->post(...)->json();

        // Validate response
        $this->validateResponse($response);

        // Return success
        return ['success' => true, ...];

    } catch (CaptchaException $e) {
        // Log error with context
        Log::error($e->getMessage(), [
            'captcha' => $this->name,
            'token' => $token,
            'action' => $this->action,
            'response' => isset($response) ? json_encode($response) : null,
        ]);

        // Return failure (doesn't re-throw)
        return [
            'success' => false,
            'error' => $e->getMessage(),
        ];
    }
}
```

**Why Not Re-throw?**

Exceptions are caught and logged, but not re-thrown. This allows:
1. Graceful degradation (form validation fails instead of 500 error)
2. User-friendly error messages
3. Logging for debugging
4. Consistent return type (`array`)

## Configuration System

### Configuration Structure

```php
// config/captcha.php
return [
    // Default driver selection
    'driver' => env('CAPTCHA_DRIVER', 'google'),

    // Driver-specific configuration
    'services' => [
        'google' => [
            'domain' => env('CAPTCHA_GOOGLE_DOMAIN', 'https://www.google.com'),
            'endpoint' => env('CAPTCHA_GOOGLE_ENDPOINT', '/recaptcha'),
            'key' => env('CAPTCHA_GOOGLE_KEY'),
            'secret' => env('CAPTCHA_GOOGLE_SECRET'),
        ],
        'cloudflare' => [
            'domain' => env('CAPTCHA_CLOUDFLARE_DOMAIN', 'https://www.cloudflare.com'),
            'endpoint' => env('CAPTCHA_CLOUDFLARE_ENDPOINT', '/captcha'),
            'key' => env('CAPTCHA_CLOUDFLARE_KEY'),
            'secret' => env('CAPTCHA_CLOUDFLARE_SECRET'),
        ],
    ],
];
```

### Configuration Access Pattern

```php
// src/Services/Captcha.php

public function setDomain(?string $domain = null): self
{
    // Priority: 1. Parameter, 2. Config, 3. Trim trailing slashes
    $this->domain = trim(
        $domain ?? config("captcha.services.{$this->name}.domain"),
        '/'
    );

    return $this;
}

// Same pattern for all configuration values:
// - setEndpoint()
// - setKey()
// - setSecret()
```

This allows:
1. **Runtime override**: `$captcha->setKey('custom-key')`
2. **Config default**: Falls back to `config('captcha.services.google.key')`
3. **Environment**: Config reads from `.env` via `env('CAPTCHA_GOOGLE_KEY')`

## Extension Points

### 1. Adding New CAPTCHA Providers

**Interface Contract:**

```php
interface Captcha
{
    // Required methods to implement
    public function verify(string $token, ?string $action = null): array;
    public function init(): string;
    public function field(?string $action = null): string;
    public function script(?string $action = null): string;

    // Inherited from abstract Captcha class
    // (already implemented, but can be overridden)
}
```

**Checklist:**
- [ ] Create service class extending `Services\Captcha`
- [ ] Implement `Contracts\Captcha` interface
- [ ] Set `protected ?string $name` property
- [ ] Add factory method to `Managers\Captcha`
- [ ] Add configuration to `Config/captcha.php`
- [ ] Create frontend JavaScript
- [ ] Add environment variables to documentation
- [ ] Add tests

### 2. Custom Validation Rules

You can create custom validation rules that use the CAPTCHA service:

```php
use Bongo\Captcha\Managers\Captcha as CaptchaManager;
use Closure;
use Illuminate\Contracts\Validation\ValidationRule;

class StrictCaptcha implements ValidationRule
{
    public function __construct(
        protected string $action,
        protected float $minScore = 0.7, // Stricter than default
    ) {}

    public function validate(string $attribute, mixed $value, Closure $fail): void
    {
        $result = app(CaptchaManager::class)->verify($value, $this->action);

        if (!$result['success']) {
            $fail('CAPTCHA verification failed.');
            return;
        }

        // Additional validation: Check hostname matches
        if ($result['hostname'] !== config('app.domain')) {
            $fail('CAPTCHA hostname mismatch.');
            return;
        }

        // Check score
        if ($result['score'] < $this->minScore) {
            $fail("CAPTCHA score too low: {$result['score']}");
        }
    }
}
```

### 3. Custom Exception Handling

Create custom exception handlers:

```php
namespace App\Exceptions;

use Bongo\Captcha\Exceptions\CaptchaException;
use Illuminate\Foundation\Exceptions\Handler as ExceptionHandler;

class Handler extends ExceptionHandler
{
    public function register(): void
    {
        $this->renderable(function (CaptchaException $e, $request) {
            return response()->json([
                'error' => 'CAPTCHA verification failed',
                'message' => $e->getMessage(),
            ], 422);
        });
    }
}
```

### 4. Middleware for Global CAPTCHA

Create middleware to protect specific routes:

```php
namespace App\Http\Middleware;

use Bongo\Captcha\Facades\Captcha;
use Closure;
use Illuminate\Http\Request;

class VerifyCaptcha
{
    public function handle(Request $request, Closure $next, string $action, float $minScore = 0.5)
    {
        $token = $request->input('captcha-response');

        if (empty($token)) {
            return back()->withErrors(['captcha' => 'CAPTCHA is required.']);
        }

        $result = Captcha::verify($token, $action);

        if (!$result['success'] || ($result['score'] ?? 0) < $minScore) {
            return back()->withErrors(['captcha' => 'CAPTCHA verification failed.']);
        }

        return $next($request);
    }
}

// Route usage:
Route::post('/contact', [ContactController::class, 'store'])
    ->middleware('captcha:contact_form,0.7');
```

### 5. Event Hooks

Add hooks for verification events:

```php
// Define events
namespace App\Events;

class CaptchaVerified
{
    public function __construct(
        public string $action,
        public float $score,
        public bool $success,
    ) {}
}

// Dispatch in custom driver
use App\Events\CaptchaVerified;

public function verify(string $token, ?string $action = null): array
{
    $result = parent::verify($token, $action);

    event(new CaptchaVerified(
        action: $action ?? '',
        score: $result['score'] ?? 0,
        success: $result['success'],
    ));

    return $result;
}
```

### 6. Custom Field Rendering

Override field/script generation:

```php
class CustomGoogleCaptcha extends GoogleCaptcha
{
    public function field(?string $action = null): string
    {
        $this->setAction($action);

        // Custom HTML structure
        return view('captcha.field', [
            'uniqueId' => $this->uniqueId,
            'reference' => $this->reference,
            'key' => $this->key,
            'action' => $this->action,
        ])->render();
    }
}
```

## Performance Considerations

### 1. Singleton Manager

`CaptchaManager` is registered as singleton:

```php
$this->app->singleton(CaptchaManager::class, function ($app) {
    return new CaptchaManager($app);
});
```

This ensures:
- Single instance per request
- Driver instances are cached by Laravel Manager
- Configuration loaded once

### 2. Lazy Driver Resolution

Drivers are created only when needed:

```php
// This does NOT create GoogleCaptcha instance yet
$manager = app(CaptchaManager::class);

// Driver created on first access
$captcha = $manager->driver('google'); // NOW GoogleCaptcha is instantiated
```

### 3. Validation Duplicate Prevention

The `$hasValidated` flag prevents multiple API calls:

```php
// First call: Makes API request
$validator->validate(); // API call to Google

// Second call: Skips API request
$validator->validated(); // No API call (flag is true)
```

Without this, each form submission could make 2-3 API calls.

### 4. Asset Publishing

JavaScript assets are published once during installation:

```bash
php artisan vendor:publish --tag=captcha:assets
```

This copies `public/js/*.js` to application's `public/js/` directory, avoiding package directory lookups on each request.

## Security Considerations

### 1. Secret Key Protection

- Never expose `CAPTCHA_*_SECRET` in frontend
- Stored in `.env` file (excluded from version control)
- Only used in server-side verification

### 2. Token Single-Use

Google reCAPTCHA tokens are single-use. The `timeout-or-duplicate` error prevents replay attacks.

### 3. Score Thresholds

Adjust score thresholds based on action sensitivity:

```php
// Public forms: Lower threshold
new Captcha('newsletter', 0.3)

// Sensitive actions: Higher threshold
new Captcha('password_reset', 0.8)

// High-risk transactions: Very high threshold
new Captcha('payment', 0.9)
```

### 4. Action Validation

Always verify the action matches:

```php
protected function validateResponse(?array $response = null): void
{
    if ($this->action && (
        !isset($response['action']) ||
        $this->action !== $response['action']
    )) {
        throw new InvalidActionException(...);
    }
}
```

This prevents token reuse across different actions.

### 5. IP Address Logging

Client IP is sent to verification API:

```php
'remoteip' => $this->request->getClientIp()
```

This helps Google detect suspicious patterns.

## Testing Strategy

### 1. Automatic Skip in Testing

```php
if (app()->environment('testing')) {
    return; // Skip verification
}
```

### 2. Test Case Base Class

```php
namespace Bongo\Captcha\Tests;

use Orchestra\Testbench\TestCase as Orchestra;

class TestCase extends Orchestra
{
    protected function getPackageProviders($app): array
    {
        return [CaptchaServiceProvider::class];
    }
}
```

### 3. Example Test Pattern

```php
use Bongo\Captcha\Tests\TestCase;

class CaptchaTest extends TestCase
{
    public function test_form_submission_with_captcha()
    {
        // CAPTCHA automatically skipped in testing environment
        $response = $this->post('/contact', [
            'name' => 'John Doe',
            'email' => 'john@example.com',
            'captcha-response' => 'test-token',
        ]);

        $response->assertRedirect('/contact/success');
    }
}
```

## Troubleshooting Guide

### Common Issues

**Issue 1: "timeout-or-duplicate" Error**

```
Cause: Token used multiple times
Solution: Check $hasValidated flag is working
```

**Issue 2: Form Doesn't Submit**

```
Cause: JavaScript errors preventing token generation
Debug:
1. Check browser console for errors
2. Verify grecaptcha is loaded: window.grecaptcha
3. Check hidden field has value before submit
```

**Issue 3: "missing-input-secret" Error**

```
Cause: CAPTCHA_GOOGLE_SECRET not set in .env
Solution: Add environment variable and clear config cache
```

**Issue 4: Score Always 0.0**

```
Cause: Wrong reCAPTCHA version (using v2 keys with v3 code)
Solution: Generate new reCAPTCHA v3 keys
```

**Issue 5: Multiple Forms Not Working**

```
Cause: Duplicate uniqueId values
Solution: Use different actions: field('form1'), field('form2')
```
