# Bongo Captcha - Claude Code Guide

## Overview

Laravel package providing CAPTCHA verification services for Bongo framework applications. Supports Google reCAPTCHA v3 and Cloudflare Turnstile through a driver-based architecture.

**Package:** `bongo/captcha`
**Namespace:** `Bongo\Captcha`
**Extends:** `Bongo\Framework\Providers\AbstractServiceProvider`

## Related Documentation

- **README.md** - Installation and usage guide
- **ARCHITECTURE.md** - Detailed architecture diagrams and flows
- **.cursorrules** - Cursor AI development guidelines
- **.github/copilot-instructions.md** - GitHub Copilot instructions

## Quick Start

### Installation

```bash
composer require bongo/captcha
```

### Configuration

```bash
# Publish config
php artisan vendor:publish --provider="Bongo\Captcha\CaptchaServiceProvider" --tag=config

# Publish assets
php artisan vendor:publish --tag=captcha:assets
```

### Environment Variables

```env
CAPTCHA_DRIVER=google

CAPTCHA_GOOGLE_KEY=your-site-key
CAPTCHA_GOOGLE_SECRET=your-secret-key

CAPTCHA_CLOUDFLARE_KEY=your-site-key
CAPTCHA_CLOUDFLARE_SECRET=your-secret-key
```

### Usage in Blade

```blade
{{-- In <head> or before </body> --}}
{!! captcha()->init() !!}

{{-- Inside form --}}
{!! captcha()->field('login') !!}

{{-- Before </body> --}}
{!! captcha()->script('login') !!}
```

### Validation Rule

```php
use Bongo\Captcha\Rules\Captcha;

public function rules(): array
{
    return [
        'captcha-response' => [
            'required',
            new Captcha(
                action: 'login',
                minScore: 0.5,
                enabled: true
            ),
        ],
    ];
}
```

## Commands

From `composer.json`:

```bash
# Testing
composer test                # Run PHPUnit tests
composer test:parallel       # Run tests in parallel
composer test:coverage       # Run tests with coverage report

# Code Quality
composer analyse             # Run PHPStan static analysis
composer format              # Run Pint (fix code style)

# Development
composer clear               # Clear Testbench skeleton
composer prepare             # Discover packages
composer build               # Build Testbench workbench
composer start               # Start Testbench server
```

## Architecture Quick Reference

### Service Provider Pattern

```php
class CaptchaServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'captcha';

    // Auto-loaded by AbstractServiceProvider:
    // - Config: src/Config/captcha.php
    // - Translations: src/Translations/en/*
    // - No routes (this package doesn't define routes)
}
```

### Driver Manager Pattern

```php
// Resolve default driver (from config)
$captcha = app(CaptchaManager::class);

// Resolve specific driver
$captcha = app(CaptchaManager::class)->driver('google');
$captcha = app(CaptchaManager::class)->driver('cloudflare');

// Using facade
use Bongo\Captcha\Facades\Captcha;
$result = Captcha::verify($token, 'login');

// Using helper
$result = captcha()->verify($token, 'login');
```

### Interface Contract

```php
interface Captcha
{
    // Verify token with CAPTCHA provider
    public function verify(string $token, ?string $action = null): array;

    // Get site key
    public function key(): string;

    // Generate <script> tag to load CAPTCHA API
    public function init(): string;

    // Generate hidden field HTML
    public function field(?string $action = null): string;

    // Generate initialization JavaScript
    public function script(?string $action = null): string;

    // Get field reference name (default: 'captcha-response')
    public function reference(): string;

    // Fluent setters (return self)
    public function setRequest(Request $request): AbstractCaptcha;
    public function setReference(?string $reference = null): AbstractCaptcha;
    public function setAction(?string $action = null): AbstractCaptcha;
    public function setDomain(?string $domain = null): AbstractCaptcha;
    public function setEndpoint(?string $endpoint = null): AbstractCaptcha;
    public function setKey(?string $key = null): AbstractCaptcha;
    public function setSecret(?string $secret = null): AbstractCaptcha;
    public function setLocale(?string $locale = null): AbstractCaptcha;
}
```

### Verification Response

```php
$result = captcha()->verify($token, 'login');

// Success response:
[
    'success' => true,
    'score' => 0.9,        // 0.0 (bot) to 1.0 (human)
    'action' => 'login',
    'hostname' => 'example.com',
]

// Failure response:
[
    'success' => false,
    'error' => 'Error message',
]
```

## Key Files Reference

| File Path | Purpose | Key Details |
|-----------|---------|-------------|
| `src/CaptchaServiceProvider.php` | Service provider | Registers CaptchaManager singleton, publishes assets |
| `src/Managers/Captcha.php` | Driver manager | Extends `Illuminate\Support\Manager`, resolves drivers |
| `src/Services/Captcha.php` | Abstract base service | Common functionality, fluent setters, validation |
| `src/Services/GoogleCaptcha.php` | Google reCAPTCHA v3 | Full implementation with API verification |
| `src/Services/CloudflareCaptcha.php` | Cloudflare Turnstile | Placeholder (returns success without verification) |
| `src/Contracts/Captcha.php` | Interface | Defines contract for all CAPTCHA drivers |
| `src/Rules/Captcha.php` | Validation rule | Laravel ValidationRule with duplicate prevention |
| `src/Facades/Captcha.php` | Facade | Facade accessor for CaptchaManager |
| `src/helpers.php` | Global helper | `captcha()` function returns CaptchaManager |
| `src/Config/captcha.php` | Configuration | Driver selection and service credentials |
| `src/Translations/en/messages.php` | Exception messages | Translatable exception messages |
| `src/Translations/en/validation.php` | Validation messages | Translatable validation error messages |
| `src/Exceptions/CaptchaException.php` | Base exception | Factory method: `invalidProperty()` |
| `src/Exceptions/*.php` | Specific exceptions | 8 exception types for different failures |
| `public/js/google-captcha.js` | Frontend JavaScript | Google reCAPTCHA integration, form submit handling |
| `public/js/cloudflare-captcha.js` | Frontend JavaScript | Cloudflare Turnstile (placeholder) |

## Class Hierarchy

```
Service Provider:
  Bongo\Framework\Providers\AbstractServiceProvider
  └── CaptchaServiceProvider

Manager:
  Illuminate\Support\Manager
  └── Bongo\Captcha\Managers\Captcha

Services:
  Bongo\Captcha\Services\Captcha (abstract)
  ├── GoogleCaptcha (implements Contracts\Captcha)
  └── CloudflareCaptcha (implements Contracts\Captcha)

Validation:
  Illuminate\Contracts\Validation\ValidationRule
  └── Bongo\Captcha\Rules\Captcha

Facades:
  Illuminate\Support\Facades\Facade
  └── Bongo\Captcha\Facades\Captcha

Exceptions:
  Exception
  └── Bongo\Captcha\Exceptions\CaptchaException
      ├── InvalidActionException
      ├── InvalidRequestException
      ├── InvalidResponseException
      ├── InvalidSecretException
      ├── InvalidTokenException
      ├── MissingSecretException
      ├── MissingTokenException
      └── TimeOutOrDuplicateException
```

## Code Style Summary

From `pint.json`:

```json
{
    "preset": "laravel",
    "rules": {
        "new_with_parentheses": false,
        "nullable_type_declaration_for_default_null_value": false,
        "declare_strict_types": true,
        "declare_parentheses": true
    }
}
```

**Key Conventions:**
- All files use `declare(strict_types=1);`
- Return types always declared: `public function key(): string`
- Fluent interfaces return `self` or `AbstractCaptcha`
- Exception messages use translation keys: `trans('captcha::messages.invalid_token')`
- Properties use camelCase with visibility keywords
- Methods use camelCase

## Common Patterns

### 1. Using CAPTCHA in Forms

```php
// Controller
public function create()
{
    return view('contact.form');
}

// Blade (contact/form.blade.php)
<form method="POST" action="{{ route('contact.store') }}">
    @csrf

    {!! captcha()->init() !!}

    <input type="text" name="name" required>
    <input type="email" name="email" required>

    {!! captcha()->field('contact') !!}

    <button type="submit">Submit</button>
</form>

{!! captcha()->script('contact') !!}

// Form Request
use Bongo\Captcha\Rules\Captcha;

class ContactRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'name' => 'required|string',
            'email' => 'required|email',
            'captcha-response' => [
                'required',
                new Captcha('contact', 0.5, true),
            ],
        ];
    }
}

// Controller
public function store(ContactRequest $request)
{
    // CAPTCHA already validated
    // Process form...
}
```

### 2. Manual Verification

```php
use Bongo\Captcha\Facades\Captcha;

$result = Captcha::verify($request->input('captcha-response'), 'login');

if ($result['success'] && $result['score'] >= 0.5) {
    // CAPTCHA passed
} else {
    // CAPTCHA failed
    return back()->withErrors(['captcha' => $result['error'] ?? 'Verification failed.']);
}
```

### 3. Switching Drivers

```php
// Use Google (default)
$result = captcha()->verify($token, 'login');

// Use Cloudflare
$result = captcha()->driver('cloudflare')->verify($token, 'login');

// Change default driver
// .env: CAPTCHA_DRIVER=cloudflare
```

### 4. Multiple Forms on Same Page

```php
{{-- Form 1: Login --}}
<form id="login">
    {!! captcha()->field('login') !!}
    <button type="submit">Login</button>
</form>

{{-- Form 2: Register --}}
<form id="register">
    {!! captcha()->field('register') !!}
    <button type="submit">Register</button>
</form>

{{-- Initialize both --}}
{!! captcha()->script('login') !!}
{!! captcha()->script('register') !!}

{{-- Each generates unique ID based on action --}}
```

### 5. Conditional CAPTCHA

```php
public function rules(): array
{
    $enabled = !auth()->check(); // Only for guests

    return [
        'captcha-response' => [
            'required_if:guest,true',
            new Captcha(
                action: 'login',
                minScore: 0.5,
                enabled: $enabled
            ),
        ],
    ];
}
```

### 6. Custom Score Thresholds

```php
// Low security (newsletter signup)
new Captcha('newsletter', 0.3)

// Medium security (contact form)
new Captcha('contact', 0.5)

// High security (password reset)
new Captcha('password_reset', 0.7)

// Very high security (payment)
new Captcha('payment', 0.9)
```

## Testing

### Automatic Skip in Tests

CAPTCHA verification is automatically skipped when `APP_ENV=testing`:

```php
// tests/Feature/LoginTest.php
public function test_login_form()
{
    $response = $this->post('/login', [
        'email' => 'user@example.com',
        'password' => 'password',
        'captcha-response' => 'any-value', // Ignored in testing
    ]);

    $response->assertRedirect('/dashboard');
}
```

### Test Case Base Class

```php
namespace Bongo\Captcha\Tests;

use Bongo\Captcha\CaptchaServiceProvider;
use Orchestra\Testbench\TestCase as Orchestra;

class TestCase extends Orchestra
{
    protected function getPackageProviders($app): array
    {
        return [CaptchaServiceProvider::class];
    }
}
```

## Important Implementation Details

### 1. Duplicate Validation Prevention

The `Captcha` validation rule uses a flag to prevent duplicate API calls:

```php
protected bool $hasValidated = false;

public function validate(string $attribute, mixed $value, Closure $fail): void
{
    if ($this->hasValidated) return; // Skip duplicate validation
    $this->hasValidated = true;

    // Make API call
}
```

**Why?** Laravel calls `validate()` multiple times (on `validate()`, `validated()`, etc.), which would trigger multiple API requests without this flag.

### 2. Unique ID Generation

Each CAPTCHA instance gets a unique ID for multiple forms support:

```php
public function makeUniqueId(): string
{
    return base64_encode("{$this->reference}-{$this->action}-");
}

// Examples:
// captcha()->field('login')    → Y2FwdGNoYS1yZXNwb25zZS1sb2dpbi0=
// captcha()->field('register') → Y2FwdGNoYS1yZXNwb25zZS1yZWdpc3Rlci0=
```

### 3. Firefox Compatibility

JavaScript binds both button clicks and form submit events:

```javascript
// Button clicks (Firefox)
submitButtons.forEach((button) => {
    button.addEventListener('click', (event) => {
        event.preventDefault();
        this.generateToken();
    });
});

// Form submit (other browsers)
form.addEventListener('submit', (event) => {
    event.preventDefault();
    this.generateToken();
});
```

### 4. Error Code Mapping

Google reCAPTCHA error codes map to specific exceptions:

| Error Code | Exception | Meaning |
|------------|-----------|---------|
| `missing-input-secret` | `MissingSecretException` | Secret not provided |
| `invalid-input-secret` | `InvalidSecretException` | Secret is invalid |
| `missing-input-response` | `MissingTokenException` | Token not provided |
| `invalid-input-response` | `InvalidTokenException` | Token is invalid/expired |
| `bad-request` | `InvalidRequestException` | Malformed request |
| `timeout-or-duplicate` | `TimeOutOrDuplicateException` | Token already used |
| (custom) | `InvalidActionException` | Action mismatch |
| (no response) | `InvalidResponseException` | Empty response |

### 5. Graceful Failure

Exceptions are caught and logged but not re-thrown:

```php
try {
    // Verify with API
} catch (CaptchaException $e) {
    Log::error($e->getMessage(), [...]);

    return ['success' => false, 'error' => $e->getMessage()];
}
```

This allows graceful degradation (validation fails instead of 500 error).

## Adding a New CAPTCHA Provider

### Step-by-Step Guide

**1. Create Service Class**

```php
// src/Services/HcaptchaCaptcha.php
<?php

declare(strict_types=1);

namespace Bongo\Captcha\Services;

use Bongo\Captcha\Contracts\Captcha as CaptchaContract;
use Illuminate\Support\Facades\Http;

class HcaptchaCaptcha extends Captcha implements CaptchaContract
{
    protected ?string $name = 'hcaptcha';

    public function verify(string $token, ?string $action = null): array
    {
        $this->setAction($action);
        $this->validateRequest();

        $response = Http::asForm()
            ->post($this->getVerifyUrl(), [
                'secret' => $this->secret,
                'response' => $token,
                'remoteip' => $this->request->getClientIp(),
            ])
            ->json();

        return [
            'success' => $response['success'] ?? false,
            'score' => $response['score'] ?? 1.0,
            'action' => $action,
            'hostname' => $response['hostname'] ?? null,
        ];
    }

    public function getVerifyUrl(): string
    {
        return "{$this->getBaseUrl()}/siteverify";
    }

    public function init(): string
    {
        $apiJs = "{$this->getBaseUrl()}/api.js?hl={$this->locale}";
        return "<script src=\"{$apiJs}\" async defer></script>";
    }

    public function field(?string $action = null): string
    {
        $this->setAction($action);
        return "<input id=\"{$this->uniqueId}\" type=\"hidden\" name=\"{$this->reference}\" data-key=\"{$this->key}\" data-action=\"{$this->action}\">";
    }

    public function script(?string $action = null): string
    {
        $this->setAction($action);
        $captchaJs = asset('js/hcaptcha-captcha.js');
        return "<script src=\"{$captchaJs}\"></script><script>new HcaptchaCaptcha().init('{$this->uniqueId}');</script>";
    }
}
```

**2. Add Factory Method**

```php
// src/Managers/Captcha.php
public function createHcaptchaDriver(): CaptchaContract
{
    return $this->container->make(HcaptchaCaptcha::class);
}
```

**3. Add Configuration**

```php
// src/Config/captcha.php
'services' => [
    'hcaptcha' => [
        'domain' => env('CAPTCHA_HCAPTCHA_DOMAIN', 'https://hcaptcha.com'),
        'endpoint' => env('CAPTCHA_HCAPTCHA_ENDPOINT', '/api'),
        'key' => env('CAPTCHA_HCAPTCHA_KEY'),
        'secret' => env('CAPTCHA_HCAPTCHA_SECRET'),
    ],
],
```

**4. Create Frontend JavaScript**

```javascript
// public/js/hcaptcha-captcha.js
function HcaptchaCaptcha() {
    this.init = function(uniqueId) {
        // Initialize hCaptcha
    }
}
```

**5. Add Environment Variables**

```env
CAPTCHA_HCAPTCHA_KEY=your-site-key
CAPTCHA_HCAPTCHA_SECRET=your-secret-key
```

**6. Use New Driver**

```env
CAPTCHA_DRIVER=hcaptcha
```

Or runtime:

```php
captcha()->driver('hcaptcha')->verify($token, 'login');
```

## Troubleshooting

### Common Issues

**1. "timeout-or-duplicate" Error**

```
Cause: Token used multiple times
Fix: Ensure $hasValidated flag in validation rule
```

**2. JavaScript Not Loading**

```
Cause: Assets not published
Fix: php artisan vendor:publish --tag=captcha:assets
```

**3. "missing-input-secret" Error**

```
Cause: CAPTCHA_GOOGLE_SECRET not set
Fix: Add to .env and clear config cache
     php artisan config:clear
```

**4. Form Won't Submit**

```
Cause: JavaScript errors
Debug:
1. Check browser console for errors
2. Verify window.grecaptcha is defined
3. Check hidden field has value before submit
4. Ensure form has hidden captcha field
```

**5. Score Always 0.0**

```
Cause: Using reCAPTCHA v2 keys with v3 code
Fix: Generate new reCAPTCHA v3 keys at
     https://www.google.com/recaptcha/admin
```

**6. Multiple Forms Not Working**

```
Cause: Same uniqueId for both forms
Fix: Use different actions:
     captcha()->field('form1')
     captcha()->field('form2')
```

## Configuration Reference

### Environment Variables

```env
# Driver Selection
CAPTCHA_DRIVER=google

# Google reCAPTCHA v3
CAPTCHA_GOOGLE_DOMAIN=https://www.google.com
CAPTCHA_GOOGLE_ENDPOINT=/recaptcha
CAPTCHA_GOOGLE_KEY=6LdXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
CAPTCHA_GOOGLE_SECRET=6LdXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

# Cloudflare Turnstile
CAPTCHA_CLOUDFLARE_DOMAIN=https://www.cloudflare.com
CAPTCHA_CLOUDFLARE_ENDPOINT=/captcha
CAPTCHA_CLOUDFLARE_KEY=0x4XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
CAPTCHA_CLOUDFLARE_SECRET=0x4XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
```

### Config File Structure

```php
// config/captcha.php
return [
    'driver' => env('CAPTCHA_DRIVER', 'google'),

    'services' => [
        'google' => [
            'domain' => env('CAPTCHA_GOOGLE_DOMAIN', 'https://www.google.com'),
            'endpoint' => env('CAPTCHA_GOOGLE_ENDPOINT', '/recaptcha'),
            'key' => env('CAPTCHA_GOOGLE_KEY'),
            'secret' => env('CAPTCHA_GOOGLE_SECRET'),
        ],
        'cloudflare' => [
            'domain' => env('CAPTCHA_CLOUDFLARE_DOMAIN', 'https://www.cloudflare.com'),
            'endpoint' => env('CAPTCHA_CLOUDFLARE_ENDPOINT', '/captcha'),
            'key' => env('CAPTCHA_CLOUDFLARE_KEY'),
            'secret' => env('CAPTCHA_CLOUDFLARE_SECRET'),
        ],
    ],
];
```

## API Reference

### CaptchaManager Methods

```php
// Get default driver
$captcha = app(CaptchaManager::class);

// Get specific driver
$captcha = app(CaptchaManager::class)->driver('google');

// Verify token
$result = $captcha->verify(string $token, ?string $action = null): array;

// Get site key
$key = $captcha->key(): string;

// Generate init script
$script = $captcha->init(): string;

// Generate hidden field
$field = $captcha->field(?string $action = null): string;

// Generate initialization script
$script = $captcha->script(?string $action = null): string;

// Get field name
$name = $captcha->reference(): string;
```

### Captcha Rule Constructor

```php
new Captcha(
    string $action,         // Action identifier (e.g., 'login', 'register')
    float $minScore = 0.5,  // Minimum score threshold (0.0 - 1.0)
    bool $enabled = true    // Enable/disable verification
)
```

### Verification Response Structure

```php
// Success
[
    'success' => true,
    'score' => 0.9,        // Float: 0.0 (bot) to 1.0 (human)
    'action' => 'login',   // String: Action identifier
    'hostname' => 'example.com', // String: Request hostname
]

// Failure
[
    'success' => false,
    'error' => 'Error message', // String: Error description
]
```

## Dependencies

**Runtime:**
- PHP 8.2+
- Laravel 10.x or 11.x
- bongo/framework ^3.0

**Development:**
- laravel/pint ^1.14
- larastan/larastan ^2.9
- orchestra/testbench ^8.22 or ^9.0
- phpstan/phpstan ^1.3

## Further Reading

For detailed information, see:

- **ARCHITECTURE.md** - Complete architecture documentation with diagrams
- **README.md** - Installation and usage guide
- **.cursorrules** - Development guidelines and patterns
- **.github/copilot-instructions.md** - Code examples and templates
