<?php

declare(strict_types=1);

namespace Bongo\Captcha\Rules;

use Bongo\Captcha\Managers\Captcha as CaptchaManager;
use Closure;
use Illuminate\Contracts\Validation\ValidationRule;

class Captcha implements ValidationRule
{
    protected bool $hasValidated = false;

    protected string $action;

    protected float $minScore;

    protected bool $enabled;

    public function __construct(string $action, float $minScore = 0.5, bool $enabled = true)
    {
        $this->action = $action;
        $this->minScore = $minScore;
        $this->enabled = $enabled;
    }

    /**
     * When a login request is injected into a controller the validate method is called
     * When you retrieve the validated data, for use after the validate method is called again
     * This does not normally matter but in the case of calling a 3rd party api like Google reCAPTCHA
     * it causes the request to be sent twice with the same token, causing the duplicate request bug
     * the workaround for this is to set a flag saying that we have already validated the captcha
     * This will prevent the validate method from being called again
     */
    public function validate(string $attribute, mixed $value, Closure $fail): void
    {
        // If we have already validated, skip further validation
        if ($this->hasValidated) {
            return;
        }

        // Otherwise this must be the first time we are validating
        $this->hasValidated = true;

        // If the value is empty, we cant go any further
        if (empty($value)) {
            $fail('captcha::validation.required')->translate([
                'attribute' => $attribute,
            ]);

            return;
        }

        // If we are currently in a testing environment, skip the validation
        if (app()->environment('testing')) {
            return;
        }

        // If not enabled, skip the validation
        if (! $this->enabled) {
            return;
        }

        // Call the Captcha service to verify the captcha response
        $result = app(CaptchaManager::class)->verify($value, $this->action);

        // If the result is not successful or does not contain a score, fail the validation
        if (! $result['success'] || ! isset($result['score'])) {
            $fail('captcha::validation.no_score')->translate([
                'attribute' => $attribute,
                'action' => $this->action,
            ]);

            return;
        }

        // If the score is not valid or below the minimum score, fail the validation
        if ($result['score'] < $this->minScore) {
            $fail('captcha::validation.min_score')->translate([
                'attribute' => $attribute,
                'action' => $this->action,
                'score' => $result['score'],
                'minScore' => $this->minScore,
            ]);
        }
    }
}
