# Bongo Dashboard - Claude Code Guidance

## Overview

The **Bongo Dashboard** package provides the main admin dashboard landing page for the Bongo CMS system. It displays widget-based quick links to all registered packages and provides quick-create functionality.

**Package**: `bongo/dashboard`
**Namespace**: `Bongo\Dashboard`
**Type**: Presentation-only Laravel package
**Dependencies**: `bongo/framework` (^3.0), Laravel 10+, PHP 8.2+

## Documentation

For detailed information, refer to these files:
- **ARCHITECTURE.md** - Detailed architecture, diagrams, and data flow
- **.cursorrules** - Comprehensive development guide with examples
- **.github/copilot-instructions.md** - Code style templates and patterns
- **README.md** - Installation and basic usage

## Quick Start

### Installation

```bash
composer require bongo/dashboard
```

The service provider is auto-discovered by Laravel.

### Available Commands

```bash
# Install dependencies
composer install

# Run tests
vendor/bin/phpunit

# Fix code style
vendor/bin/pint

# Check code style
vendor/bin/pint --test

# Static analysis
vendor/bin/phpstan analyse

# Update dependencies
rm -f composer.lock && composer update -W
```

## Architecture Quick Reference

### Package Structure

```
src/
├── Http/Controllers/Backend/
│   └── DashboardController.php      # Renders dashboard view
├── Routes/
│   └── backend.php                  # Admin routes (/admin)
├── Views/backend/
│   ├── index.blade.php              # Main dashboard template
│   └── partials/
│       ├── widget.blade.php         # Package link widget
│       └── quick_create.blade.php   # Quick-create dropdown item
├── Translations/en/
│   └── backend.php                  # English translations
└── DashboardServiceProvider.php     # Service provider
```

### Key Components

#### Service Provider
**File**: `src/DashboardServiceProvider.php`

```php
namespace Bongo\Dashboard;

use Bongo\Framework\Providers\AbstractServiceProvider;

class DashboardServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'dashboard';
}
```

Extends `AbstractServiceProvider` from `bongo/framework` which auto-registers:
- **Routes**: `src/Routes/backend.php` → `/admin` prefix with `auth` + `employee` middleware
- **Views**: `src/Views/` → `dashboard::` namespace
- **Translations**: `src/Translations/` → `dashboard::` namespace

#### Controller
**File**: `src/Http/Controllers/Backend/DashboardController.php`

```php
namespace Bongo\Dashboard\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;

class DashboardController extends AbstractController
{
    public function index()
    {
        return view('dashboard::backend.index');
    }
}
```

Renders the dashboard view. The `$packages` variable is automatically provided by the framework.

#### Routes
**File**: `src/Routes/backend.php`

```php
Route::get('/', [DashboardController::class, 'index'])->name('index');
```

- **Registered as**: `backend.dashboard.index`
- **URL**: `/admin`
- **Middleware**: `auth`, `employee` (auto-applied)

### View Architecture

The dashboard uses a component-based Blade template structure:

```blade
framework::backend.layouts.app
    └── dashboard::backend.index
        ├── blade::content_header (title + actions)
        │   ├── blade::linkButton (website link)
        │   └── blade::dropdown (quick-create menu)
        └── blade::content_body
            └── dashboard::backend.partials.widget (package cards)
```

## Key Files Reference

| File | Purpose | Lines |
|------|---------|-------|
| `src/DashboardServiceProvider.php` | Service provider, registers package | 11 |
| `src/Http/Controllers/Backend/DashboardController.php` | Main controller, renders dashboard | 14 |
| `src/Routes/backend.php` | Route definitions | 8 |
| `src/Views/backend/index.blade.php` | Main dashboard view | 33 |
| `src/Views/backend/partials/widget.blade.php` | Package widget card | 16 |
| `src/Views/backend/partials/quick_create.blade.php` | Quick-create dropdown item | 25 |
| `src/Translations/en/backend.php` | English translations | 7 |
| `tests/TestCase.php` | Base test case for tests | 27 |

## Integration with Bongo Framework

This package heavily relies on `bongo/framework`:

### Framework Provides

1. **AbstractServiceProvider**: Auto-bootstrapping for routes, views, translations
2. **AbstractController**: Base controller with common functionality
3. **Backend Layout**: `framework::backend.layouts.app` template
4. **Blade Components**: `blade::content_header`, `blade::content_body`, `blade::linkButton`, etc.
5. **Package Registry**: Injects `$packages` array into dashboard view

### Package Data Structure

The framework provides this `$packages` array to the dashboard view:

```php
[
    [
        'name' => 'Pages',              // Display name
        'route' => 'backend.page',      // Route prefix
        'url' => '/admin/pages',        // Index URL
        'icon' => 'document',           // Icon identifier (optional)
    ],
    // ... more packages
]
```

## Common Tasks

### Adding a Dashboard Section

1. **Modify Controller** (fetch data):
```php
public function index()
{
    $statistics = [
        'users' => User::count(),
        'posts' => Post::count(),
    ];

    return view('dashboard::backend.index', compact('statistics'));
}
```

2. **Update View** (`src/Views/backend/index.blade.php`):
```blade
@component('blade::content_body')
    <section class="dashboard-stats mb-8">
        <h3>{{ trans('dashboard::backend.statistics') }}</h3>
        <!-- Stats widgets here -->
    </section>

    <section class="dashboard-links">
        <!-- Existing content -->
    </section>
@endcomponent
```

3. **Add Translations** (`src/Translations/en/backend.php`):
```php
return [
    'statistics' => 'Statistics',
    // ...
];
```

### Customizing Widget Appearance

Edit `src/Views/backend/partials/widget.blade.php`:
- Modify Tailwind CSS classes
- Change hover effects
- Add additional data attributes
- Customize icon display logic

### Adding New Routes

1. **Add Controller Method** (`src/Http/Controllers/Backend/DashboardController.php`):
```php
public function statistics(): JsonResponse
{
    return response()->json(['data' => ...]);
}
```

2. **Register Route** (`src/Routes/backend.php`):
```php
Route::get('/statistics', [DashboardController::class, 'statistics'])
    ->name('statistics');
```

## Testing

### Running Tests

```bash
# Run all tests
vendor/bin/phpunit

# Run with coverage
vendor/bin/phpunit --coverage-html coverage

# Run specific test
vendor/bin/phpunit --filter testDashboardLoads
```

### Writing Tests

All tests extend `Bongo\Dashboard\Tests\TestCase`:

```php
namespace Bongo\Dashboard\Tests\Feature;

use Bongo\Dashboard\Tests\TestCase;

class DashboardTest extends TestCase
{
    /** @test */
    public function it_displays_the_dashboard(): void
    {
        $response = $this->get(route('backend.dashboard.index'));

        $response->assertStatus(200);
        $response->assertViewIs('dashboard::backend.index');
        $response->assertSee(trans('dashboard::backend.quick_links'));
    }
}
```

## Code Style

### PHP Standards
- **PHP Version**: 8.2+
- **Laravel**: 10+
- **Style**: Laravel Pint (PSR-12 with Laravel preset)
- **Type Declarations**: Always use strict types and return types
- **Namespace**: `Bongo\Dashboard`

### Blade Conventions
- Use `{{ }}` for output (auto-escaped)
- Use `@` directives for control structures
- Always use `trans()` for translations, never hardcode strings
- Component-based structure preferred
- Extract reusable sections to partials

### Naming Conventions
- Controllers: `{Name}Controller` (singular, PascalCase)
- Views: `kebab-case.blade.php`
- Routes: Use `->name()` for all routes
- Translations: `dashboard::backend.{key}`

## Dependencies

### Required
- `php` >= 8.2
- `illuminate/contracts` ^10.0
- `bongo/framework` ^3.0

### Development
- `laravel/pint` ^1.0 (code style)
- `phpunit/phpunit` ^10.0 (testing)
- `orchestra/testbench` ^8.0 (package testing)
- `nunomaduro/larastan` ^2.0 (static analysis)

## Important Notes

1. **No Database**: This package has no models, migrations, or database dependencies
2. **Presentation Only**: Pure view layer, no business logic
3. **Framework Dependency**: Heavily relies on `bongo/framework` for functionality
4. **Package Registry**: The `$packages` variable is injected by the framework, not fetched in the controller
5. **Authentication**: All routes are protected by `auth` + `employee` middleware
6. **Route Prefix**: All routes auto-prefixed with `/admin` by framework
7. **Icon System**: Icons are optional and loaded from `bongo/blade` package

## Troubleshooting

### Dashboard Not Loading
- Verify `DashboardServiceProvider` is registered
- Check route exists: `php artisan route:list | grep dashboard`
- Ensure authentication middleware is configured

### Packages Not Displaying
- Verify framework is providing `$packages` variable
- Check view composer or middleware in `bongo/framework`

### Icons Not Showing
- Ensure `bongo/blade` package is installed
- Verify icon view exists: `view()->exists('blade::icons.{icon}')`

### Styling Issues
- Check Tailwind CSS is compiled
- Verify `framework::backend.layouts.app` loads stylesheets

## Getting Help

For more detailed information:
- Read **ARCHITECTURE.md** for system design and data flow
- Check **.cursorrules** for comprehensive development guide
- Review test files in `tests/` for usage examples

## Version History

See commit history:
```bash
git log --oneline
```

Current branch: `v3.0` (Laravel 10, PHP 8.2)
