# GitHub Copilot Instructions - Bongo Dashboard

## Project Overview

This is the **bongo/dashboard** Laravel package that provides the admin dashboard landing page for the Bongo CMS system. It displays widget-based quick links to all registered packages and provides quick-create functionality.

**Namespace**: `Bongo\Dashboard`
**Dependencies**: `bongo/framework` (^3.0), Laravel 10+, PHP 8.2+

## Architecture

### Key Classes

#### DashboardServiceProvider
**Path**: `src/DashboardServiceProvider.php`

```php
namespace Bongo\Dashboard;

use Bongo\Framework\Providers\AbstractServiceProvider;

class DashboardServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'dashboard';
}
```

**Purpose**: Registers the dashboard package with Laravel. Extends `AbstractServiceProvider` which auto-registers routes, views, and translations.

#### DashboardController
**Path**: `src/Http/Controllers/Backend/DashboardController.php`

```php
namespace Bongo\Dashboard\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;

class DashboardController extends AbstractController
{
    public function index()
    {
        return view('dashboard::backend.index');
    }
}
```

**Purpose**: Renders the main dashboard view. The `$packages` variable is automatically provided by the framework.

### Route Structure

**File**: `src/Routes/backend.php`

```php
Route::get('/', [DashboardController::class, 'index'])->name('index');
```

- Registered route name: `backend.dashboard.index`
- URL: `/admin` (backend routes auto-prefixed)
- Middleware: `auth`, `employee` (auto-applied to backend routes)

## Code Style Templates

### Adding a New Dashboard Section

```php
// In DashboardController.php
public function index()
{
    $statistics = $this->gatherStatistics();

    return view('dashboard::backend.index', compact('statistics'));
}

private function gatherStatistics(): array
{
    return [
        'total_users' => User::count(),
        'total_posts' => Post::count(),
    ];
}
```

### Creating a New Widget Partial

```blade
{{-- src/Views/backend/partials/custom_widget.blade.php --}}
<div class="col-span-1 bg-white overflow-hidden shadow rounded-sm p-4">
    <div class="flex items-center">
        <div class="flex-shrink-0 bg-indigo-500 rounded-md p-3">
            <div class="h-6 w-6 text-white">
                {{-- Icon here --}}
            </div>
        </div>
        <div class="ml-5">
            <div class="text-sm font-medium text-gray-500">{{ $title }}</div>
            <div class="text-2xl font-semibold text-gray-900">{{ $value }}</div>
        </div>
    </div>
</div>
```

### Adding Translation Keys

```php
// src/Translations/en/backend.php
return [
    'index' => 'Dashboard',
    'website' => 'Website',
    'quick_links' => 'Quick Links',
    'new_section' => 'New Section Title',
];
```

### Adding a Custom Route

```php
// src/Routes/backend.php
use Bongo\Dashboard\Http\Controllers\Backend\DashboardController;
use Illuminate\Support\Facades\Route;

Route::get('/', [DashboardController::class, 'index'])->name('index');
Route::get('/statistics', [DashboardController::class, 'statistics'])->name('statistics');
```

### Writing Feature Tests

```php
// tests/Feature/DashboardTest.php
namespace Bongo\Dashboard\Tests\Feature;

use Bongo\Dashboard\Tests\TestCase;

class DashboardTest extends TestCase
{
    /** @test */
    public function it_displays_the_dashboard_page(): void
    {
        $response = $this->get(route('backend.dashboard.index'));

        $response->assertStatus(200);
        $response->assertViewIs('dashboard::backend.index');
    }

    /** @test */
    public function it_shows_quick_links_section(): void
    {
        $response = $this->get(route('backend.dashboard.index'));

        $response->assertSee(trans('dashboard::backend.quick_links'));
    }
}
```

## Common Patterns

### View Component Usage

```blade
{{-- Extending backend layout --}}
@extends('framework::backend.layouts.app')

{{-- Setting page title --}}
@section('page_title', 'Dashboard')

{{-- Content header with actions --}}
@component('blade::content_header', ['hasTabs' => true])
    {{ trans('dashboard::backend.index') }}
    @slot('actions')
        @linkButton([
            'name' => trans('dashboard::backend.website'),
            'url' => '/',
            'target' => '_blank'
        ])
    @endslot
@endcomponent

{{-- Content body --}}
@component('blade::content_body')
    <section class="dashboard-content">
        {{-- Content here --}}
    </section>
@endcomponent
```

### Conditional Icon Display

```blade
@if (isset($package['icon']) && view()->exists('blade::icons.' . $package['icon']))
    <div class="h-6 w-6">
        @include('blade::icons.' . $package['icon'])
    </div>
@endif
```

### Route Existence Checking

```blade
@if(route_exists("{$package['route']}.create"))
    <a href="{{ route("{$package['route']}.create") }}">
        Create {{ Str::singular($package['name']) }}
    </a>
@endif
```

### Responsive Grid Layouts

```blade
<div class="grid grid-cols-2 gap-2 lg:grid-cols-4">
    @foreach($packages as $package)
        {{-- Widget here --}}
    @endforeach
</div>
```

## Testing Guidelines

### Test Structure

- Use `Bongo\Dashboard\Tests\TestCase` as base class
- TestCase extends `Orchestra\Testbench\TestCase`
- Service provider auto-registered in tests
- Use Orchestra Testbench for package testing

### Common Test Assertions

```php
// Route testing
$response->assertStatus(200);
$response->assertViewIs('dashboard::backend.index');
$response->assertViewHas('packages');

// Translation testing
$response->assertSee(trans('dashboard::backend.index'));

// Component testing
$response->assertSeeInOrder(['Dashboard', 'Quick Links']);
```

## File Structure Reference

```
src/
├── Http/Controllers/Backend/
│   └── DashboardController.php      # Main controller
├── Routes/
│   └── backend.php                  # Backend routes (admin only)
├── Views/backend/
│   ├── index.blade.php              # Main dashboard view
│   └── partials/
│       ├── widget.blade.php         # Package widget component
│       └── quick_create.blade.php   # Quick-create dropdown item
├── Translations/en/
│   └── backend.php                  # English translations
└── DashboardServiceProvider.php     # Package service provider

tests/
├── TestCase.php                     # Base test case
├── Feature/                         # Feature tests
└── Unit/                            # Unit tests
```

## Key Integration Points

### Framework Dependencies

1. **AbstractServiceProvider** (`bongo/framework`): Auto-registers resources
2. **AbstractController** (`bongo/framework`): Base controller functionality
3. **Backend Layout** (`framework::backend.layouts.app`): Standard admin layout
4. **Blade Components** (`blade::*`): UI components (buttons, forms, icons)
5. **Package Registry**: Framework provides `$packages` array to views

### Expected Variables

The dashboard view expects these variables (provided by framework):

```php
$packages = [
    [
        'name' => 'Package Name',      // Display name
        'route' => 'package.route',    // Route prefix
        'url' => '/admin/package',     // Index URL
        'icon' => 'icon-name',         // Optional icon identifier
    ],
    // ... more packages
];
```

## Code Style Guidelines

- **PHP**: Use Laravel Pint with Laravel preset
- **Type Declarations**: Always use strict types and return types
- **Docblocks**: Use for complex methods (optional for simple ones)
- **Blade**: Use `{{ }}` for output, `@` directives for logic
- **Translations**: Always use `trans()` or `{{ trans() }}`, never hardcode strings
- **Classes**: One class per file, PascalCase naming
- **Methods**: camelCase naming, descriptive names

## Commands Reference

```bash
# Run tests
vendor/bin/phpunit

# Fix code style
vendor/bin/pint

# Check code style
vendor/bin/pint --test

# Static analysis
vendor/bin/phpstan analyse

# Update dependencies
composer update -W
```

## Notes

- This package has **no models or database** - it's purely presentational
- The `$packages` variable is **injected by the framework**, not fetched in the controller
- Quick-create functionality filters out packages without `.create` routes
- All routes are backend-only (admin area)
- Icons are optional and loaded from the `blade` package
