# Bongo Dashboard - Architecture Documentation

## Overview

The **Bongo Dashboard** package provides the main landing page for the Bongo CMS admin area. It displays a widget-based interface showing all registered admin packages with quick links and quick-create functionality.

**Package**: `bongo/dashboard`
**Namespace**: `Bongo\Dashboard`
**Type**: Laravel Package
**Purpose**: Admin dashboard presentation layer

## System Architecture

### High-Level Component Diagram

```
┌─────────────────────────────────────────────────────────────┐
│                     Laravel Application                      │
├─────────────────────────────────────────────────────────────┤
│                                                               │
│  ┌────────────────────────────────────────────────────┐    │
│  │         DashboardServiceProvider                    │    │
│  │  (extends AbstractServiceProvider)                  │    │
│  └───────────────┬────────────────────────────────────┘    │
│                  │                                           │
│                  │ Auto-registers                            │
│                  ├─── Routes (backend.php)                   │
│                  ├─── Views (backend/)                       │
│                  └─── Translations (en/)                     │
│                                                               │
│  ┌────────────────────────────────────────────────────┐    │
│  │              HTTP Layer                             │    │
│  │  ┌──────────────────────────────────────────┐     │    │
│  │  │   DashboardController                     │     │    │
│  │  │   - index(): View                         │     │    │
│  │  └──────────────────────────────────────────┘     │    │
│  └────────────────────────────────────────────────────┘    │
│                                                               │
│  ┌────────────────────────────────────────────────────┐    │
│  │           Presentation Layer                        │    │
│  │  ┌──────────────────────────────────────────┐     │    │
│  │  │   Views (Blade Templates)                 │     │    │
│  │  │   - index.blade.php (main view)           │     │    │
│  │  │   - partials/widget.blade.php             │     │    │
│  │  │   - partials/quick_create.blade.php       │     │    │
│  │  └──────────────────────────────────────────┘     │    │
│  └────────────────────────────────────────────────────┘    │
│                                                               │
└─────────────────────────────────────────────────────────────┘
         │                                     ▲
         │ Uses                                │ Provides
         ▼                                     │
┌─────────────────────────────────────────────┴─────────────┐
│                  Bongo Framework                           │
│  - AbstractServiceProvider (auto-bootstrapping)            │
│  - AbstractController (base controller)                    │
│  - Backend layouts (framework::backend.layouts.app)        │
│  - Blade components (blade::*)                             │
│  - Package registry ($packages array)                      │
└────────────────────────────────────────────────────────────┘
```

## Directory Structure

```
dashboard/
├── src/
│   ├── Http/
│   │   └── Controllers/
│   │       └── Backend/
│   │           └── DashboardController.php    # Renders dashboard view
│   ├── Routes/
│   │   └── backend.php                        # Admin routes (/admin)
│   ├── Views/
│   │   └── backend/
│   │       ├── index.blade.php                # Main dashboard template
│   │       └── partials/
│   │           ├── widget.blade.php           # Package link widget
│   │           └── quick_create.blade.php     # Quick-create dropdown item
│   ├── Translations/
│   │   └── en/
│   │       └── backend.php                    # English translations
│   └── DashboardServiceProvider.php           # Service provider
│
├── tests/
│   ├── TestCase.php                           # Orchestra Testbench base
│   ├── Feature/                               # Feature tests
│   └── Unit/                                  # Unit tests
│
├── composer.json                              # Package dependencies
├── phpunit.xml                                # PHPUnit configuration
└── README.md                                  # Package documentation
```

## Core Components

### 1. Service Provider

**File**: `src/DashboardServiceProvider.php`

```php
namespace Bongo\Dashboard;

use Bongo\Framework\Providers\AbstractServiceProvider;

class DashboardServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'dashboard';
}
```

**Responsibilities**:
- Registers the package with Laravel
- Inherits auto-registration from `AbstractServiceProvider`:
  - Routes: `src/Routes/backend.php` → `/admin` with auth middleware
  - Views: `src/Views/` → `dashboard::` namespace
  - Translations: `src/Translations/` → `dashboard::` namespace

**Lifecycle**: Registered during Laravel's service provider discovery phase.

### 2. Controller Layer

**File**: `src/Http/Controllers/Backend/DashboardController.php`

```php
namespace Bongo\Dashboard\Http\Controllers\Backend;

use Bongo\Framework\Http\Controllers\AbstractController;

class DashboardController extends AbstractController
{
    public function index()
    {
        return view('dashboard::backend.index');
    }
}
```

**Responsibilities**:
- Renders the dashboard view
- Receives `$packages` array from framework (via middleware/view composer)

**Class Hierarchy**:
```
AbstractController (bongo/framework)
    └── DashboardController
```

### 3. Route Configuration

**File**: `src/Routes/backend.php`

```php
use Bongo\Dashboard\Http\Controllers\Backend\DashboardController;
use Illuminate\Support\Facades\Route;

Route::get('/', [DashboardController::class, 'index'])->name('index');
```

**Route Resolution**:
- Pattern: `/`
- Registered As: `backend.dashboard.index`
- Full URL: `/admin` (framework adds prefix)
- Middleware: `['web', 'auth', 'employee']` (auto-applied)

### 4. View Layer

#### Main Dashboard View

**File**: `src/Views/backend/index.blade.php`

```blade
@extends('framework::backend.layouts.app')
@section('page_title', 'Dashboard')

@section('content')
    @component('blade::content_header', ['hasTabs' => true])
        {{ trans('dashboard::backend.index') }}
        @slot('actions')
            @linkButton([...])
            <dropdown>...</dropdown>
        @endslot
    @endcomponent

    @component('blade::content_body')
        <section class="dashboard-links">
            <h3>{{ trans('dashboard::backend.quick_links') }}</h3>
            <div class="grid grid-cols-2 gap-2 lg:grid-cols-4">
                @foreach($packages as $package)
                    @include('dashboard::backend.partials.widget', ['package' => $package])
                @endforeach
            </div>
        </section>
    @endcomponent
@stop
```

**Template Hierarchy**:
```
framework::backend.layouts.app
    └── dashboard::backend.index
        ├── blade::content_header
        │   ├── blade::linkButton
        │   └── blade::dropdown
        └── blade::content_body
            └── dashboard::backend.partials.widget
```

#### Widget Partial

**File**: `src/Views/backend/partials/widget.blade.php`

**Purpose**: Renders an individual package as a clickable card.

**Props**: `$package` array with keys: `name`, `url`, `icon` (optional)

**Output**: Responsive card with icon and label linking to package index.

#### Quick Create Partial

**File**: `src/Views/backend/partials/quick_create.blade.php`

**Purpose**: Renders dropdown menu item for quick-creating package records.

**Props**: `$package` array with keys: `name`, `route`, `icon` (optional)

**Logic**: Only rendered if `route("{$package['route']}.create")` exists.

## Data Flow Diagram

```
┌──────────────────────────────────────────────────────────────┐
│  1. User requests /admin                                      │
└────────────────────┬─────────────────────────────────────────┘
                     │
                     ▼
┌──────────────────────────────────────────────────────────────┐
│  2. Laravel Router                                            │
│     - Matches route: backend.dashboard.index                 │
│     - Applies middleware: web, auth, employee                │
└────────────────────┬─────────────────────────────────────────┘
                     │
                     ▼
┌──────────────────────────────────────────────────────────────┐
│  3. Framework Middleware/View Composer                        │
│     - Gathers registered packages                            │
│     - Builds $packages array                                 │
└────────────────────┬─────────────────────────────────────────┘
                     │
                     ▼
┌──────────────────────────────────────────────────────────────┐
│  4. DashboardController::index()                              │
│     - Returns view('dashboard::backend.index')               │
└────────────────────┬─────────────────────────────────────────┘
                     │
                     ▼
┌──────────────────────────────────────────────────────────────┐
│  5. Blade Template Rendering                                  │
│     - Extends framework::backend.layouts.app                 │
│     - Loops through $packages                                │
│     - Includes widget and quick_create partials              │
└────────────────────┬─────────────────────────────────────────┘
                     │
                     ▼
┌──────────────────────────────────────────────────────────────┐
│  6. HTML Response                                             │
│     - Dashboard with package widgets                         │
│     - Quick-create dropdown                                  │
│     - Fully rendered admin dashboard                         │
└──────────────────────────────────────────────────────────────┘
```

## Package Registration System

The dashboard relies on a package registry provided by the Bongo framework:

### Package Data Structure

```php
[
    [
        'name' => 'Pages',              // Display name
        'route' => 'backend.page',      // Route prefix
        'url' => '/admin/pages',        // Index URL
        'icon' => 'document',           // Icon identifier (optional)
    ],
    [
        'name' => 'Posts',
        'route' => 'backend.post',
        'url' => '/admin/posts',
        'icon' => 'newspaper',
    ],
    // ... more packages
]
```

### Package Discovery Flow

```
Framework Bootstrap
    │
    ├── Discovers Service Providers
    │   └── Each package registers via ServiceProvider
    │
    ├── Builds Package Registry
    │   └── Collects route info, names, icons
    │
    └── Injects into Views
        └── Via view composer or middleware
            └── Available as $packages variable
```

## Extension Points

### 1. Adding Dashboard Widgets

**Location**: `src/Views/backend/index.blade.php`

**Example**: Add a statistics widget

```blade
@component('blade::content_body')
    {{-- New statistics section --}}
    <section class="dashboard-stats mb-8">
        <h3>{{ trans('dashboard::backend.statistics') }}</h3>
        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
            {{-- Custom stat widgets here --}}
        </div>
    </section>

    {{-- Existing quick links --}}
    <section class="dashboard-links">
        ...
    </section>
@endcomponent
```

### 2. Customizing Widget Appearance

**Location**: `src/Views/backend/partials/widget.blade.php`

**Approach**: Modify Tailwind classes, add additional data attributes, customize hover effects.

### 3. Adding Controller Methods

**Location**: `src/Http/Controllers/Backend/DashboardController.php`

**Example**: Add dashboard statistics endpoint

```php
public function statistics(): JsonResponse
{
    return response()->json([
        'users' => User::count(),
        'pages' => Page::count(),
    ]);
}
```

Register route in `src/Routes/backend.php`:

```php
Route::get('/statistics', [DashboardController::class, 'statistics'])
    ->name('statistics');
```

### 4. Adding New Translations

**Location**: `src/Translations/en/backend.php`

```php
return [
    'index' => 'Dashboard',
    'website' => 'Website',
    'quick_links' => 'Quick Links',
    'statistics' => 'Statistics',  // New key
];
```

### 5. Filtering Packages

**Location**: `src/Views/backend/index.blade.php`

**Example**: Only show specific package types

```blade
@foreach($packages as $package)
    @if(in_array($package['route'], ['backend.page', 'backend.post']))
        @include('dashboard::backend.partials.widget', ['package' => $package])
    @endif
@endforeach
```

## Dependency Reference

### Framework Dependencies

| Component | Source | Purpose |
|-----------|--------|---------|
| `AbstractServiceProvider` | `bongo/framework` | Auto-bootstrapping base class |
| `AbstractController` | `bongo/framework` | Base controller with common methods |
| `framework::backend.layouts.app` | `bongo/framework` | Admin layout template |
| `blade::*` components | `bongo/blade` | UI components (buttons, dropdowns) |
| Package Registry | `bongo/framework` | Provides `$packages` array |

### External Dependencies

| Package | Version | Purpose |
|---------|---------|---------|
| `php` | >=8.2 | Runtime |
| `illuminate/contracts` | ^10.0 | Laravel contracts |
| `laravel/pint` | ^1.0 | Code style fixer |
| `phpunit/phpunit` | ^10.0 | Testing framework |
| `orchestra/testbench` | ^8.0 | Package testing |

## Testing Architecture

### Test Structure

```
tests/
├── TestCase.php              # Base test case for all tests
├── Feature/                  # Feature/integration tests
│   └── DashboardTest.php     # Dashboard rendering tests
└── Unit/                     # Unit tests
    └── ControllerTest.php    # Controller method tests
```

### TestCase Setup

**File**: `tests/TestCase.php`

```php
namespace Bongo\Dashboard\Tests;

use Orchestra\Testbench\TestCase as Orchestra;
use Bongo\Dashboard\DashboardServiceProvider;

class TestCase extends Orchestra
{
    protected function getPackageProviders($app): array
    {
        return [DashboardServiceProvider::class];
    }

    protected function getEnvironmentSetUp($app)
    {
        // Environment configuration
    }
}
```

### Test Examples

```php
// Feature test
public function test_dashboard_page_loads(): void
{
    $response = $this->get(route('backend.dashboard.index'));
    $response->assertStatus(200);
    $response->assertViewIs('dashboard::backend.index');
}

// View test
public function test_dashboard_displays_package_widgets(): void
{
    $packages = [['name' => 'Test', 'url' => '/test']];
    $view = view('dashboard::backend.index', compact('packages'));
    $this->assertStringContainsString('Test', $view->render());
}
```

## Performance Considerations

### Rendering Optimization

- **Package Loop**: Minimal logic in Blade loops
- **Icon Loading**: Lazy icon loading (only if view exists)
- **Caching**: Framework may cache package registry

### View Performance

- **Partials**: Extracted for reusability and maintainability
- **Components**: Framework components are optimized
- **Grid Layout**: CSS Grid for efficient rendering

## Security Considerations

### Authentication & Authorization

- **Middleware**: All routes protected by `auth` + `employee` middleware
- **Route Guards**: Automatic via framework's backend route registration
- **CSRF**: Handled by Laravel's web middleware group

### XSS Prevention

- **Blade Escaping**: All output uses `{{ }}` (auto-escaped)
- **Safe Includes**: Only includes from known view paths
- **Icon Rendering**: View existence check before inclusion

## Future Extension Ideas

### Potential Features

1. **Dashboard Customization**: User-specific widget preferences
2. **Quick Actions**: Frequently used actions pinned to dashboard
3. **Statistics Widgets**: Real-time metrics and charts
4. **Activity Feed**: Recent changes across all packages
5. **Search**: Quick search across all admin content
6. **Notifications**: Admin notifications and alerts

### Adding New Features

To add a new dashboard section:

1. **Add Controller Method**: Fetch data in `DashboardController`
2. **Add Route**: Register in `src/Routes/backend.php`
3. **Add View Section**: Create partial or modify `index.blade.php`
4. **Add Translations**: Update translation files
5. **Add Tests**: Write feature tests for new functionality

## Troubleshooting

### Common Issues

**Issue**: Dashboard not loading
**Check**: Ensure `DashboardServiceProvider` is registered in Laravel

**Issue**: Packages not displaying
**Check**: Verify framework is providing `$packages` variable

**Issue**: Icons not showing
**Check**: Ensure `bongo/blade` package is installed and icons exist

**Issue**: Routes not working
**Check**: Run `php artisan route:list` and verify `backend.dashboard.index` exists

## Summary

The Bongo Dashboard package is a **presentation-only package** that:

- Provides a clean, widget-based admin landing page
- Integrates seamlessly with the Bongo framework
- Displays all registered packages dynamically
- Offers quick-create functionality via dropdown
- Uses modern Tailwind CSS for responsive design
- Has minimal dependencies and no database requirements

It serves as the central hub for navigating the Bongo CMS admin area.
