# GitHub Copilot Instructions - Bongo Document Package

## Project Overview

Laravel package for document management (upload, storage, retrieval) within the Bongo framework. Handles PDF, Office documents, images with UUID-based access and soft deletes.

**Namespace**: `Bongo\Document`
**Framework**: Laravel 10+ with `bongo/framework` ^3.0

## Key Classes and Relationships

### Core Model

```php
namespace Bongo\Document\Models;

class Document extends AbstractModel
{
    use HasUUID;
    use SoftDeletes;

    protected $fillable = [
        'name', 'title', 'path', 'ext', 'size', 'type',
        'sort_order', 'created_by', 'updated_by'
    ];

    protected $appends = ['src'];

    // Accessors
    public function getFilePathAttribute(): string;  // path + name
    public function getSrcAttribute(): string;       // Storage::url()

    // Scopes
    public function scopeOfType($query, $type);

    // Helpers
    public function isPdf(): bool;
}
```

### Controllers

**API Controller** (`src/Http/Controllers/Api/DocumentController.php`):
```php
class DocumentController extends AbstractApiController
{
    public function index(): JsonResponse;          // List all
    public function store(StoreDocumentRequest): JsonResponse;  // Upload
    public function destroy(Request): JsonResponse;  // Delete
}
```

**Backend Controller** (`src/Http/Controllers/Backend/DocumentController.php`):
```php
class DocumentController extends AbstractController
{
    public function index(): View;             // Admin interface
    public function show($uuid);               // Redirect to asset
    public function download($uuid);           // Force download
}
```

### Request Validation

```php
namespace Bongo\Document\Http\Requests;

class StoreDocumentRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'file' => [
                'required',
                'max:10240',  // 10MB
                'mimes:' . config('document.allowed_document_types')
            ],
        ];
    }
}
```

### Action Class

```php
namespace Bongo\Document\Actions;

class MakeSureUploadsDirectoryExists
{
    public function execute(): void;  // Ensures directory exists
}
```

### Service Provider

```php
namespace Bongo\Document;

class DocumentServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'document';
    // Auto-loads config, routes, views, migrations, translations
}
```

## Code Style Templates

### Adding a New Controller Method

```php
public function methodName(Request $request): JsonResponse
{
    // Validate or use FormRequest

    // Perform action
    $document = Document::where('uuid', $request->uuid)->firstOrFail();

    // Return response
    return response()->json(['item' => $document]);
}
```

### Adding a Model Scope

```php
public function scopeScopeName($query, $parameter)
{
    return $query->where('column', $parameter);
}

// Usage: Document::scopeName($value)->get();
```

### Creating an Action Class

```php
namespace Bongo\Document\Actions;

class ActionName
{
    public function execute(Document $document): void
    {
        // Single-purpose operation
    }
}

// Usage: (new ActionName())->execute($document);
```

### File Upload Pattern

```php
public function store(StoreDocumentRequest $request): JsonResponse
{
    // 1. Ensure directory exists
    (new MakeSureUploadsDirectoryExists())->execute();

    // 2. Process file
    $file = $request->file('file');
    $extension = $file->getClientOriginalExtension();
    $size = $file->getSize();

    // 3. Generate unique filename
    $fileName = Str::slug(rtrim($file->getClientOriginalName(), $extension));
    $fileName = $fileName . '-' . uniqid() . '.' . $extension;

    // 4. Get upload path from config
    $filePath = config('document.public_path') . 'uploads/';

    // 5. Store file
    if ($file->storePubliclyAs($filePath, $fileName)) {
        // 6. Create database record
        $document = new Document();
        $document->name = $fileName;
        $document->path = $filePath;
        $document->ext = $extension;
        $document->size = $size;
        $document->type = 'upload';
        $document->created_by = user()?->id;
        $document->updated_by = user()?->id;
        $document->save();

        return response()->json(['item' => $document]);
    }

    return response()->json('error', 400);
}
```

### UUID Access Pattern

```php
// Always use UUID in public-facing routes for security
$document = Document::where('uuid', $uuid)->firstOrFail();

// Never expose auto-increment IDs in URLs
Route::get('{uuid}', [Controller::class, 'show'])->name('show');
```

### Storage Operations

```php
// Check if file exists
if (Storage::exists($document->file_path)) {
    Storage::delete($document->file_path);
}

// Download file
return Storage::download($document->file_path);

// Get URL
$url = Storage::url($document->file_path);
// Or use accessor
$url = $document->src;
```

## Common Patterns

### Route Definition (Backend)

```php
// src/Routes/backend.php
Route::prefix('documents')
    ->as('document.')
    ->group(function () {
        Route::get('/', [DocumentController::class, 'index'])->name('index');
        Route::get('{uuid?}', [DocumentController::class, 'show'])->name('show');
    });
```

### Route Definition (API)

```php
// src/Routes/api.php
Route::prefix('documents')
    ->as('document.')
    ->group(function () {
        Route::get('/', [DocumentController::class, 'index'])->name('index');
        Route::post('store', [DocumentController::class, 'store'])->name('store');
    });
```

### Configuration Access

```php
// Always use config helper
$allowedTypes = config('document.allowed_document_types');
$publicPath = config('document.public_path');
$tmpPath = config('document.tmp_path');
```

### Translation Usage

```blade
{{-- In Blade templates --}}
@lang('document::document.index')
@lang('document::document.store_success')
```

```php
// In controllers
__('document::document.store_success')
```

### Soft Delete Queries

```php
// Active documents only
Document::all();

// Include soft deleted
Document::withTrashed()->get();

// Only soft deleted
Document::onlyTrashed()->get();

// Restore
$document->restore();

// Permanent delete
$document->forceDelete();
```

## Database Schema

```sql
CREATE TABLE documents (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    uuid CHAR(36) INDEX,
    name VARCHAR(255) NULLABLE,
    title VARCHAR(255) NULLABLE,
    path VARCHAR(255) NOT NULL,
    ext VARCHAR(255) NULLABLE,
    size VARCHAR(255) NULLABLE,
    type VARCHAR(255) NULLABLE,
    sort_order INT NULLABLE,
    created_by INT UNSIGNED NULLABLE INDEX,
    updated_by INT UNSIGNED NULLABLE INDEX,
    deleted_by INT UNSIGNED NULLABLE INDEX,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    deleted_at TIMESTAMP NULLABLE
);
```

## Configuration

```php
// config/document.php
return [
    'public_path' => 'public/',
    'tmp_path' => 'public/tmp/',
    'allowed_document_types' => 'png,jpg,jpeg,gif,pdf,odt,doc,docx,dot,odp,ppt,pptx,pps,pot,csv,txt,rtf,ods,xls,xlsx,xlm,xla,xlc,xlt,xlw',
];
```

## Testing Pattern

```php
namespace Bongo\Document\Tests;

class FeatureTest extends TestCase
{
    public function test_feature()
    {
        // Arrange
        $document = Document::factory()->create();

        // Act
        $response = $this->get(route('backend.document.show', $document->uuid));

        // Assert
        $response->assertStatus(302);
    }
}
```

## Middleware Applied by Framework

- **backend.php routes**: `web`, `auth`, `employee`
- **api.php routes**: `api`, `auth:sanctum`

## Helper Functions Available

```php
user()  // Get current authenticated user (from bongo/framework)
```

## Extending the Package

### Add New File Type Validation

1. Update `config/document.php` → `allowed_document_types`
2. Add specific handling in `StoreDocumentRequest` if needed

### Add New Model Method

```php
// src/Models/Document.php
public function methodName(): ReturnType
{
    // Implementation
}
```

### Add New Route

Edit `src/Routes/backend.php` or `src/Routes/api.php` - routes auto-register via `AbstractServiceProvider`.

## Key Dependencies

- `bongo/framework` ^3.0 - Base classes, traits, helpers
- Laravel 10+ - Framework features
- PHP 8.2+ - Language features
