# Architecture Documentation - Bongo Document Package

## Overview

The Bongo Document package provides document management functionality for the Bongo framework ecosystem. It handles file uploads, storage management, and metadata tracking for various document types including PDFs, Office documents, and images.

**Package**: `bongo/document`
**Namespace**: `Bongo\Document`
**Version**: ^3.0
**Requirements**: PHP >=8.2, Laravel ^10.0, bongo/framework ^3.0

## Directory Structure

```
bongo/document/
├── src/
│   ├── Actions/
│   │   └── MakeSureUploadsDirectoryExists.php    # Ensures upload directory exists
│   ├── Config/
│   │   └── document.php                           # Package configuration
│   ├── Http/
│   │   ├── Controllers/
│   │   │   ├── Api/
│   │   │   │   └── DocumentController.php         # API endpoints (CRUD)
│   │   │   └── Backend/
│   │   │       └── DocumentController.php         # Admin interface
│   │   └── Requests/
│   │       └── StoreDocumentRequest.php           # File upload validation
│   ├── Migrations/
│   │   └── 2021_01_01_000001_create_documents_table.php
│   ├── Models/
│   │   └── Document.php                           # Document model
│   ├── Routes/
│   │   ├── api.php                                # API routes (auth:sanctum)
│   │   └── backend.php                            # Backend routes (auth+employee)
│   ├── Seeders/
│   │   └── PackageSeeder.php                      # Package registration seeder
│   ├── Translations/
│   │   └── en/
│   │       └── document.php                       # English translations
│   ├── Views/
│   │   └── backend/
│   │       └── index.blade.php                    # Admin interface view
│   └── DocumentServiceProvider.php                # Service provider
├── tests/
│   └── TestCase.php                               # Base test case
├── composer.json                                  # Package dependencies
├── phpunit.xml                                    # PHPUnit configuration
└── README.md                                      # Package documentation
```

## Class Hierarchy

```
AbstractModel (bongo/framework)
    └── Document
            Uses: HasUUID (bongo/framework)
            Uses: SoftDeletes (Laravel)

AbstractController (bongo/framework)
    └── Backend\DocumentController

AbstractApiController (bongo/framework)
    └── Api\DocumentController

AbstractServiceProvider (bongo/framework)
    └── DocumentServiceProvider

FormRequest (Laravel)
    └── StoreDocumentRequest
```

## Service Provider Architecture

### DocumentServiceProvider

The service provider extends `Bongo\Framework\Providers\AbstractServiceProvider`, which provides automatic bootstrapping:

```php
<?php

namespace Bongo\Document;

use Bongo\Framework\Providers\AbstractServiceProvider;

class DocumentServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'document';
}
```

### Auto-Bootstrap Behaviour

The `AbstractServiceProvider` automatically registers:

| Resource | Source | Registration |
|----------|--------|-------------|
| **Config** | `src/Config/document.php` | `config('document.*')` |
| **Routes** | `src/Routes/api.php` | Prefix: `/api/documents`, Middleware: `api`, `auth:sanctum` |
| **Routes** | `src/Routes/backend.php` | Prefix: `/admin/documents`, Middleware: `web`, `auth`, `employee` |
| **Views** | `src/Views/` | Namespace: `document::` |
| **Migrations** | `src/Migrations/` | Auto-discovered |
| **Translations** | `src/Translations/` | Namespace: `document::` |

## Data Flow Diagrams

### File Upload Flow

```
┌─────────────────────────────────────────────────────────────────────┐
│                         File Upload Flow                             │
└─────────────────────────────────────────────────────────────────────┘

Client (Vue Component)
    │
    │ POST /api/documents/store
    │ FormData: { file: File }
    ▼
Api\DocumentController::store(StoreDocumentRequest)
    │
    ├─► StoreDocumentRequest::rules()
    │       │
    │       ├─► Validate: required
    │       ├─► Validate: max:10240 (10MB)
    │       └─► Validate: mimes:config('document.allowed_document_types')
    │
    ├─► MakeSureUploadsDirectoryExists::execute()
    │       │
    │       ├─► Check storage driver
    │       └─► Create directory if not exists
    │
    ├─► Process file metadata
    │       ├─► Get extension
    │       ├─► Get size
    │       └─► Generate unique filename (slug + uniqid)
    │
    ├─► Store file
    │       │
    │       └─► Storage::storePubliclyAs($path, $fileName)
    │
    ├─► Create Document record
    │       │
    │       ├─► Set name, title, path, ext, size, type
    │       ├─► Set created_by, updated_by (from user())
    │       └─► Save to database
    │
    └─► Return JsonResponse
            └─► { item: Document }
```

### Document Retrieval Flow

```
┌─────────────────────────────────────────────────────────────────────┐
│                      Document Retrieval Flow                         │
└─────────────────────────────────────────────────────────────────────┘

Client Request
    │
    ├─► GET /admin/documents/{uuid}  (View)
    │       │
    │       ▼
    │   Backend\DocumentController::show($uuid)
    │       │
    │       ├─► Document::where('uuid', $uuid)->firstOrFail()
    │       ├─► Generate URL: asset($document->src)
    │       │       │
    │       │       └─► Storage::url($document->file_path)
    │       │
    │       └─► redirect()->away($url)
    │
    └─► GET /admin/documents/download/{uuid}  (Download)
            │
            ▼
        Backend\DocumentController::download($uuid)
            │
            ├─► Document::where('uuid', $uuid)->firstOrFail()
            └─► Storage::download($document->file_path)
```

### Document Deletion Flow

```
┌─────────────────────────────────────────────────────────────────────┐
│                      Document Deletion Flow                          │
└─────────────────────────────────────────────────────────────────────┘

Client (Vue Component)
    │
    │ POST /api/documents/delete
    │ Body: { id: number }
    ▼
Api\DocumentController::destroy(Request)
    │
    ├─► Document::findOrFail($request->id)
    │
    ├─► Check if file exists
    │       │
    │       └─► Storage::exists($document->file_path)
    │
    ├─► Delete file from storage
    │       │
    │       └─► Storage::delete($document->file_path)
    │
    ├─► Soft delete database record
    │       │
    │       └─► $document->delete()  (SoftDeletes)
    │
    └─► Return JsonResponse
            └─► 'Document deleted'
```

## Model Architecture

### Document Model

**Class**: `Bongo\Document\Models\Document`
**Extends**: `Bongo\Framework\Models\AbstractModel`
**Traits**: `HasUUID`, `SoftDeletes`

#### Properties

```php
protected $fillable = [
    'name',         // Filename with extension
    'title',        // Display title
    'path',         // Storage directory path
    'ext',          // File extension
    'size',         // File size in bytes (string)
    'type',         // Document type/category
    'sort_order',   // Optional ordering
    'created_by',   // User ID who created
    'updated_by',   // User ID who last updated
];

protected $appends = [
    'src',          // Full URL to document
];
```

#### Accessors

| Accessor | Return Type | Purpose |
|----------|-------------|---------|
| `getFilePathAttribute()` | `string` | Combines `path` + `name` for full file path |
| `getSrcAttribute()` | `string` | Returns full URL via `Storage::url()` |

#### Scopes

| Scope | Parameters | Purpose |
|-------|------------|---------|
| `scopeOfType()` | `$query, $type` | Filter documents by type |

#### Methods

| Method | Return Type | Purpose |
|--------|-------------|---------|
| `isPdf()` | `bool` | Check if document is a PDF |

#### Relationships

Currently, the Document model has no defined relationships but can be polymorphically related to other models through the Bongo framework's relationship system.

## Controller Architecture

### API Controller

**Class**: `Bongo\Document\Http\Controllers\Api\DocumentController`
**Extends**: `Bongo\Framework\Http\Controllers\AbstractApiController`
**Routes**: Prefixed with `/api/documents`, middleware `auth:sanctum`

#### Methods

| Method | HTTP | Route | Purpose | Request | Response |
|--------|------|-------|---------|---------|----------|
| `index()` | GET | `/api/documents` | List all documents | - | `JsonResponse` with items array |
| `store()` | POST | `/api/documents/store` | Upload document | `StoreDocumentRequest` | `JsonResponse` with item |
| `destroy()` | POST | `/api/documents/delete` | Delete document | `Request` with id | `JsonResponse` |

### Backend Controller

**Class**: `Bongo\Document\Http\Controllers\Backend\DocumentController`
**Extends**: `Bongo\Framework\Http\Controllers\AbstractController`
**Routes**: Prefixed with `/admin/documents`, middleware `auth`, `employee`

#### Methods

| Method | HTTP | Route | Purpose | Parameters | Response |
|--------|------|-------|---------|------------|----------|
| `index()` | GET | `/admin/documents` | Admin interface | - | `View` |
| `show()` | GET | `/admin/documents/{uuid}` | View document | `$uuid` | `RedirectResponse` |
| `download()` | GET | `/admin/documents/download/{uuid}` | Download document | `$uuid` | `StreamedResponse` |

## Request Validation

### StoreDocumentRequest

**Class**: `Bongo\Document\Http\Requests\StoreDocumentRequest`
**Extends**: `Illuminate\Foundation\Http\FormRequest`

#### Validation Rules

```php
public function rules(): array
{
    return [
        'file' => [
            'required',
            'max:10240',  // Maximum 10MB
            'mimes:' . config('document.allowed_document_types')
        ],
    ];
}
```

The `mimes` rule validates against the configured allowed types from `config/document.php`.

## Action Classes

### MakeSureUploadsDirectoryExists

**Class**: `Bongo\Document\Actions\MakeSureUploadsDirectoryExists`
**Pattern**: Single-action class

#### Purpose

Ensures the uploads directory exists before file upload operations.

#### Algorithm

```
1. Get upload path from config('document.public_path')
2. Check storage driver
   ├─► If 'local': Use File::ensureDirectoryExists()
   └─► If remote: Use Storage::makeDirectory()
```

#### Usage

```php
(new MakeSureUploadsDirectoryExists())->execute();
```

## Configuration

### Config File: `config/document.php`

| Key | Type | Default | Purpose |
|-----|------|---------|---------|
| `public_path` | string | `'public/'` | Base path for public documents |
| `tmp_path` | string | `'public/tmp/'` | Temporary upload path |
| `allowed_document_types` | string | `'png,jpg,jpeg,gif,pdf,...'` | Allowed MIME types (comma-separated) |

### Allowed Document Types

The default configuration supports:

- **Images**: png, jpg, jpeg, gif
- **PDFs**: pdf
- **OpenDocument**: odt, odp, ods
- **Microsoft Office**: doc, docx, dot, ppt, pptx, pps, pot, xls, xlsx, xlm, xla, xlc, xlt, xlw
- **Other**: csv, txt, rtf

## Database Schema

### documents Table

```sql
CREATE TABLE documents (
    -- Primary Keys
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    uuid CHAR(36) INDEX,

    -- File Metadata
    name VARCHAR(255) NULLABLE,
    title VARCHAR(255) NULLABLE,
    path VARCHAR(255) NOT NULL,
    ext VARCHAR(255) NULLABLE,
    size VARCHAR(255) NULLABLE,
    type VARCHAR(255) NULLABLE,
    sort_order INT NULLABLE,

    -- Audit Fields
    created_by INT UNSIGNED NULLABLE INDEX,
    updated_by INT UNSIGNED NULLABLE INDEX,
    deleted_by INT UNSIGNED NULLABLE INDEX,

    -- Timestamps
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    deleted_at TIMESTAMP NULLABLE
);
```

### Indexes

- `uuid` - Indexed for fast UUID lookups
- `created_by` - Indexed for filtering by creator
- `updated_by` - Indexed for filtering by updater
- `deleted_by` - Indexed for soft delete queries

## Routes

### Backend Routes (`src/Routes/backend.php`)

| Method | URI | Name | Controller Method | Purpose |
|--------|-----|------|-------------------|---------|
| GET | `/admin/documents` | `backend.document.index` | `index()` | Admin interface |
| GET | `/admin/documents/{uuid}` | `backend.document.show` | `show($uuid)` | View document |
| GET | `/admin/documents/download/{uuid}` | `backend.document.download` | `download($uuid)` | Download document |

### API Routes (`src/Routes/api.php`)

| Method | URI | Name | Controller Method | Purpose |
|--------|-----|------|-------------------|---------|
| GET | `/api/documents` | `api.document.index` | `index()` | List documents |
| POST | `/api/documents/store` | `api.document.store` | `store()` | Upload document |
| POST | `/api/documents/delete` | `api.document.destroy` | `destroy()` | Delete document |

## Frontend Integration

### Vue Component

The backend admin interface uses a Vue component for document management:

```blade
<document-manager
    index-url="{{ route('api.document.index') }}"
    store-url="{{ route('api.document.store') }}"
    show-url="{{ route('backend.document.show') }}"
    download-url="{{ route('backend.document.download') }}"
    delete-url="{{ route('api.document.destroy') }}"
></document-manager>
```

The `document-manager` component is provided by the Bongo framework's frontend system.

## Traits Reference

### HasUUID (from bongo/framework)

**Purpose**: Automatically generates UUID on model creation

**Provides**:
- Automatic UUID generation on `creating` event
- `uuid` attribute population

### SoftDeletes (from Laravel)

**Purpose**: Enables soft deletion of records

**Provides**:
- `delete()` - Soft deletes record (sets `deleted_at`)
- `restore()` - Restores soft-deleted record
- `forceDelete()` - Permanently deletes record
- `trashed()` - Check if record is soft-deleted
- Scopes: `withTrashed()`, `onlyTrashed()`

## Extension Points

### Adding New Document Types

1. **Update configuration** (`src/Config/document.php`):
   ```php
   'allowed_document_types' => 'existing,types,new,type',
   ```

2. **Validation automatically updates** via `StoreDocumentRequest`

### Adding New Routes

**Backend route** (`src/Routes/backend.php`):
```php
Route::get('custom/{uuid}', [DocumentController::class, 'customMethod'])
    ->name('custom');
```

**API route** (`src/Routes/api.php`):
```php
Route::post('bulk-upload', [DocumentController::class, 'bulkUpload'])
    ->name('bulk-upload');
```

### Adding Model Scopes

```php
// src/Models/Document.php
public function scopeByExtension($query, string $ext)
{
    return $query->where('ext', $ext);
}

// Usage
Document::byExtension('pdf')->get();
```

### Adding Accessors/Mutators

```php
// src/Models/Document.php
public function getFormattedSizeAttribute(): string
{
    return $this->size ? round($this->size / 1024, 2) . ' KB' : 'Unknown';
}

// Usage
$document->formatted_size;
```

### Adding New Actions

```php
// src/Actions/GenerateThumbnail.php
namespace Bongo\Document\Actions;

class GenerateThumbnail
{
    public function execute(Document $document): void
    {
        // Generate thumbnail logic
    }
}

// Usage
(new GenerateThumbnail())->execute($document);
```

### Adding Model Events

```php
// src/DocumentServiceProvider.php
protected function bootEvents(): void
{
    Document::created(function ($document) {
        // Handle document created event
    });

    Document::deleted(function ($document) {
        // Handle document deletion
        // Clean up files, etc.
    });
}
```

### Adding Relationships

```php
// src/Models/Document.php
public function creator()
{
    return $this->belongsTo(User::class, 'created_by');
}

public function updater()
{
    return $this->belongsTo(User::class, 'updated_by');
}

// Usage
$document->creator->name;
```

## How to Add New Features

### Feature: Bulk Upload

1. **Add route** (`src/Routes/api.php`):
   ```php
   Route::post('bulk-upload', [DocumentController::class, 'bulkUpload'])
       ->name('bulk-upload');
   ```

2. **Create request** (`src/Http/Requests/BulkUploadRequest.php`):
   ```php
   public function rules(): array
   {
       return [
           'files' => 'required|array',
           'files.*' => [
               'required',
               'max:10240',
               'mimes:' . config('document.allowed_document_types')
           ],
       ];
   }
   ```

3. **Add controller method**:
   ```php
   public function bulkUpload(BulkUploadRequest $request): JsonResponse
   {
       $documents = collect();

       foreach ($request->file('files') as $file) {
           // Process each file (similar to store method)
           $document = $this->processUpload($file);
           $documents->push($document);
       }

       return response()->json(['items' => $documents]);
   }
   ```

### Feature: Document Categories

1. **Add migration** (`src/Migrations/add_category_to_documents.php`):
   ```php
   Schema::table('documents', function (Blueprint $table) {
       $table->string('category')->nullable()->after('type');
   });
   ```

2. **Update model** (`src/Models/Document.php`):
   ```php
   protected $fillable = [..., 'category'];

   public function scopeInCategory($query, string $category)
   {
       return $query->where('category', $category);
   }
   ```

3. **Update form validation** (`src/Http/Requests/StoreDocumentRequest.php`):
   ```php
   'category' => 'nullable|string|max:255',
   ```

### Feature: Document Preview

1. **Add route** (`src/Routes/backend.php`):
   ```php
   Route::get('preview/{uuid}', [DocumentController::class, 'preview'])
       ->name('preview');
   ```

2. **Add controller method**:
   ```php
   public function preview($uuid): View
   {
       $document = Document::where('uuid', $uuid)->firstOrFail();

       return view('document::backend.preview', compact('document'));
   }
   ```

3. **Create view** (`src/Views/backend/preview.blade.php`):
   ```blade
   @extends('framework::backend.layouts.app')

   @section('content')
       <iframe src="{{ $document->src }}" width="100%" height="800px"></iframe>
   @stop
   ```

## Testing Architecture

### Base Test Case

```php
namespace Bongo\Document\Tests;

use Orchestra\Testbench\TestCase as Orchestra;
use Bongo\Document\DocumentServiceProvider;

class TestCase extends Orchestra
{
    protected function getPackageProviders($app): array
    {
        return [DocumentServiceProvider::class];
    }
}
```

### Example Test Structure

```php
namespace Bongo\Document\Tests\Feature;

use Bongo\Document\Models\Document;
use Bongo\Document\Tests\TestCase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;

class DocumentUploadTest extends TestCase
{
    public function test_can_upload_document()
    {
        Storage::fake('local');

        $file = UploadedFile::fake()->create('document.pdf', 100);

        $response = $this->postJson(route('api.document.store'), [
            'file' => $file
        ]);

        $response->assertStatus(200);
        $this->assertDatabaseHas('documents', [
            'ext' => 'pdf',
        ]);
    }
}
```

## Security Considerations

1. **UUID-based access**: Public routes use UUID instead of auto-increment IDs to prevent enumeration attacks
2. **File validation**: Strict MIME type and size validation via `StoreDocumentRequest`
3. **Authentication**: Backend requires `auth` + `employee` middleware, API requires `auth:sanctum`
4. **Soft deletes**: Documents can be recovered, files remain until force deleted
5. **Storage abstraction**: Uses Laravel's Storage facade for flexibility and security

## Performance Considerations

1. **Indexes**: UUID, created_by, updated_by are indexed for fast lookups
2. **Lazy loading**: Use eager loading for relationships to prevent N+1 queries
3. **Storage drivers**: Supports local and remote storage (S3, etc.)
4. **File size limit**: Default 10MB max upload, configurable via validation
