# CLAUDE.md - Bongo Document Package

## Overview

The Bongo Document package provides document management functionality for the Bongo framework, handling file uploads, storage, and retrieval for PDFs, Office documents, and images with UUID-based access and soft delete support.

**Package**: `bongo/document`
**Namespace**: `Bongo\Document`
**Requirements**: PHP >=8.2, Laravel ^10.0, bongo/framework ^3.0

## Quick Links

- **Architecture Details**: See `ARCHITECTURE.md` for complete class diagrams, data flows, and extension points
- **Cursor AI Instructions**: See `.cursorrules` for development conventions and common tasks
- **GitHub Copilot Instructions**: See `.github/copilot-instructions.md` for code templates and patterns

## Commands

```bash
# Testing
composer test                  # Run PHPUnit tests
vendor/bin/phpunit            # Direct PHPUnit execution

# Code Quality
composer lint                  # Check code style (php-cs-fixer)
composer fix                   # Fix code style issues
vendor/bin/pint --test        # Check style with Laravel Pint
vendor/bin/pint               # Fix style with Laravel Pint

# Development
composer install              # Install dependencies
composer update -W            # Update dependencies
```

## Architecture Quick Reference

### Service Provider

Extends `Bongo\Framework\Providers\AbstractServiceProvider` for auto-bootstrapping:

```php
class DocumentServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'document';
    // Auto-loads: config, routes, views, migrations, translations
}
```

### Routes

**Backend** (`/admin/documents`):
- `GET /` → `index()` - Admin interface
- `GET /{uuid}` → `show($uuid)` - View document
- `GET /download/{uuid}` → `download($uuid)` - Download document
- Middleware: `web`, `auth`, `employee` (auto-applied)

**API** (`/api/documents`):
- `GET /` → `index()` - List all documents
- `POST /store` → `store()` - Upload document
- `POST /delete` → `destroy()` - Delete document
- Middleware: `api`, `auth:sanctum` (auto-applied)

### Model

```php
Document extends AbstractModel
    Uses: HasUUID, SoftDeletes
    Fillable: name, title, path, ext, size, type, sort_order, created_by, updated_by
    Appends: src (full URL)
    Scopes: ofType($type)
    Methods: isPdf(), getFilePathAttribute(), getSrcAttribute()
```

### Controllers

- `Api\DocumentController` - CRUD operations, returns JSON
- `Backend\DocumentController` - Admin interface, view/download

### Configuration

```php
config('document.public_path')              // 'public/'
config('document.tmp_path')                 // 'public/tmp/'
config('document.allowed_document_types')   // MIME types string
```

### File Upload Flow

```
Client → StoreDocumentRequest (validation)
      → MakeSureUploadsDirectoryExists (action)
      → Storage::storePubliclyAs() (file storage)
      → Document::create() (database record)
      → JsonResponse (response)
```

## Key Files

| File | Purpose | Lines |
|------|---------|-------|
| `src/DocumentServiceProvider.php` | Service provider (minimal, extends AbstractServiceProvider) | 10 |
| `src/Models/Document.php` | Main model with UUID, soft deletes, accessors | 84 |
| `src/Http/Controllers/Api/DocumentController.php` | API CRUD operations | 77 |
| `src/Http/Controllers/Backend/DocumentController.php` | Admin view/download | 33 |
| `src/Http/Requests/StoreDocumentRequest.php` | File upload validation | 21 |
| `src/Actions/MakeSureUploadsDirectoryExists.php` | Directory creation action | 27 |
| `src/Config/document.php` | Configuration (paths, allowed types) | 7 |
| `src/Routes/backend.php` | Backend routes | 17 |
| `src/Routes/api.php` | API routes | 17 |
| `src/Migrations/2021_01_01_000001_create_documents_table.php` | Database schema | 50 |
| `src/Views/backend/index.blade.php` | Vue component integration | 21 |

## Database Schema

**documents table**:
- `id` (PK), `uuid` (indexed)
- File metadata: `name`, `title`, `path`, `ext`, `size`, `type`, `sort_order`
- Audit: `created_by`, `updated_by`, `deleted_by`
- Timestamps: `created_at`, `updated_at`, `deleted_at`

## Code Style Summary

- **PSR-12** coding standard with Laravel preset
- **Return types**: Always declare return types
- **Naming**: Singular controllers/models, plural tables/route prefixes
- **Actions**: Single-purpose classes with `execute()` method
- **UUID access**: Use UUID in public routes, not auto-increment IDs
- **Configuration**: Always use `config()` helper, never hardcode paths
- **Storage**: Use `Storage` facade for abstraction

## Common Patterns

### Upload a Document

```php
// Request validation (automatic)
$file = $request->file('file');

// Ensure directory exists
(new MakeSureUploadsDirectoryExists())->execute();

// Generate unique filename
$fileName = Str::slug(rtrim($file->getClientOriginalName(), $extension));
$fileName = $fileName . '-' . uniqid() . '.' . $extension;

// Store file
$file->storePubliclyAs(config('document.public_path') . 'uploads/', $fileName);

// Create record
$document = Document::create([...]);
```

### Retrieve by UUID

```php
$document = Document::where('uuid', $uuid)->firstOrFail();
```

### Access File URL

```php
// Via accessor (appended)
$url = $document->src;

// Via attribute
$url = Storage::url($document->file_path);
```

### Delete Document

```php
// Delete file from storage
if (Storage::exists($document->file_path)) {
    Storage::delete($document->file_path);
}

// Soft delete record
$document->delete();
```

## Framework Integration

This package depends on `bongo/framework` for:

- `AbstractServiceProvider` - Auto-bootstrapping of config, routes, views, migrations, translations
- `AbstractModel` - Base model functionality
- `AbstractController` - Backend controller base
- `AbstractApiController` - API controller base
- `HasUUID` trait - Automatic UUID generation
- `user()` helper - Get current authenticated user

## Extension Points

1. **Add document types**: Update `config/document.php` → `allowed_document_types`
2. **Add routes**: Edit `src/Routes/backend.php` or `src/Routes/api.php`
3. **Add model scopes**: Add `scopeMethodName()` methods to `Document` model
4. **Add actions**: Create new class in `src/Actions/` with `execute()` method
5. **Add relationships**: Define methods in `Document` model (e.g., `creator()`, `category()`)

## Testing

Tests use Orchestra Testbench with automatic `DocumentServiceProvider` loading.

```php
class MyTest extends \Bongo\Document\Tests\TestCase
{
    public function test_feature()
    {
        // Test implementation
    }
}
```

## Security

- UUID-based public access (prevents ID enumeration)
- File type and size validation
- Auth middleware on all routes
- Employee middleware on backend routes
- Soft deletes for recovery
- Storage abstraction for flexibility

## Frontend Integration

The package includes a Vue component for the admin interface:

```blade
<document-manager
    index-url="{{ route('api.document.index') }}"
    store-url="{{ route('api.document.store') }}"
    show-url="{{ route('backend.document.show') }}"
    download-url="{{ route('backend.document.download') }}"
    delete-url="{{ route('api.document.destroy') }}"
></document-manager>
```

Component provided by Bongo framework's frontend system.

## Supported File Types

Images: png, jpg, jpeg, gif
PDFs: pdf
OpenDocument: odt, odp, ods
Microsoft Office: doc, docx, dot, ppt, pptx, pps, pot, xls, xlsx, xlm, xla, xlc, xlt, xlw
Other: csv, txt, rtf

Maximum upload size: 10MB (configurable via `StoreDocumentRequest`)
