# Bongo Document Package

A Laravel package for managing document uploads (PDFs, Office documents, images) with UUID-based access, soft deletes, and storage abstraction.

## Features

- File upload with validation (type, size)
- Support for PDFs, Office documents, and images
- UUID-based public access (security)
- Soft delete support
- Storage abstraction (local, S3, etc.)
- Admin interface with Vue component
- RESTful API endpoints
- Audit trail (created_by, updated_by)

## Requirements

- PHP 8.2+
- Laravel 10+
- bongo/framework ^3.0

## Installation

### Via Composer

```bash
composer require bongo/document
```

### Service Provider

Laravel 11+ auto-discovers the service provider. For Laravel 10, add to `config/app.php`:

```php
Bongo\Document\DocumentServiceProvider::class,
```

### Run Migrations

```bash
php artisan migrate
```

This creates the `documents` table with UUID support and soft deletes.

## Configuration

Publish the configuration file (optional):

```bash
php artisan vendor:publish --tag=document-config
```

Configuration options in `config/document.php`:

```php
return [
    'public_path' => 'public/',           // Base path for public documents
    'tmp_path' => 'public/tmp/',          // Temporary upload path
    'allowed_document_types' => 'png,jpg,jpeg,gif,pdf,odt,doc,docx,...',
];
```

## Usage

### Routes

The package automatically registers these routes:

**Backend (Admin)**:
- `GET /admin/documents` - Admin interface
- `GET /admin/documents/{uuid}` - View document
- `GET /admin/documents/download/{uuid}` - Download document

**API**:
- `GET /api/documents` - List all documents
- `POST /api/documents/store` - Upload document
- `POST /api/documents/delete` - Delete document

### Upload a Document

**Via API**:

```bash
curl -X POST /api/documents/store \
  -H "Authorization: Bearer {token}" \
  -F "file=@document.pdf"
```

**Via Controller**:

```php
use Bongo\Document\Http\Requests\StoreDocumentRequest;
use Bongo\Document\Models\Document;
use Bongo\Document\Actions\MakeSureUploadsDirectoryExists;

public function upload(StoreDocumentRequest $request)
{
    // Ensure directory exists
    (new MakeSureUploadsDirectoryExists())->execute();

    // Process file
    $file = $request->file('file');
    $extension = $file->getClientOriginalExtension();
    $fileName = Str::slug(rtrim($file->getClientOriginalName(), $extension));
    $fileName = $fileName . '-' . uniqid() . '.' . $extension;

    // Store file
    $filePath = config('document.public_path') . 'uploads/';
    if ($file->storePubliclyAs($filePath, $fileName)) {
        $document = Document::create([
            'name' => $fileName,
            'title' => $fileName,
            'path' => $filePath,
            'ext' => $extension,
            'size' => $file->getSize(),
            'type' => 'upload',
            'created_by' => user()?->id,
            'updated_by' => user()?->id,
        ]);

        return response()->json(['item' => $document]);
    }
}
```

### Retrieve Documents

```php
use Bongo\Document\Models\Document;

// Get all documents
$documents = Document::all();

// Get by UUID (recommended for public access)
$document = Document::where('uuid', $uuid)->firstOrFail();

// Get by type
$pdfs = Document::ofType('upload')->get();

// Filter PDFs
$pdfs = Document::all()->filter(fn($doc) => $doc->isPdf());
```

### Access Document URL

```php
// Via accessor (automatic)
$url = $document->src;

// Via Storage facade
use Illuminate\Support\Facades\Storage;
$url = Storage::url($document->file_path);
```

### Download Document

```php
use Illuminate\Support\Facades\Storage;

return Storage::download($document->file_path);
```

### Delete Document

```php
use Illuminate\Support\Facades\Storage;

// Delete file from storage
if (Storage::exists($document->file_path)) {
    Storage::delete($document->file_path);
}

// Soft delete record
$document->delete();

// Restore soft deleted
$document->restore();

// Force delete
$document->forceDelete();
```

## Model

### Document Model

**Properties**:
- `id` - Auto-increment primary key
- `uuid` - Unique identifier for public access
- `name` - Filename with extension
- `title` - Display title
- `path` - Storage directory path
- `ext` - File extension
- `size` - File size in bytes
- `type` - Document type/category
- `sort_order` - Optional ordering
- `created_by`, `updated_by`, `deleted_by` - Audit fields

**Accessors**:
- `file_path` - Full path (path + name)
- `src` - Full URL via Storage::url()

**Scopes**:
- `ofType($type)` - Filter by type

**Methods**:
- `isPdf()` - Check if document is PDF

### Example Usage

```php
// Create document
$document = Document::create([
    'name' => 'document.pdf',
    'title' => 'My Document',
    'path' => 'public/uploads/',
    'ext' => 'pdf',
    'size' => 1024,
    'type' => 'upload',
]);

// Access URL
echo $document->src; // Full URL

// Check if PDF
if ($document->isPdf()) {
    // PDF-specific handling
}

// Filter by type
$uploads = Document::ofType('upload')->get();
```

## Frontend Integration

The package includes a Vue component for the admin interface:

```blade
@extends('framework::backend.layouts.app')

@section('content')
    <document-manager
        index-url="{{ route('api.document.index') }}"
        store-url="{{ route('api.document.store') }}"
        show-url="{{ route('backend.document.show') }}"
        download-url="{{ route('backend.document.download') }}"
        delete-url="{{ route('api.document.destroy') }}"
    ></document-manager>
@stop
```

## Validation

File uploads are validated via `StoreDocumentRequest`:

- **Required**: File must be present
- **Max size**: 10MB (10240 KB)
- **MIME types**: Configured via `document.allowed_document_types`

Default allowed types:
- Images: png, jpg, jpeg, gif
- PDFs: pdf
- OpenDocument: odt, odp, ods
- Microsoft Office: doc, docx, dot, ppt, pptx, pps, pot, xls, xlsx, xlm, xla, xlc, xlt, xlw
- Other: csv, txt, rtf

## Testing

```bash
# Run tests
composer test

# Or directly
vendor/bin/phpunit
```

## Code Quality

```bash
# Check code style
composer lint
vendor/bin/pint --test

# Fix code style
composer fix
vendor/bin/pint
```

## Security

- **UUID access**: Public routes use UUID instead of auto-increment IDs to prevent enumeration
- **File validation**: Strict MIME type and size validation
- **Authentication**: All routes require authentication
- **Authorization**: Backend routes require employee role
- **Soft deletes**: Documents can be recovered after deletion
- **Storage abstraction**: Supports secure remote storage (S3, etc.)

## Extending

### Add New Document Type

Update `config/document.php`:

```php
'allowed_document_types' => 'existing,types,mp4,mov',
```

### Add Model Scope

```php
// In Document model
public function scopeByExtension($query, string $ext)
{
    return $query->where('ext', $ext);
}

// Usage
Document::byExtension('pdf')->get();
```

### Add Route

```php
// In src/Routes/backend.php
Route::get('preview/{uuid}', [DocumentController::class, 'preview'])
    ->name('preview');
```

## Architecture

This package extends `Bongo\Framework\Providers\AbstractServiceProvider` which auto-bootstraps:

- **Config**: `src/Config/document.php`
- **Routes**: `src/Routes/api.php`, `src/Routes/backend.php`
- **Views**: `src/Views/` with namespace `document::`
- **Migrations**: Auto-discovered from `src/Migrations/`
- **Translations**: `src/Translations/` with namespace `document::`

For detailed architecture documentation, see `ARCHITECTURE.md`.

## Contributing

Please follow Laravel coding standards and run tests before submitting PRs.

## License

MIT License. See LICENSE file for details.

## Links

- **Bitbucket**: https://bitbucket.org/designtec/document
- **Private Composer Repository**: https://designtecpackages.co.uk
- **Author**: Stuart Elliott (stuart.elliott@bespokeuk.com)
- **Company**: Bespoke UK (https://bespokeuk.com)
