<?php

namespace Bongo\Document\Http\Controllers\Api;

use Bongo\Document\Actions\MakeSureUploadsDirectoryExists;
use Bongo\Document\Http\Requests\StoreDocumentRequest;
use Bongo\Document\Models\Document;
use Bongo\Framework\Http\Controllers\AbstractApiController;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class DocumentController extends AbstractApiController
{
    public function index(): JsonResponse
    {
        return response()->json([
            'items' => Document::all(),
        ]);
    }

    public function store(StoreDocumentRequest $request): JsonResponse
    {
        // Make sure the uploads directory exists
        (new MakeSureUploadsDirectoryExists())->execute();

        // Get the file, its size and extension
        $file = $request->file('file');
        $extension = $file->getClientOriginalExtension();
        $size = $file->getSize();

        // Create a unique filename
        $fileName = rtrim($file->getClientOriginalName(), $extension);
        $fileName = Str::slug($fileName);
        $fileName = $fileName.'-'.uniqid().'.'.$extension;

        // Get the upload path
        $filePath = config('document.public_path').'uploads/';

        // Upload the file and save
        if ($file->storePubliclyAs($filePath, $fileName)) {

            // Create the document
            $document = new Document();
            $document->name = $fileName;
            $document->title = $fileName;
            $document->path = $filePath;
            $document->ext = $extension;
            $document->size = $size;
            $document->type = 'upload';
            $document->created_by = user() ? user()->id : null;
            $document->updated_by = user() ? user()->id : null;
            $document->save();

            return response()->json(['item' => $document]);
        }

        return response()->json('error', 400);
    }

    public function destroy(Request $request): JsonResponse
    {
        $document = Document::findOrFail($request->id);

        // If the file exists in storage then delete it
        if (Storage::exists($document->file_path)) {
            Storage::delete($document->file_path);
        }

        // Delete the document
        $document->delete();

        return response()->json('Document deleted');
    }
}
