# GitHub Copilot Instructions - Bongo Enquiry Package

## Project Overview

This is the **bongo/enquiry** package for the Bongo Laravel framework. It provides a contact form system with spam protection (Honeypot + reCAPTCHA) and multi-recipient email notifications.

**Core Purpose**: Handle contact form submissions, validate input, prevent spam, and send notifications to users, administrators, and CRM systems.

**Technology Stack:**
- PHP 8.2+
- Laravel 10+
- Spatie Laravel Honeypot (spam protection)
- Google reCAPTCHA v3 (via bongo/captcha)
- Orchestra Testbench (testing)

## Key Classes and Relationships

### Controller Layer

**EnquiryController** (`src/Http/Controllers/Frontend/EnquiryController.php`)
```php
namespace Bongo\Enquiry\Http\Controllers\Frontend;

class EnquiryController extends AbstractController
{
    public function store(StoreEnquiryRequest $request): RedirectResponse
    {
        // Creates Enquiry DTO, sends 3 emails, redirects with success
    }
}
```

**Related Classes:**
- Uses `StoreEnquiryRequest` for validation
- Creates `Enquiry` DTO from request
- Sends `EnquiryMailable`, `AdminEnquiryMailable`, `CrmMailable`
- Relies on `setting()` helper for admin email

### Validation Layer

**StoreEnquiryRequest** (`src/Http/Requests/StoreEnquiryRequest.php`)
```php
namespace Bongo\Enquiry\Http\Requests;

class StoreEnquiryRequest extends FormRequest
{
    public function rules(): array
    {
        return [
            'email' => 'required|string|email:rfc,dns',
            'name' => 'sometimes|required',
            // ... other optional fields
            'captcha-response' => new Captcha(/* ... */),
        ];
    }
}
```

**Dependencies:**
- `Bongo\Captcha\Rules\Captcha` - reCAPTCHA validation
- `setting()->reCaptchaEnabled()` - Global reCAPTCHA toggle
- `config('enquiry.recaptcha.*')` - Package-specific settings

### Model Layer

**Enquiry** (`src/Models/Enquiry.php`)
```php
namespace Bongo\Enquiry\Models;

class Enquiry
{
    protected ?string $name = null;
    protected ?string $email = null;
    // ... other properties

    // Typed getters and setters with automatic formatting
    public function setName(?string $name): void { /* ucwords */ }
    public function setEmail(?string $email): void { /* strtolower */ }
}
```

**Design Pattern**: DTO (Data Transfer Object)
- No database persistence
- Auto-formatting in setters (capitalize names, lowercase emails)
- Nullable properties for optional fields

### Mailable Layer

**EnquiryMailable** (`src/Mailables/EnquiryMailable.php`)
```php
namespace Bongo\Enquiry\Mailables;

class EnquiryMailable extends Mailable
{
    protected Enquiry $enquiry;

    public function build(): EnquiryMailable
    {
        return $this
            ->from(config("settings.mail_from_address"), config("settings.mail_from_name"))
            ->replyTo(setting("client::company.email"), setting("client::company.name"))
            ->subject('Thank you for your enquiry')
            ->view('enquiry::mail.enquiry', ['enquiry' => $this->enquiry])
            ->text('enquiry::mail.enquiry_plain', ['enquiry' => $this->enquiry]);
    }
}
```

**AdminEnquiryMailable** - Similar structure, sends to admin with user as Reply-To
**CrmMailable** - Plain text only, sends to CRM system

### Service Provider

**EnquiryServiceProvider** (`src/EnquiryServiceProvider.php`)
```php
namespace Bongo\Enquiry;

class EnquiryServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'enquiry';
}
```

**Automatic Registration** (via parent class):
- Config: `config/enquiry.php`
- Routes: `routes/frontend.php`
- Views: `views/` → `enquiry::*`
- Translations: `translations/` → `trans('enquiry::*')`

## Class Relationships Diagram

```
┌─────────────────────────────────────────────────────────────┐
│                     HTTP Request Flow                        │
└─────────────────────────────────────────────────────────────┘

POST /contact/store
    ↓
[ProtectAgainstSpam Middleware] ← Spatie\Honeypot\ProtectAgainstSpam
    ↓
[StoreEnquiryRequest] ← Validates input + reCAPTCHA
    ↓
[EnquiryController::store()]
    ↓
Creates → [Enquiry DTO]
    ↓
Sends → [EnquiryMailable] → User confirmation
    ↓
Sends → [AdminEnquiryMailable] → Admin notification
    ↓
Sends → [CrmMailable] → CRM system (optional)
    ↓
redirect()->to(config('enquiry.success_url'))->success(trans('enquiry::frontend.store_success'))

┌─────────────────────────────────────────────────────────────┐
│                   Dependency Tree                            │
└─────────────────────────────────────────────────────────────┘

EnquiryServiceProvider
    └── AbstractServiceProvider (bongo/framework)
            ↓
        Auto-registers routes, views, config, translations

EnquiryController
    ├── AbstractController (bongo/framework)
    ├── StoreEnquiryRequest
    │       └── Captcha (bongo/captcha)
    ├── Enquiry (DTO)
    ├── EnquiryMailable
    ├── AdminEnquiryMailable
    └── CrmMailable
```

## Code Style Templates

### Adding a New Form Field

**Step 1: Update Enquiry DTO**
```php
// src/Models/Enquiry.php

protected ?string $companyName = null;

public function getCompanyName(): ?string
{
    return $this->companyName;
}

public function setCompanyName(?string $companyName): void
{
    if (! empty($companyName)) {
        $this->companyName = ucwords($companyName);
    }
}
```

**Step 2: Add Validation Rule**
```php
// src/Http/Requests/StoreEnquiryRequest.php

public function rules(): array
{
    return [
        // ... existing rules
        'company_name' => 'sometimes|required|string|max:255',
    ];
}
```

**Step 3: Update Controller**
```php
// src/Http/Controllers/Frontend/EnquiryController.php

public function store(StoreEnquiryRequest $request): RedirectResponse
{
    $enquiry = new Enquiry();
    // ... existing setters
    $enquiry->setCompanyName($request->get('company_name'));

    // ... send emails
}
```

**Step 4: Update Email Templates**
```blade
{{-- src/Views/mail/admin_enquiry.blade.php --}}

@if($enquiry->getCompanyName())
    <tr>
        <td style="{{ $cellStyle }}"><strong>Company:</strong></td>
        <td style="{{ $cellStyle }}">{{ $enquiry->getCompanyName() }}</td>
    </tr>
@endif
```

### Creating a New Mailable

```php
namespace Bongo\Enquiry\Mailables;

use Bongo\Enquiry\Models\Enquiry;
use Illuminate\Mail\Mailable;

class CustomMailable extends Mailable
{
    protected Enquiry $enquiry;

    public function __construct(Enquiry $enquiry)
    {
        $this->enquiry = $enquiry;
    }

    public function build(): self
    {
        return $this
            ->from(
                config("settings.mail_from_address"),
                config("settings.mail_from_name")
            )
            ->replyTo(
                $this->enquiry->getEmail(),
                $this->enquiry->getName()
            )
            ->subject('Your custom subject')
            ->view('enquiry::mail.custom', [
                'enquiry' => $this->enquiry,
            ])
            ->text('enquiry::mail.custom_plain', [
                'enquiry' => $this->enquiry,
            ]);
    }
}
```

### Email Template Structure

**HTML Template** (`src/Views/mail/*.blade.php`):
```blade
@extends('framework::mail.layouts.app')

@php
    $cellStyle = 'padding-top: 10px; padding-right: 0; padding-bottom: 10px; padding-left: 0; border:0';
@endphp

@section('content')
    @component('framework::mail.layouts.partials.section', ['hasBorder' => true])
        <h1 style="margin-top:0; margin-bottom:20px; font-size:26px;">
            Email Heading
        </h1>

        <table role="presentation" style="width: 100%;">
            <tr>
                <td style="{{ $cellStyle }}"><strong>Name:</strong></td>
                <td style="{{ $cellStyle }}">{{ $enquiry->getName() }}</td>
            </tr>
            <tr>
                <td style="{{ $cellStyle }}"><strong>Email:</strong></td>
                <td style="{{ $cellStyle }}">{{ $enquiry->getEmail() }}</td>
            </tr>
        </table>
    @endcomponent
@endsection
```

**Plain Text Template** (`src/Views/mail/*_plain.blade.php`):
```blade
Email Heading

Name: {{ $enquiry->getName() }}
Email: {{ $enquiry->getEmail() }}
Phone: {{ $enquiry->getPhone() }}

@if($enquiry->getMessage())
Message:
{{ $enquiry->getMessage() }}
@endif
```

### Writing Tests

```php
namespace Bongo\Enquiry\Tests\Feature;

use Bongo\Enquiry\Tests\TestCase;
use Illuminate\Support\Facades\Mail;

class EnquirySubmissionTest extends TestCase
{
    public function test_valid_enquiry_sends_emails(): void
    {
        Mail::fake();

        $response = $this->post(route('frontend.enquiry.store'), [
            'name' => 'John Doe',
            'email' => 'john@example.com',
            'phone' => '01234567890',
            'message' => 'Test enquiry message',
        ]);

        $response->assertRedirect(config('enquiry.success_url'));
        $response->assertSessionHas('success', trans('enquiry::frontend.store_success'));

        Mail::assertSent(EnquiryMailable::class);
        Mail::assertSent(AdminEnquiryMailable::class);
    }

    public function test_invalid_email_returns_validation_error(): void
    {
        $response = $this->post(route('frontend.enquiry.store'), [
            'email' => 'invalid-email',
            'name' => 'John Doe',
        ]);

        $response->assertSessionHasErrors('email');
    }
}
```

## Common Patterns

### Configuration Access
```php
// Accessing package config
config('enquiry.prefix')              // 'contact'
config('enquiry.success_url')         // '/thank-you'
config('enquiry.crm_email')           // null or email address
config('enquiry.recaptcha.enabled')   // true/false
config('enquiry.recaptcha.min_score') // 0.5

// Accessing framework settings
setting("client::company.email")      // Admin email
setting("client::company.name")       // Company name
setting("client::company.bcc_email")  // Optional BCC
setting()->reCaptchaEnabled()         // Global reCAPTCHA toggle
```

### Route Definition
```php
// src/Routes/frontend.php

Route::prefix(config('enquiry.prefix'))
    ->as('enquiry.')
    ->group(function () {
        Route::post('store', [EnquiryController::class, 'store'])
            ->middleware(ProtectAgainstSpam::class)
            ->name('store');
    });

// Results in:
// URL: POST /contact/store
// Name: frontend.enquiry.store
```

### Email Sending Pattern
```php
use Illuminate\Support\Facades\Mail;

// Send to single recipient
Mail::to($email)->send(new CustomMailable($enquiry));

// Send with BCC
Mail::to($primaryEmail)
    ->bcc($bccEmail)
    ->send(new CustomMailable($enquiry));

// Conditional sending
if ($shouldSend) {
    Mail::to($email)->send(new CustomMailable($enquiry));
}
```

### Validation Patterns
```php
// Optional field (only validated if present)
'field_name' => 'sometimes|required|string|max:255',

// Always required
'email' => 'required|string|email:rfc,dns',

// Custom rule with parameters
'captcha-response' => new Captcha(
    action: 'enquiry',
    minScore: 0.5,
    enabled: true,
),
```

## Configuration Options

### config/enquiry.php
```php
return [
    // URL prefix for routes (default: 'contact')
    // POST URL becomes: /{prefix}/store
    'prefix' => 'contact',

    // Show HTTP referrer in emails (default: false)
    'show_referrer' => false,

    // Optional CRM email for notifications
    'crm_email' => null,

    // Redirect URL after successful submission
    'success_url' => '/thank-you',

    // reCAPTCHA configuration
    'recaptcha' => [
        'enabled' => true,      // Enable/disable at package level
        'min_score' => 0.5,     // Threshold: 0.1 (lenient) to 1.0 (strict)
    ],
];
```

## Integration Requirements

### Required Dependencies
- **bongo/framework** - Base abstractions and helpers
- **bongo/captcha** - reCAPTCHA validation rule
- **spatie/laravel-honeypot** - Spam protection middleware

### Expected Settings (via setting() helper)
```php
setting("client::company.email")       // Required: Admin notification email
setting("client::company.name")        // Required: Company name for emails
setting("client::company.bcc_email")   // Optional: BCC recipient
setting()->reCaptchaEnabled()          // Required: Global reCAPTCHA toggle
```

### Expected Config Values
```php
config("settings.mail_from_address")   // Required: From email for all mailables
config("settings.mail_from_name")      // Required: From name for all mailables
```

### Framework Extensions
```php
// Controller uses success() macro from bongo/framework
redirect()->to('/thank-you')->success('Message');

// Translation helper
trans('enquiry::frontend.store_success');
```

## Testing Commands

```bash
# Run all tests
vendor/bin/phpunit

# Run specific test
vendor/bin/phpunit --filter test_valid_enquiry_sends_emails

# Run with coverage
vendor/bin/phpunit --coverage-html coverage

# Code style check
vendor/bin/pint --test

# Code style fix
vendor/bin/pint

# Static analysis
vendor/bin/phpstan analyse
```

## File Naming Conventions

- **Controllers**: `{Entity}Controller.php` (singular)
- **Mailables**: `{Description}Mailable.php` (e.g., `AdminEnquiryMailable`)
- **Requests**: `{Action}{Entity}Request.php` (e.g., `StoreEnquiryRequest`)
- **Models**: `{Entity}.php` (singular, e.g., `Enquiry`)
- **Views**: Snake case (e.g., `admin_enquiry.blade.php`, `enquiry_plain.blade.php`)
- **Routes**: `frontend.php`, `backend.php`, `api.php`, `web.php`, `custom.php`

## Common Gotchas

1. **reCAPTCHA double-check**: Both global setting (`setting()->reCaptchaEnabled()`) AND package config (`config('enquiry.recaptcha.enabled')`) must be true
2. **CRM email is optional**: Always check `config('enquiry.crm_email')` exists before sending
3. **Enquiry is a DTO**: Not an Eloquent model - no database persistence
4. **Route naming**: All routes automatically get `frontend.` prefix from AbstractServiceProvider
5. **Email views**: Must provide both HTML and plain text versions
6. **Setting() helper**: Requires bongo/framework and proper setting provider configuration
