# Bongo Enquiry Package

[![Latest Version](https://img.shields.io/badge/version-3.0-blue.svg)](https://designtecpackages.co.uk)
[![PHP Version](https://img.shields.io/badge/php-%3E%3D8.2-8892BF.svg)](https://php.net/)
[![Laravel Version](https://img.shields.io/badge/laravel-10.x-FF2D20.svg)](https://laravel.com)
[![License](https://img.shields.io/badge/license-MIT-green.svg)](LICENSE)

A Laravel package providing contact form functionality with spam protection (Honeypot + reCAPTCHA) and multi-recipient email notifications.

## Features

- **Single POST Route**: Simple `/contact/store` endpoint for form submissions
- **Spam Protection**: Two-layer protection using Spatie Honeypot + Google reCAPTCHA v3
- **Multi-Recipient Emails**: Automatically sends to user, admin, and optional CRM
- **Flexible Validation**: RFC/DNS email validation with configurable optional fields
- **No Database Required**: Stateless DTO-based architecture
- **Extensible**: Easy to add custom fields, mailables, or validation rules
- **Framework Integration**: Extends `bongo/framework` for automatic bootstrapping

## Requirements

- PHP 8.2 or higher
- Laravel 10.x or higher
- `bongo/framework` ^3.0
- `spatie/laravel-honeypot` ^4.0

## Installation

### Step 1: Install via Composer

```bash
composer require bongo/enquiry
```

### Step 2: Laravel Auto-Discovery

The service provider will be automatically discovered by Laravel. No need to manually register it in `config/app.php`.

### Step 3: Publish Configuration (Optional)

```bash
php artisan vendor:publish --provider="Bongo\Enquiry\EnquiryServiceProvider" --tag="config"
```

This publishes `config/enquiry.php` where you can customize:
- Route prefix (default: `contact`)
- Success redirect URL (default: `/thank-you`)
- reCAPTCHA settings
- CRM email address

### Step 4: Configure Environment Variables

Add to your `.env` file:

```env
# Optional: CRM email for enquiry notifications
ENQUIRY_CRM_EMAIL=crm@example.com
```

### Step 5: Configure Required Settings

Ensure these settings are configured via your `setting()` helper:

```php
setting("client::company.email")    // Admin notification email
setting("client::company.name")     // Company name for emails
```

## Usage

### Basic Form Submission

Submit a POST request to `/contact/store` with the following fields:

```html
<form method="POST" action="{{ route('frontend.enquiry.store') }}">
    @csrf
    <x-honeypot />

    <input type="text" name="name" required>
    <input type="email" name="email" required>
    <textarea name="message"></textarea>

    <!-- reCAPTCHA v3 (if enabled) -->
    <input type="hidden" name="captcha-response" id="captcha-response">

    <button type="submit">Send Enquiry</button>
</form>
```

### Available Form Fields

**Required:**
- `email` - User's email address (validated with RFC + DNS)

**Optional:**
- `name` - Full name
- `first_name` - First name
- `last_name` - Last name
- `phone` - Phone number
- `address` - Street address
- `postcode` - Postal code
- `message` - Message content

All optional fields use `sometimes|required` validation (validated only if present).

### Route Information

**Route Name**: `frontend.enquiry.store`
**HTTP Method**: `POST`
**Default URL**: `/contact/store`
**Middleware**: `Spatie\Honeypot\ProtectAgainstSpam`

### Email Flow

Upon successful submission, three emails are sent:

1. **User Confirmation** (`EnquiryMailable`)
   - To: User's email address
   - Subject: "Thank you for your enquiry"
   - Reply-To: Admin email

2. **Admin Notification** (`AdminEnquiryMailable`)
   - To: `setting("client::company.email")`
   - Subject: "New enquiry from: {user_email}"
   - Reply-To: User's email/name

3. **CRM Notification** (`CrmMailable`) - *Optional*
   - To: `config('enquiry.crm_email')`
   - Subject: "New enquiry from: {user_email}"
   - Format: Plain text only

## Configuration

### Config File: `config/enquiry.php`

```php
return [
    // URL prefix for routes
    'prefix' => 'contact',

    // Show HTTP referrer in emails
    'show_referrer' => false,

    // Optional CRM email address
    'crm_email' => env('ENQUIRY_CRM_EMAIL'),

    // Redirect URL after successful submission
    'success_url' => '/thank-you',

    // Google reCAPTCHA v3 configuration
    'recaptcha' => [
        'enabled' => true,           // Enable/disable at package level
        'min_score' => 0.5,          // Score threshold (0.1 = lenient, 1.0 = strict)
    ],
];
```

### Customizing Routes

Change the route prefix in `config/enquiry.php`:

```php
'prefix' => 'contact-us',  // Changes URL to /contact-us/store
```

### Customizing Success URL

Set the redirect URL after successful submission:

```php
'success_url' => '/contact/thank-you',
```

### Configuring reCAPTCHA

Adjust the score threshold for more or less strict spam filtering:

```php
'recaptcha' => [
    'enabled' => true,
    'min_score' => 0.3,  // More lenient (accepts more submissions)
],
```

**Note**: Both `setting()->reCaptchaEnabled()` (global) AND `config('enquiry.recaptcha.enabled')` (package) must be `true` for reCAPTCHA to be active.

## Spam Protection

### Two-Layer Protection

**Layer 1: Honeypot**
- Middleware from `spatie/laravel-honeypot`
- Detects bots filling hidden fields
- Validates submission time (prevents too-fast submissions)
- Returns HTTP 419 if spam detected

**Layer 2: reCAPTCHA v3**
- Google reCAPTCHA score analysis (0.0-1.0)
- Configurable score threshold
- No CAPTCHA challenges for users
- Validation rule from `bongo/captcha` package

## Extending the Package

### Adding a New Form Field

**Step 1**: Add property to `Enquiry` DTO:

```php
// src/Models/Enquiry.php

protected ?string $companyName = null;

public function getCompanyName(): ?string
{
    return $this->companyName;
}

public function setCompanyName(?string $companyName): void
{
    if (! empty($companyName)) {
        $this->companyName = ucwords($companyName);
    }
}
```

**Step 2**: Add validation rule:

```php
// src/Http/Requests/StoreEnquiryRequest.php

'company_name' => 'sometimes|required|string|max:255',
```

**Step 3**: Update controller:

```php
// src/Http/Controllers/Frontend/EnquiryController.php

$enquiry->setCompanyName($request->get('company_name'));
```

**Step 4**: Update email templates:

```blade
{{-- src/Views/mail/admin_enquiry.blade.php --}}

@if($enquiry->getCompanyName())
    <tr>
        <td><strong>Company:</strong></td>
        <td>{{ $enquiry->getCompanyName() }}</td>
    </tr>
@endif
```

### Creating a Custom Mailable

```php
namespace Bongo\Enquiry\Mailables;

use Bongo\Enquiry\Models\Enquiry;
use Illuminate\Mail\Mailable;

class CustomMailable extends Mailable
{
    protected Enquiry $enquiry;

    public function __construct(Enquiry $enquiry)
    {
        $this->enquiry = $enquiry;
    }

    public function build(): self
    {
        return $this
            ->from(config("settings.mail_from_address"), config("settings.mail_from_name"))
            ->subject('Custom notification')
            ->view('enquiry::mail.custom', ['enquiry' => $this->enquiry]);
    }
}
```

## Testing

### Running Tests

```bash
# Install dependencies
composer install

# Run all tests
vendor/bin/phpunit

# Run with coverage
vendor/bin/phpunit --coverage-html coverage
```

### Example Test

```php
use Bongo\Enquiry\Tests\TestCase;
use Illuminate\Support\Facades\Mail;

class EnquiryTest extends TestCase
{
    public function test_enquiry_sends_emails(): void
    {
        Mail::fake();

        $response = $this->post(route('frontend.enquiry.store'), [
            'name' => 'John Doe',
            'email' => 'john@example.com',
            'message' => 'Test message',
        ]);

        $response->assertRedirect(config('enquiry.success_url'));
        Mail::assertSent(EnquiryMailable::class);
        Mail::assertSent(AdminEnquiryMailable::class);
    }
}
```

## Code Quality

### Laravel Pint (Code Style)

```bash
# Check code style
vendor/bin/pint --test

# Fix code style automatically
vendor/bin/pint
```

### Static Analysis

```bash
vendor/bin/phpstan analyse
```

## Troubleshooting

### Emails Not Sending

1. Verify mail configuration: `config('settings.mail_from_address')`
2. Check admin email: `setting('client::company.email')`
3. Review logs: `storage/logs/laravel.log`
4. Test mail driver: `php artisan tinker` → `Mail::raw('test', fn($m) => $m->to('test@example.com'))`

### reCAPTCHA Not Working

1. Verify global setting: `setting()->reCaptchaEnabled()` returns `true`
2. Check package config: `config('enquiry.recaptcha.enabled')` is `true`
3. Ensure `bongo/captcha` package is installed
4. Verify reCAPTCHA API keys are configured

### Route Not Found

1. Clear route cache: `php artisan route:clear`
2. List routes: `php artisan route:list | grep enquiry`
3. Verify service provider is registered (auto-discovered)

### Honeypot Blocking Valid Users

1. Review Honeypot config: `config/honeypot.php`
2. Adjust `amount_of_seconds` threshold (default: 1 second)
3. Ensure honeypot field is properly rendered in form

## Documentation

- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Detailed architecture documentation with diagrams
- **[CLAUDE.md](CLAUDE.md)** - Quick reference guide for Claude Code
- **[.cursorrules](.cursorrules)** - Coding conventions and common tasks
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)** - Code style templates

## Dependencies

| Package | Version | Purpose |
|---------|---------|---------|
| `php` | `>=8.2` | Runtime requirement |
| `illuminate/contracts` | `^10.0` | Laravel framework contracts |
| `bongo/framework` | `^3.0` | Base abstractions and helpers |
| `spatie/laravel-honeypot` | `^4.0` | Spam protection middleware |

**External Dependencies:**
- `bongo/captcha` - reCAPTCHA validation rule
- Framework settings via `setting()` helper

## Package Structure

```
cms/enquiry/
├── src/
│   ├── Config/enquiry.php                    # Configuration
│   ├── Http/
│   │   ├── Controllers/Frontend/
│   │   │   └── EnquiryController.php         # Form handler
│   │   └── Requests/
│   │       └── StoreEnquiryRequest.php       # Validation
│   ├── Mailables/                            # Email classes
│   │   ├── EnquiryMailable.php
│   │   ├── AdminEnquiryMailable.php
│   │   └── CrmMailable.php
│   ├── Models/Enquiry.php                    # DTO
│   ├── Routes/frontend.php                   # Route definitions
│   ├── Translations/en/frontend.php          # Translations
│   ├── Views/mail/*.blade.php                # Email templates
│   └── EnquiryServiceProvider.php            # Service provider
├── tests/
│   └── TestCase.php                          # Test base class
├── composer.json
├── phpunit.xml
└── README.md
```

## Contributing

This package is part of the Bongo framework ecosystem. For bugs or feature requests, please contact the package maintainer.

## Security

If you discover any security-related issues, please email stuart.elliott@bespokeuk.com instead of using the issue tracker.

## Credits

- **Stuart Elliott** - [Bespoke UK](https://bespokeuk.com)
- All contributors

## License

The MIT License (MIT). Please see [LICENSE](LICENSE) for more information.

## Links

- **Repository**: [https://bitbucket.org/designtec/enquiry](https://bitbucket.org/designtec/enquiry)
- **Package Registry**: [https://designtecpackages.co.uk](https://designtecpackages.co.uk)
- **Author Website**: [https://bespokeuk.com](https://bespokeuk.com)

---

**Version**: 3.0
**Last Updated**: 2026-01-19
