# Bongo Enquiry Package - Architecture Documentation

## Table of Contents
1. [Overview](#overview)
2. [Directory Structure](#directory-structure)
3. [Request Lifecycle](#request-lifecycle)
4. [Class Architecture](#class-architecture)
5. [Email Flow](#email-flow)
6. [Spam Protection](#spam-protection)
7. [Extension Points](#extension-points)
8. [Adding New Features](#adding-new-features)

---

## Overview

The **bongo/enquiry** package is a Laravel contact form system designed for simplicity and flexibility. It handles form submissions, validates input, prevents spam, and sends multi-recipient email notifications.

**Core Principles:**
- **Stateless DTO Model**: `Enquiry` is a data transfer object with no database persistence
- **Framework Integration**: Extends `bongo/framework` abstractions for automatic bootstrapping
- **Multi-layer Spam Protection**: Combines Honeypot and reCAPTCHA
- **Flexible Email Recipients**: User confirmation, admin notification, optional CRM integration

**Technology Stack:**
- PHP 8.2+
- Laravel 10+
- Spatie Laravel Honeypot 4.0
- Google reCAPTCHA v3 (via bongo/captcha)

---

## Directory Structure

```
cms/enquiry/
├── src/
│   ├── Config/
│   │   └── enquiry.php                           # Package configuration
│   │
│   ├── Http/
│   │   ├── Controllers/
│   │   │   └── Frontend/
│   │   │       └── EnquiryController.php         # POST /contact/store handler
│   │   └── Requests/
│   │       └── StoreEnquiryRequest.php           # Validation + reCAPTCHA
│   │
│   ├── Mailables/
│   │   ├── EnquiryMailable.php                   # User confirmation email
│   │   ├── AdminEnquiryMailable.php              # Admin notification email
│   │   └── CrmMailable.php                       # CRM plain text email
│   │
│   ├── Models/
│   │   └── Enquiry.php                           # DTO with getters/setters
│   │
│   ├── Routes/
│   │   └── frontend.php                          # Public route definitions
│   │
│   ├── Translations/
│   │   └── en/
│   │       └── frontend.php                      # Success message translation
│   │
│   ├── Views/
│   │   └── mail/
│   │       ├── enquiry.blade.php                 # User HTML email
│   │       ├── enquiry_plain.blade.php           # User plain text
│   │       ├── admin_enquiry.blade.php           # Admin HTML email
│   │       ├── admin_enquiry_plain.blade.php     # Admin plain text
│   │       └── crm_plain.blade.php               # CRM plain text
│   │
│   └── EnquiryServiceProvider.php                # Service provider
│
├── tests/
│   └── TestCase.php                              # Orchestra Testbench base
│
├── composer.json                                 # Dependencies and autoload
├── phpunit.xml                                   # Test configuration
└── README.md                                     # Package documentation
```

### Key Directories

**Config (`src/Config/`)**:
- Single config file defining route prefix, reCAPTCHA settings, CRM email, success URL

**Controllers (`src/Http/Controllers/Frontend/`)**:
- Single action controller for handling form submissions
- Only frontend controller (no backend/admin interface)

**Mailables (`src/Mailables/`)**:
- Three separate mailable classes for different recipients
- Each builds its own email structure with from/reply-to/subject

**Models (`src/Models/`)**:
- DTO pattern (not Eloquent model)
- No database interaction
- Properties with typed getters/setters

**Routes (`src/Routes/`)**:
- Only `frontend.php` (no backend, API, or custom routes)
- Single POST route with Honeypot middleware

**Views (`src/Views/mail/`)**:
- Blade templates for HTML and plain text emails
- Extend framework mail layout

---

## Request Lifecycle

### Full Request Flow

```
┌─────────────────────────────────────────────────────────────────────┐
│                        HTTP POST Request                             │
│                     POST /contact/store                              │
└───────────────────────────────┬─────────────────────────────────────┘
                                ↓
┌─────────────────────────────────────────────────────────────────────┐
│                    1. Route Middleware                               │
│         [Spatie\Honeypot\ProtectAgainstSpam]                        │
│                                                                      │
│  - Checks for honeypot field presence                               │
│  - Validates submission time (not too fast)                         │
│  - Returns 419 if spam detected                                     │
└───────────────────────────────┬─────────────────────────────────────┘
                                ↓
┌─────────────────────────────────────────────────────────────────────┐
│                    2. Form Request Validation                        │
│              [StoreEnquiryRequest::rules()]                         │
│                                                                      │
│  - Validates email (required, RFC/DNS)                              │
│  - Validates optional fields (name, phone, message, etc.)           │
│  - Validates reCAPTCHA via Captcha rule                             │
│  - Returns 422 with errors if validation fails                      │
└───────────────────────────────┬─────────────────────────────────────┘
                                ↓
┌─────────────────────────────────────────────────────────────────────┐
│                 3. Controller Processing                             │
│           [EnquiryController::store()]                              │
│                                                                      │
│  Step 1: Create Enquiry DTO                                         │
│  Step 2: Populate from validated request data                       │
│  Step 3: Send EnquiryMailable to user                               │
│  Step 4: Send AdminEnquiryMailable to admin                         │
│  Step 5: Send CrmMailable to CRM (if configured)                    │
│  Step 6: Redirect to success URL with flash message                 │
└───────────────────────────────┬─────────────────────────────────────┘
                                ↓
┌─────────────────────────────────────────────────────────────────────┐
│                    4. Email Dispatch                                 │
│              [Laravel Mail Queue]                                    │
│                                                                      │
│  EnquiryMailable:                                                   │
│    To: User's email                                                 │
│    From: settings.mail_from_address                                 │
│    Reply-To: Admin email                                            │
│    Subject: "Thank you for your enquiry"                            │
│                                                                      │
│  AdminEnquiryMailable:                                              │
│    To: setting("client::company.email")                             │
│    From: settings.mail_from_address                                 │
│    Reply-To: User's email/name                                      │
│    Subject: "New enquiry from: {email}"                             │
│                                                                      │
│  CrmMailable (optional):                                            │
│    To: config('enquiry.crm_email')                                  │
│    From: settings.mail_from_address                                 │
│    Subject: "New enquiry from: {email}"                             │
└───────────────────────────────┬─────────────────────────────────────┘
                                ↓
┌─────────────────────────────────────────────────────────────────────┐
│                    5. Success Response                               │
│                                                                      │
│  Redirect to: config('enquiry.success_url')  [default: /thank-you]  │
│  Flash message: trans('enquiry::frontend.store_success')            │
│  HTTP Status: 302 Redirect                                          │
└─────────────────────────────────────────────────────────────────────┘
```

### Detailed Request Steps

#### Step 1: Route Middleware (Honeypot)

**File**: `src/Routes/frontend.php:10-12`

```php
Route::post('store', [EnquiryController::class, 'store'])
    ->middleware(ProtectAgainstSpam::class)
    ->name('store');
```

The Honeypot middleware (`Spatie\Honeypot\ProtectAgainstSpam`) runs first:
- Checks for hidden honeypot field in form data
- Validates submission wasn't too fast (configurable time threshold)
- Returns HTTP 419 (Page Expired) if spam detected

#### Step 2: Form Request Validation

**File**: `src/Http/Requests/StoreEnquiryRequest.php`

```php
public function rules(): array
{
    $recaptchaEnabled = setting()->reCaptchaEnabled() && config('enquiry.recaptcha.enabled');
    $recaptchaMinScore = config('enquiry.recaptcha.min_score', 0.5);

    return [
        'email' => 'required|string|email:rfc,dns',
        'name' => 'sometimes|required',
        // ... other optional fields

        'captcha-response' => new Captcha(
            action: 'enquiry',
            minScore: $recaptchaMinScore,
            enabled: $recaptchaEnabled,
        ),
    ];
}
```

**Validation Logic:**
- `email` is always required with RFC and DNS validation
- Other fields (`name`, `first_name`, `last_name`, `phone`, `address`, `postcode`, `message`) are optional but validated if present
- reCAPTCHA validation runs if both global setting and package config are enabled
- Returns HTTP 422 with error bag if validation fails

#### Step 3: Controller Processing

**File**: `src/Http/Controllers/Frontend/EnquiryController.php:20-51`

```php
public function store(StoreEnquiryRequest $request): RedirectResponse
{
    // Create DTO and populate from request
    $enquiry = new Enquiry();
    $enquiry->setName($request->get('name'));
    $enquiry->setEmail($request->get('email'));
    // ... set other fields

    // Send three emails
    Mail::to($enquiry->getEmail())->send(new EnquiryMailable($enquiry));
    Mail::to(setting("client::company.email"))->send(new AdminEnquiryMailable($enquiry));

    if (config('enquiry.crm_email')) {
        Mail::to(config('enquiry.crm_email'))->send(new CrmMailable($enquiry));
    }

    // Redirect with success message
    return redirect()
        ->to(config('enquiry.success_url'))
        ->success(trans('enquiry::frontend.store_success'));
}
```

**Processing Steps:**
1. Create new `Enquiry` DTO instance
2. Populate DTO using setters (automatic formatting applied)
3. Send confirmation email to user
4. Send notification email to admin
5. Conditionally send notification to CRM system
6. Redirect to success URL with flash message

#### Step 4: Email Dispatch

Emails are dispatched synchronously using Laravel's Mail facade. Each mailable:
- Builds its own configuration (from, reply-to, subject)
- References appropriate Blade views
- Passes `$enquiry` DTO to views

#### Step 5: Success Response

Controller returns redirect response:
- Target URL: `config('enquiry.success_url')` (default: `/thank-you`)
- Flash message: `trans('enquiry::frontend.store_success')` (default: "Thank you for your enquiry!")
- Uses `success()` macro from bongo/framework

---

## Class Architecture

### Class Diagram

```
┌──────────────────────────────────────────────────────────────────────┐
│                         Service Provider Layer                        │
└──────────────────────────────────────────────────────────────────────┘

    Bongo\Framework\Providers\AbstractServiceProvider
                            ↑
                            │ extends
                            │
                  EnquiryServiceProvider
                            │
                            │ registers (via parent boot())
                            ├─→ Config (enquiry.php)
                            ├─→ Routes (frontend.php)
                            ├─→ Views (enquiry::*)
                            └─→ Translations (enquiry::*)


┌──────────────────────────────────────────────────────────────────────┐
│                          Controller Layer                             │
└──────────────────────────────────────────────────────────────────────┘

    Bongo\Framework\Http\Controllers\AbstractController
                            ↑
                            │ extends
                            │
                     EnquiryController
                            │
                            │ uses
                            ├─→ StoreEnquiryRequest (validation)
                            ├─→ Enquiry (DTO)
                            ├─→ EnquiryMailable
                            ├─→ AdminEnquiryMailable
                            └─→ CrmMailable


┌──────────────────────────────────────────────────────────────────────┐
│                         Validation Layer                              │
└──────────────────────────────────────────────────────────────────────┘

    Illuminate\Foundation\Http\FormRequest
                            ↑
                            │ extends
                            │
                  StoreEnquiryRequest
                            │
                            │ uses
                            └─→ Bongo\Captcha\Rules\Captcha


┌──────────────────────────────────────────────────────────────────────┐
│                            Model Layer                                │
└──────────────────────────────────────────────────────────────────────┘

                         Enquiry (DTO)
                            │
                            │ properties
                            ├─→ ?string $name
                            ├─→ ?string $firstName
                            ├─→ ?string $lastName
                            ├─→ ?string $email
                            ├─→ ?string $phone
                            ├─→ ?string $message
                            ├─→ ?string $address
                            └─→ ?string $postcode
                            │
                            │ methods (for each property)
                            ├─→ getName(): ?string
                            ├─→ setName(?string): void  [ucwords()]
                            ├─→ getEmail(): ?string
                            └─→ setEmail(?string): void [strtolower()]


┌──────────────────────────────────────────────────────────────────────┐
│                           Mailable Layer                              │
└──────────────────────────────────────────────────────────────────────┘

    Illuminate\Mail\Mailable
                ↑
                │ extends
                ├─────────────────┬──────────────────┐
                │                 │                  │
        EnquiryMailable    AdminEnquiryMailable    CrmMailable
                │                 │                  │
                │ composition     │ composition      │ composition
                └─→ Enquiry       └─→ Enquiry        └─→ Enquiry
                │                 │                  │
                │ views           │ views            │ views
                ├─→ enquiry::     ├─→ enquiry::      └─→ enquiry::
                │   mail.enquiry  │   mail.admin_        mail.crm_plain
                └─→ enquiry::     │   _enquiry
                    mail.enquiry_ └─→ enquiry::
                    plain             mail.admin_
                                      _enquiry_plain
```

### Class Responsibilities

#### EnquiryServiceProvider

**File**: `src/EnquiryServiceProvider.php`

**Responsibility**: Bootstrap package resources

**Properties:**
- `protected string $module = 'enquiry'` - Defines module name for auto-registration

**Inherited Behavior** (from `AbstractServiceProvider`):
- Registers `config/enquiry.php` → accessible via `config('enquiry.*')`
- Registers routes from `routes/frontend.php` with `frontend.*` naming
- Registers views from `views/` with `enquiry::` namespace
- Registers translations from `translations/` with `enquiry::` namespace

#### EnquiryController

**File**: `src/Http/Controllers/Frontend/EnquiryController.php`

**Responsibility**: Handle form submission and coordinate email sending

**Methods:**
```php
public function store(StoreEnquiryRequest $request): RedirectResponse
```

**Dependencies:**
- `StoreEnquiryRequest` - Validated form data
- `Enquiry` - DTO for data transfer
- `EnquiryMailable`, `AdminEnquiryMailable`, `CrmMailable` - Email classes
- `Mail` facade - Email dispatch
- `setting()` helper - Access admin email configuration
- `config()` helper - Access package configuration

**Logic Flow:**
1. Create DTO from validated request
2. Send user confirmation email
3. Send admin notification email
4. Conditionally send CRM email
5. Redirect with success message

#### StoreEnquiryRequest

**File**: `src/Http/Requests/StoreEnquiryRequest.php`

**Responsibility**: Validate form input and reCAPTCHA

**Methods:**
```php
public function rules(): array
```

**Validation Rules:**
- `email`: Required, string, RFC + DNS validation
- `name`, `first_name`, `last_name`, `phone`, `address`, `postcode`, `message`: Optional (`sometimes|required`)
- `captcha-response`: Custom Captcha rule with configurable score threshold

**Configuration Dependencies:**
- `setting()->reCaptchaEnabled()` - Global reCAPTCHA toggle
- `config('enquiry.recaptcha.enabled')` - Package-level toggle
- `config('enquiry.recaptcha.min_score')` - Score threshold (0.1-1.0)

#### Enquiry (DTO)

**File**: `src/Models/Enquiry.php`

**Responsibility**: Data transfer object with type-safe getters/setters

**Design Pattern**: DTO (Data Transfer Object)
- No database persistence
- No validation logic
- Automatic formatting in setters

**Properties:**
```php
protected ?string $name = null;
protected ?string $firstName = null;
protected ?string $lastName = null;
protected ?string $email = null;
protected ?string $phone = null;
protected ?string $message = null;
protected ?string $address = null;
protected ?string $postcode = null;
```

**Formatting Rules:**
- `setName()`: `ucwords()` - Capitalizes each word
- `setFirstName()`, `setLastName()`: `ucfirst()` - Capitalizes first letter
- `setEmail()`: `strtolower()` - Lowercases entire string
- Other setters: No formatting

**Null Safety**: All setters check `!empty()` before assignment

#### EnquiryMailable

**File**: `src/Mailables/EnquiryMailable.php`

**Responsibility**: User confirmation email

**Configuration:**
- **To**: User's email address
- **From**: `config("settings.mail_from_address")` + name
- **Reply-To**: Admin email (for user responses)
- **BCC**: Optional `setting("client::company.bcc_email")`
- **Subject**: "Thank you for your enquiry"
- **Views**: `enquiry::mail.enquiry` (HTML) + `enquiry::mail.enquiry_plain` (text)

**Method:**
```php
public function build(): EnquiryMailable
```

#### AdminEnquiryMailable

**File**: `src/Mailables/AdminEnquiryMailable.php`

**Responsibility**: Admin notification email

**Configuration:**
- **To**: `setting("client::company.email")`
- **From**: `config("settings.mail_from_address")` + name
- **Reply-To**: User's email/name (for easy admin replies)
- **Subject**: "New enquiry from: {user_email}"
- **Views**: `enquiry::mail.admin_enquiry` (HTML) + `enquiry::mail.admin_enquiry_plain` (text)

**Method:**
```php
public function build(): AdminEnquiryMailable
```

#### CrmMailable

**File**: `src/Mailables/CrmMailable.php`

**Responsibility**: CRM system notification (plain text only)

**Configuration:**
- **To**: `config('enquiry.crm_email')`
- **From**: `config("settings.mail_from_address")` + name
- **Subject**: "New enquiry from: {user_email}"
- **Views**: `enquiry::mail.crm_plain` (text only, no HTML)

**Method:**
```php
public function build(): CrmMailable
```

---

## Email Flow

### Email Dispatch Sequence

```
┌───────────────────────────────────────────────────────────────────┐
│                     Email Dispatch Flow                            │
└───────────────────────────────────────────────────────────────────┘

EnquiryController::store()
        │
        ├─→ [1] Mail::to($enquiry->getEmail())
        │            ->send(new EnquiryMailable($enquiry))
        │                    │
        │                    └─→ EnquiryMailable::build()
        │                            │
        │                            ├─→ From: settings.mail_from_address
        │                            ├─→ Reply-To: admin email
        │                            ├─→ BCC: company.bcc_email (if set)
        │                            ├─→ Subject: "Thank you for your enquiry"
        │                            ├─→ View: enquiry::mail.enquiry
        │                            └─→ Text: enquiry::mail.enquiry_plain
        │
        ├─→ [2] Mail::to(setting("client::company.email"))
        │            ->send(new AdminEnquiryMailable($enquiry))
        │                    │
        │                    └─→ AdminEnquiryMailable::build()
        │                            │
        │                            ├─→ From: settings.mail_from_address
        │                            ├─→ Reply-To: user email/name
        │                            ├─→ Subject: "New enquiry from: {email}"
        │                            ├─→ View: enquiry::mail.admin_enquiry
        │                            └─→ Text: enquiry::mail.admin_enquiry_plain
        │
        └─→ [3] if (config('enquiry.crm_email'))
                    Mail::to(config('enquiry.crm_email'))
                        ->send(new CrmMailable($enquiry))
                                │
                                └─→ CrmMailable::build()
                                        │
                                        ├─→ From: settings.mail_from_address
                                        ├─→ Subject: "New enquiry from: {email}"
                                        └─→ Text: enquiry::mail.crm_plain
```

### Email Template Structure

All email templates extend the framework layout:

```blade
@extends('framework::mail.layouts.app')

@section('content')
    @component('framework::mail.layouts.partials.section', ['hasBorder' => true])
        <!-- Email content here -->
    @endcomponent
@endsection
```

**Template Hierarchy:**
```
framework::mail.layouts.app (provided by bongo/framework)
    └─→ @section('content')
            └─→ framework::mail.layouts.partials.section
                    └─→ Email content (headings, tables, text)
```

---

## Spam Protection

### Two-Layer Protection Strategy

```
┌───────────────────────────────────────────────────────────────────┐
│                       Spam Protection Layers                       │
└───────────────────────────────────────────────────────────────────┘

Layer 1: Honeypot (Middleware)
    │
    ├─→ Hidden form field check
    │   └─→ Bots typically fill all fields
    │
    ├─→ Submission time threshold
    │   └─→ Bots submit too quickly
    │
    └─→ If spam detected: HTTP 419 (Page Expired)

Layer 2: reCAPTCHA v3 (Validation Rule)
    │
    ├─→ Google reCAPTCHA score analysis (0.0-1.0)
    │   └─→ Score based on user interaction patterns
    │
    ├─→ Configurable threshold (default: 0.5)
    │   ├─→ 0.1 = Very lenient (more false positives)
    │   └─→ 1.0 = Very strict (more false negatives)
    │
    └─→ If score too low: Validation error
```

### Honeypot Configuration

**Middleware**: `Spatie\Honeypot\ProtectAgainstSpam`

**Applied to**: `Route::post('store')` in `frontend.php:11`

**Configuration File**: `config/honeypot.php` (Laravel application level)

**Key Settings:**
```php
return [
    'name_field_name' => 'your_name_field',  // Hidden field name
    'valid_from_field_name' => 'valid_from', // Timestamp field name
    'amount_of_seconds' => 1,                 // Minimum submission time (seconds)
];
```

### reCAPTCHA Configuration

**Validation Rule**: `Bongo\Captcha\Rules\Captcha`

**Applied in**: `StoreEnquiryRequest::rules():25-29`

**Configuration:**
```php
'captcha-response' => new Captcha(
    action: 'enquiry',                                           // Action identifier
    minScore: config('enquiry.recaptcha.min_score', 0.5),       // Score threshold
    enabled: setting()->reCaptchaEnabled()                      // Global toggle
          && config('enquiry.recaptcha.enabled'),               // Package toggle
),
```

**Double-Check System:**
1. Global setting via `setting()->reCaptchaEnabled()` (site-wide)
2. Package config via `config('enquiry.recaptcha.enabled')` (package-level)
3. Both must be `true` for reCAPTCHA validation to run

**Score Threshold:**
- `0.1` - Very lenient (accepts almost all submissions)
- `0.5` - Balanced (default, recommended)
- `1.0` - Very strict (rejects most submissions, even legitimate ones)

---

## Extension Points

### Configuration Extensions

**File**: `src/Config/enquiry.php`

Add new configuration keys:
```php
return [
    'prefix' => 'contact',
    'success_url' => '/thank-you',
    'crm_email' => null,

    // New configuration
    'enable_notifications' => true,
    'notification_channels' => ['mail', 'slack'],
    'admin_emails' => ['admin@example.com', 'support@example.com'],
];
```

### Route Extensions

**File**: `src/Routes/frontend.php`

Add additional routes:
```php
Route::prefix(config('enquiry.prefix'))
    ->as('enquiry.')
    ->group(function () {
        // Existing route
        Route::post('store', [EnquiryController::class, 'store'])
            ->middleware(ProtectAgainstSpam::class)
            ->name('store');

        // New routes
        Route::get('form', [EnquiryController::class, 'form'])->name('form');
        Route::get('success', [EnquiryController::class, 'success'])->name('success');
    });
```

### Controller Extensions

**Add New Methods:**
```php
class EnquiryController extends AbstractController
{
    // Existing store method
    public function store(StoreEnquiryRequest $request): RedirectResponse { }

    // New methods
    public function form(): View
    {
        return view('enquiry::form');
    }

    public function success(): View
    {
        return view('enquiry::success');
    }
}
```

### Model Extensions (Add Fields)

**File**: `src/Models/Enquiry.php`

Add new properties:
```php
class Enquiry
{
    // Existing properties
    protected ?string $email = null;

    // New properties
    protected ?string $companyName = null;
    protected ?string $jobTitle = null;
    protected ?array $interests = null;

    // New getters/setters
    public function getCompanyName(): ?string
    {
        return $this->companyName;
    }

    public function setCompanyName(?string $companyName): void
    {
        if (! empty($companyName)) {
            $this->companyName = ucwords($companyName);
        }
    }

    public function getInterests(): ?array
    {
        return $this->interests;
    }

    public function setInterests(?array $interests): void
    {
        $this->interests = $interests;
    }
}
```

### Mailable Extensions

**Create New Mailable:**
```php
namespace Bongo\Enquiry\Mailables;

use Bongo\Enquiry\Models\Enquiry;
use Illuminate\Mail\Mailable;

class SlackNotificationMailable extends Mailable
{
    protected Enquiry $enquiry;

    public function __construct(Enquiry $enquiry)
    {
        $this->enquiry = $enquiry;
    }

    public function build(): self
    {
        return $this
            ->from(config("settings.mail_from_address"))
            ->to(config('enquiry.slack_email'))
            ->subject('New enquiry via Slack')
            ->text('enquiry::mail.slack_notification');
    }
}
```

**Send from Controller:**
```php
Mail::to(config('enquiry.slack_email'))
    ->send(new SlackNotificationMailable($enquiry));
```

### Validation Extensions

**File**: `src/Http/Requests/StoreEnquiryRequest.php`

Add custom validation rules:
```php
public function rules(): array
{
    return [
        // Existing rules
        'email' => 'required|string|email:rfc,dns',

        // New rules
        'company_name' => 'required|string|max:255',
        'interests' => 'required|array|min:1',
        'interests.*' => 'string|in:product_a,product_b,service_c',
        'terms_accepted' => 'required|accepted',
    ];
}
```

**Add Custom Messages:**
```php
public function messages(): array
{
    return [
        'interests.required' => 'Please select at least one area of interest.',
        'terms_accepted.accepted' => 'You must accept the terms and conditions.',
    ];
}
```

---

## Adding New Features

### Example: Add Company Name Field

**Step 1: Update Enquiry DTO**

File: `src/Models/Enquiry.php`

```php
class Enquiry
{
    protected ?string $companyName = null;

    public function getCompanyName(): ?string
    {
        return $this->companyName;
    }

    public function setCompanyName(?string $companyName): void
    {
        if (! empty($companyName)) {
            $this->companyName = ucwords($companyName);
        }
    }
}
```

**Step 2: Add Validation Rule**

File: `src/Http/Requests/StoreEnquiryRequest.php`

```php
public function rules(): array
{
    return [
        // ... existing rules
        'company_name' => 'sometimes|required|string|max:255',
    ];
}
```

**Step 3: Update Controller**

File: `src/Http/Controllers/Frontend/EnquiryController.php`

```php
public function store(StoreEnquiryRequest $request): RedirectResponse
{
    $enquiry = new Enquiry();
    // ... existing setters
    $enquiry->setCompanyName($request->get('company_name'));

    // ... send emails and redirect
}
```

**Step 4: Update Email Templates**

File: `src/Views/mail/admin_enquiry.blade.php`

```blade
@if($enquiry->getCompanyName())
    <tr>
        <td style="{{ $cellStyle }}"><strong>Company:</strong></td>
        <td style="{{ $cellStyle }}">{{ $enquiry->getCompanyName() }}</td>
    </tr>
@endif
```

File: `src/Views/mail/admin_enquiry_plain.blade.php`

```blade
@if($enquiry->getCompanyName())
Company: {{ $enquiry->getCompanyName() }}
@endif
```

### Example: Add Multiple Admin Recipients

**Step 1: Update Config**

File: `src/Config/enquiry.php`

```php
return [
    // ... existing config
    'admin_emails' => [
        'admin@example.com',
        'support@example.com',
        'sales@example.com',
    ],
];
```

**Step 2: Update Controller**

File: `src/Http/Controllers/Frontend/EnquiryController.php`

```php
public function store(StoreEnquiryRequest $request): RedirectResponse
{
    $enquiry = new Enquiry();
    // ... populate enquiry

    // Send to user
    Mail::to($enquiry->getEmail())
        ->send(new EnquiryMailable($enquiry));

    // Send to multiple admins
    $adminEmails = config('enquiry.admin_emails', [setting("client::company.email")]);
    foreach ($adminEmails as $adminEmail) {
        Mail::to($adminEmail)
            ->send(new AdminEnquiryMailable($enquiry));
    }

    // ... send CRM email and redirect
}
```

### Example: Add Queue Support for Emails

**Step 1: Update Mailables to Implement ShouldQueue**

File: `src/Mailables/EnquiryMailable.php`

```php
use Illuminate\Contracts\Queue\ShouldQueue;

class EnquiryMailable extends Mailable implements ShouldQueue
{
    // Existing code...
}
```

**Step 2: Update Controller to Queue Emails**

File: `src/Http/Controllers/Frontend/EnquiryController.php`

```php
public function store(StoreEnquiryRequest $request): RedirectResponse
{
    $enquiry = new Enquiry();
    // ... populate enquiry

    // Queue emails instead of sending immediately
    Mail::to($enquiry->getEmail())
        ->queue(new EnquiryMailable($enquiry));

    Mail::to(setting("client::company.email"))
        ->queue(new AdminEnquiryMailable($enquiry));

    if (config('enquiry.crm_email')) {
        Mail::to(config('enquiry.crm_email'))
            ->queue(new CrmMailable($enquiry));
    }

    // ... redirect
}
```

### Example: Add File Attachment Support

**Step 1: Update Enquiry DTO**

File: `src/Models/Enquiry.php`

```php
use Illuminate\Http\UploadedFile;

class Enquiry
{
    // ... existing properties

    protected ?UploadedFile $attachment = null;

    public function getAttachment(): ?UploadedFile
    {
        return $this->attachment;
    }

    public function setAttachment(?UploadedFile $attachment): void
    {
        $this->attachment = $attachment;
    }
}
```

**Step 2: Add Validation Rule**

File: `src/Http/Requests/StoreEnquiryRequest.php`

```php
public function rules(): array
{
    return [
        // ... existing rules
        'attachment' => 'sometimes|file|mimes:pdf,doc,docx|max:5120', // 5MB max
    ];
}
```

**Step 3: Update Controller**

File: `src/Http/Controllers/Frontend/EnquiryController.php`

```php
public function store(StoreEnquiryRequest $request): RedirectResponse
{
    $enquiry = new Enquiry();
    // ... existing setters

    if ($request->hasFile('attachment')) {
        $enquiry->setAttachment($request->file('attachment'));
    }

    // ... send emails and redirect
}
```

**Step 4: Update Mailable to Attach File**

File: `src/Mailables/AdminEnquiryMailable.php`

```php
public function build(): AdminEnquiryMailable
{
    $mailable = $this
        ->from(config("settings.mail_from_address"))
        ->replyTo($this->enquiry->getEmail(), $this->enquiry->getName())
        ->subject('New enquiry from: '.$this->enquiry->getEmail())
        ->view('enquiry::mail.admin_enquiry', ['enquiry' => $this->enquiry])
        ->text('enquiry::mail.admin_enquiry_plain', ['enquiry' => $this->enquiry]);

    // Attach file if present
    if ($this->enquiry->getAttachment()) {
        $mailable->attach(
            $this->enquiry->getAttachment()->getRealPath(),
            [
                'as' => $this->enquiry->getAttachment()->getClientOriginalName(),
                'mime' => $this->enquiry->getAttachment()->getMimeType(),
            ]
        );
    }

    return $mailable;
}
```

---

## Testing Strategy

### Unit Tests

**Test Enquiry DTO:**
```php
public function test_enquiry_sets_name_with_proper_capitalization(): void
{
    $enquiry = new Enquiry();
    $enquiry->setName('john doe');

    $this->assertEquals('John Doe', $enquiry->getName());
}

public function test_enquiry_lowercases_email(): void
{
    $enquiry = new Enquiry();
    $enquiry->setEmail('Test@EXAMPLE.COM');

    $this->assertEquals('test@example.com', $enquiry->getEmail());
}
```

### Feature Tests

**Test Form Submission:**
```php
public function test_valid_enquiry_redirects_to_success_url(): void
{
    $response = $this->post(route('frontend.enquiry.store'), [
        'name' => 'John Doe',
        'email' => 'john@example.com',
        'message' => 'Test message',
    ]);

    $response->assertRedirect(config('enquiry.success_url'));
    $response->assertSessionHas('success');
}

public function test_enquiry_sends_all_emails(): void
{
    Mail::fake();

    $this->post(route('frontend.enquiry.store'), [
        'email' => 'john@example.com',
        'name' => 'John Doe',
    ]);

    Mail::assertSent(EnquiryMailable::class);
    Mail::assertSent(AdminEnquiryMailable::class);
}
```

### Integration Tests

**Test Spam Protection:**
```php
public function test_honeypot_blocks_spam(): void
{
    $response = $this->post(route('frontend.enquiry.store'), [
        'email' => 'spam@example.com',
        'your_name_field' => 'filled by bot', // Honeypot field
    ]);

    $response->assertStatus(419); // Page Expired
}
```

---

## Configuration Reference

### Config File: `src/Config/enquiry.php`

| Key | Type | Default | Description |
|-----|------|---------|-------------|
| `prefix` | `string` | `'contact'` | URL prefix for routes (results in `/contact/store`) |
| `show_referrer` | `bool` | `false` | Show HTTP referrer in emails |
| `crm_email` | `?string` | `null` | Optional CRM email address for notifications |
| `success_url` | `string` | `'/thank-you'` | Redirect URL after successful submission |
| `recaptcha.enabled` | `bool` | `true` | Package-level reCAPTCHA toggle |
| `recaptcha.min_score` | `float` | `0.5` | reCAPTCHA score threshold (0.1-1.0) |

### Settings Reference (via `setting()` helper)

| Setting Key | Type | Description |
|-------------|------|-------------|
| `client::company.email` | `string` | Admin email for notifications |
| `client::company.name` | `string` | Company name for email headers |
| `client::company.bcc_email` | `?string` | Optional BCC recipient |
| `reCaptchaEnabled()` | `bool` | Global reCAPTCHA toggle (method) |

### Mail Config Reference

| Config Key | Type | Description |
|------------|------|-------------|
| `settings.mail_from_address` | `string` | From email for all mailables |
| `settings.mail_from_name` | `string` | From name for all mailables |

---

## Summary

The **bongo/enquiry** package is designed with simplicity and extensibility in mind:

- **Minimal Service Provider**: Just sets module name, parent handles registration
- **Stateless DTO**: Enquiry model has no database persistence
- **Layered Spam Protection**: Honeypot + reCAPTCHA
- **Flexible Email Recipients**: User, admin, and optional CRM
- **Easy Extension Points**: Add fields, routes, mailables, or validation rules

For additional information, see:
- [`.cursorrules`](.cursorrules) - Detailed coding conventions and common tasks
- [`.github/copilot-instructions.md`](.github/copilot-instructions.md) - Code style templates
- [`CLAUDE.md`](CLAUDE.md) - Quick reference guide
