# Bongo Enquiry Package - Claude Code Guide

## Overview

The **bongo/enquiry** package provides a contact form system for Laravel applications with spam protection and multi-recipient email notifications.

**Quick Summary:**
- Handles contact form submissions via single POST route
- Validates input with RFC/DNS email validation + optional reCAPTCHA
- Prevents spam using Spatie Honeypot middleware
- Sends confirmation to user, notification to admin, optional CRM email
- No database persistence (DTO-based)

**Related Documentation:**
- [ARCHITECTURE.md](ARCHITECTURE.md) - Detailed architecture and flow diagrams
- [.cursorrules](.cursorrules) - Coding conventions and common tasks
- [.github/copilot-instructions.md](.github/copilot-instructions.md) - Code templates

---

## Commands

### Testing
```bash
# Run all tests
vendor/bin/phpunit

# Run specific test file
vendor/bin/phpunit tests/TestCase.php

# Run with coverage
vendor/bin/phpunit --coverage-html coverage
```

### Code Quality
```bash
# Check code style issues
vendor/bin/pint --test

# Fix code style automatically
vendor/bin/pint

# Run static analysis
vendor/bin/phpstan analyse
```

### Dependencies
```bash
# Install dependencies
composer install

# Update all dependencies
composer update -W

# Update composer.lock only
composer update --lock
```

---

## Quick Architecture Reference

### Request Flow

```
POST /contact/store
    ↓
[Honeypot Middleware] → Spam check
    ↓
[StoreEnquiryRequest] → Validation + reCAPTCHA
    ↓
[EnquiryController::store()]
    ↓
1. Create Enquiry DTO
2. Send EnquiryMailable → User
3. Send AdminEnquiryMailable → Admin
4. Send CrmMailable → CRM (if configured)
    ↓
Redirect to /thank-you with success message
```

### Service Provider Bootstrap

**EnquiryServiceProvider** extends `Bongo\Framework\Providers\AbstractServiceProvider`:

```php
class EnquiryServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'enquiry';
}
```

The parent class automatically registers:
- **Config**: `src/Config/enquiry.php` → `config('enquiry.*')`
- **Routes**: `src/Routes/frontend.php` → Named with `frontend.` prefix
- **Views**: `src/Views/` → `enquiry::*` namespace
- **Translations**: `src/Translations/` → `trans('enquiry::*')`

### Key Route

**Name**: `frontend.enquiry.store`
**URL**: `POST /contact/store` (configurable via `enquiry.prefix`)
**Middleware**: `Spatie\Honeypot\ProtectAgainstSpam`

---

## Key Files

| File | Purpose | Key Methods/Properties |
|------|---------|----------------------|
| `src/EnquiryServiceProvider.php` | Bootstraps package | `$module = 'enquiry'` |
| `src/Config/enquiry.php` | Configuration | `prefix`, `success_url`, `crm_email`, `recaptcha` |
| `src/Routes/frontend.php` | Route definitions | `POST /contact/store` |
| `src/Http/Controllers/Frontend/EnquiryController.php` | Form handler | `store(StoreEnquiryRequest)` |
| `src/Http/Requests/StoreEnquiryRequest.php` | Validation | `rules()` |
| `src/Models/Enquiry.php` | DTO | Getters/setters with auto-formatting |
| `src/Mailables/EnquiryMailable.php` | User email | `build()` |
| `src/Mailables/AdminEnquiryMailable.php` | Admin email | `build()` |
| `src/Mailables/CrmMailable.php` | CRM email | `build()` |
| `src/Views/mail/*.blade.php` | Email templates | Extend `framework::mail.layouts.app` |
| `src/Translations/en/frontend.php` | Translations | `store_success` message |
| `tests/TestCase.php` | Test base class | Orchestra Testbench setup |

---

## Configuration Quick Reference

### config/enquiry.php

```php
return [
    'prefix' => 'contact',              // Route prefix: /contact/store
    'show_referrer' => false,           // Show HTTP referrer in emails
    'crm_email' => null,                // Optional CRM recipient
    'success_url' => '/thank-you',      // Redirect after submission

    'recaptcha' => [
        'enabled' => true,              // Package-level toggle
        'min_score' => 0.5,             // Score threshold (0.1-1.0)
    ],
];
```

### Required Settings (via setting() helper)

```php
setting("client::company.email")       // Admin notification email
setting("client::company.name")        // Company name for emails
setting("client::company.bcc_email")   // Optional BCC recipient
setting()->reCaptchaEnabled()          // Global reCAPTCHA toggle
```

### Required Config Values

```php
config("settings.mail_from_address")   // From email for all mailables
config("settings.mail_from_name")      // From name for all mailables
```

---

## Key Classes Reference

### EnquiryController

**Location**: `src/Http/Controllers/Frontend/EnquiryController.php:20-51`

**Method**: `store(StoreEnquiryRequest $request): RedirectResponse`

**Logic:**
1. Create `Enquiry` DTO
2. Populate from validated request data
3. Send user confirmation email
4. Send admin notification email
5. Conditionally send CRM email
6. Redirect to `config('enquiry.success_url')` with success flash

**Dependencies:**
- `StoreEnquiryRequest` - Validated form data
- `Enquiry` - DTO
- `EnquiryMailable`, `AdminEnquiryMailable`, `CrmMailable` - Emails
- `setting()` helper - Admin email
- `config()` helper - Package config

### Enquiry (DTO)

**Location**: `src/Models/Enquiry.php`

**Properties:**
- `?string $name`, `$firstName`, `$lastName`
- `?string $email`, `$phone`, `$message`
- `?string $address`, `$postcode`

**Formatting:**
- `setName()`: Applies `ucwords()` (capitalize each word)
- `setFirstName()`, `setLastName()`: Apply `ucfirst()` (capitalize first letter)
- `setEmail()`: Applies `strtolower()` (lowercase)

**Design**: DTO (no database persistence)

### StoreEnquiryRequest

**Location**: `src/Http/Requests/StoreEnquiryRequest.php:10-31`

**Validation Rules:**
- `email` - Required, RFC + DNS validation
- `name`, `first_name`, `last_name`, `phone`, `address`, `postcode`, `message` - Optional (`sometimes|required`)
- `captcha-response` - Custom `Captcha` rule with configurable threshold

**reCAPTCHA Logic:**
```php
$recaptchaEnabled = setting()->reCaptchaEnabled() && config('enquiry.recaptcha.enabled');
```
Both global setting AND package config must be `true`.

### Mailables

**EnquiryMailable** (`src/Mailables/EnquiryMailable.php:22-48`):
- To: User's email
- Reply-To: Admin email
- Subject: "Thank you for your enquiry"
- Views: `enquiry::mail.enquiry` + `enquiry::mail.enquiry_plain`

**AdminEnquiryMailable** (`src/Mailables/AdminEnquiryMailable.php:18-36`):
- To: Admin email (`setting("client::company.email")`)
- Reply-To: User's email/name
- Subject: "New enquiry from: {user_email}"
- Views: `enquiry::mail.admin_enquiry` + `enquiry::mail.admin_enquiry_plain`

**CrmMailable** (`src/Mailables/CrmMailable.php:18-29`):
- To: `config('enquiry.crm_email')` (optional)
- Subject: "New enquiry from: {user_email}"
- View: `enquiry::mail.crm_plain` (plain text only)

---

## Spam Protection

### Two Layers

**1. Honeypot (Middleware)**:
- Applied via `ProtectAgainstSpam::class` on route
- Checks for hidden honeypot field
- Validates submission time (not too fast)
- Returns HTTP 419 if spam detected

**2. reCAPTCHA (Validation)**:
- Google reCAPTCHA v3 via `Bongo\Captcha\Rules\Captcha`
- Score-based (0.0-1.0)
- Configurable threshold (default: 0.5)
- Requires both global setting AND package config enabled

---

## Code Style Summary

### PHP Standards
- **PHP Version**: 8.2+
- **Laravel Version**: 10+
- **Code Style**: Laravel Pint (Laravel preset)
- **Type Declarations**: Required on public methods
- **Nullable Types**: Use `?string` for optional properties

### Naming Conventions
- **Controllers**: Singular + `Controller` (e.g., `EnquiryController`)
- **Mailables**: Descriptive + `Mailable` (e.g., `AdminEnquiryMailable`)
- **Requests**: `{Action}{Entity}Request` (e.g., `StoreEnquiryRequest`)
- **Routes**: Named with `frontend.` prefix (e.g., `frontend.enquiry.store`)
- **Config Keys**: Snake case (e.g., `success_url`, `crm_email`)

### Email Templates
- Extend `framework::mail.layouts.app`
- Use `@component('framework::mail.layouts.partials.section')`
- Always provide both HTML and plain text versions
- Name pattern: `{description}.blade.php` + `{description}_plain.blade.php`

---

## Common Tasks

### Add a New Form Field

**1. Update Enquiry DTO** (`src/Models/Enquiry.php`):
```php
protected ?string $companyName = null;

public function getCompanyName(): ?string
{
    return $this->companyName;
}

public function setCompanyName(?string $companyName): void
{
    if (! empty($companyName)) {
        $this->companyName = ucwords($companyName);
    }
}
```

**2. Add Validation** (`src/Http/Requests/StoreEnquiryRequest.php`):
```php
'company_name' => 'sometimes|required|string|max:255',
```

**3. Update Controller** (`src/Http/Controllers/Frontend/EnquiryController.php`):
```php
$enquiry->setCompanyName($request->get('company_name'));
```

**4. Update Email Templates** (`src/Views/mail/admin_enquiry.blade.php`):
```blade
@if($enquiry->getCompanyName())
    <tr>
        <td style="{{ $cellStyle }}"><strong>Company:</strong></td>
        <td style="{{ $cellStyle }}">{{ $enquiry->getCompanyName() }}</td>
    </tr>
@endif
```

### Configure CRM Email

**In `.env`:**
```env
ENQUIRY_CRM_EMAIL=crm@example.com
```

**In `config/enquiry.php`:**
```php
'crm_email' => env('ENQUIRY_CRM_EMAIL'),
```

### Change Route Prefix

**Update `config/enquiry.php`:**
```php
'prefix' => 'contact-us',  // Changes URL to /contact-us/store
```

### Adjust reCAPTCHA Threshold

**Update `config/enquiry.php`:**
```php
'recaptcha' => [
    'enabled' => true,
    'min_score' => 0.3,  // More lenient (default: 0.5)
],
```

Lower score = more lenient, higher score = more strict.

---

## Testing

### Writing Tests

**Base Class**: `Bongo\Enquiry\Tests\TestCase`

**Example Test:**
```php
namespace Bongo\Enquiry\Tests\Feature;

use Bongo\Enquiry\Tests\TestCase;
use Illuminate\Support\Facades\Mail;

class EnquiryTest extends TestCase
{
    public function test_valid_submission_sends_emails(): void
    {
        Mail::fake();

        $response = $this->post(route('frontend.enquiry.store'), [
            'name' => 'John Doe',
            'email' => 'john@example.com',
            'message' => 'Test message',
        ]);

        $response->assertRedirect(config('enquiry.success_url'));
        Mail::assertSent(EnquiryMailable::class);
        Mail::assertSent(AdminEnquiryMailable::class);
    }
}
```

---

## Dependencies

### Required Packages

| Package | Version | Purpose |
|---------|---------|---------|
| `php` | `>=8.2` | Runtime |
| `illuminate/contracts` | `^10.0` | Laravel contracts |
| `bongo/framework` | `^3.0` | Base abstractions |
| `spatie/laravel-honeypot` | `^4.0` | Spam protection |

### External Dependencies

- **bongo/captcha** - Provides `Captcha` validation rule for reCAPTCHA
- **Framework settings** - `setting()` helper for accessing client settings

---

## Troubleshooting

### Emails Not Sending

1. Check mail config: `config('settings.mail_from_address')`
2. Verify admin email: `setting('client::company.email')`
3. Check logs: `storage/logs/laravel.log`
4. Test mail driver: `php artisan tinker` → `Mail::raw('test', fn($m) => $m->to('test@example.com'))`

### reCAPTCHA Not Working

1. Verify global setting: `setting()->reCaptchaEnabled()` must be `true`
2. Check package config: `config('enquiry.recaptcha.enabled')` must be `true`
3. Ensure `bongo/captcha` package is installed
4. Check reCAPTCHA API keys in settings

### Route Not Found

1. Clear route cache: `php artisan route:clear`
2. Check route exists: `php artisan route:list | grep enquiry`
3. Verify service provider registered in `config/app.php`
4. Check prefix config: `config('enquiry.prefix')`

### Honeypot Blocking Valid Users

1. Review Honeypot config: `config/honeypot.php`
2. Check time threshold (users submitting too quickly)
3. Adjust `amount_of_seconds` setting

---

## Package Integration Notes

### Required Settings

This package expects these settings via `setting()` helper:
- `client::company.email` - Admin notification email (required)
- `client::company.name` - Company name for emails (required)
- `client::company.bcc_email` - Optional BCC recipient
- `reCaptchaEnabled()` - Global reCAPTCHA toggle (required)

### Mail Configuration

Uses these config values:
- `settings.mail_from_address` - From address for all emails (required)
- `settings.mail_from_name` - From name for all emails (required)

### Framework Macros

Controller uses `redirect()->success()` macro from bongo/framework.

---

## Version Information

- **Package**: bongo/enquiry
- **PHP**: 8.2+
- **Laravel**: 10+
- **Framework**: bongo/framework ^3.0
- **License**: MIT

---

## Further Reading

- **Detailed Architecture**: [ARCHITECTURE.md](ARCHITECTURE.md) - Class diagrams, flow charts, extension points
- **Coding Standards**: [.cursorrules](.cursorrules) - Detailed conventions and best practices
- **Code Templates**: [.github/copilot-instructions.md](.github/copilot-instructions.md) - Common patterns
- **Repository**: [https://bitbucket.org/designtec/enquiry](https://bitbucket.org/designtec/enquiry)
