# Architecture Documentation - bongo/enum

## Overview

The `bongo/enum` package provides a simple service for extracting MySQL ENUM column values, caching them, and exposing them through a service class and global helper function. This is useful for populating form dropdowns and validation rules from database schema.

**Purpose**: Bridge database schema (ENUM columns) with application logic (forms, validation)
**Pattern**: Static service with Laravel caching
**Scope**: MySQL ENUM introspection only

## Directory Structure

```
enum/
├── src/
│   ├── EnumServiceProvider.php    # Laravel service provider
│   ├── Services/
│   │   └── Enum.php               # Core ENUM extraction service
│   └── helpers.php                # Global helper function
├── tests/
│   └── TestCase.php               # Orchestra Testbench base test case
├── composer.json                  # Package dependencies and autoload
├── phpunit.xml                    # PHPUnit configuration
├── .editorconfig                  # Editor configuration
├── .styleci.yml                   # StyleCI configuration
└── README.md                      # Package documentation
```

## Class Architecture

### Service Provider Layer

```
┌─────────────────────────────────────────────────────────────┐
│ EnumServiceProvider (extends AbstractServiceProvider)       │
│                                                              │
│ Properties:                                                  │
│   protected string $module = 'enum'                          │
│                                                              │
│ Methods:                                                     │
│   + register(): void                                         │
│   │   ├─ parent::register()                                 │
│   │   └─ $this->app->bind('enum', Enum::class)              │
│   │                                                          │
│   + boot(): void                                             │
│       ├─ parent::register() [BUG: should be parent::boot()] │
│       └─ $this->app->booted(fn => include helpers.php)      │
└─────────────────────────────────────────────────────────────┘
         │
         │ extends
         ▼
┌─────────────────────────────────────────────────────────────┐
│ Bongo\Framework\Providers\AbstractServiceProvider           │
│ (from bongo/framework package)                              │
│                                                              │
│ Provides automatic bootstrapping:                           │
│   - Config loading                                           │
│   - Route registration                                       │
│   - View registration                                        │
│   - Migration loading                                        │
│   - Middleware registration                                  │
│                                                              │
│ Note: This package doesn't use most of these features       │
└─────────────────────────────────────────────────────────────┘
```

### Service Layer

```
┌─────────────────────────────────────────────────────────────┐
│ Bongo\Enum\Services\Enum                                     │
│                                                              │
│ Static Methods:                                              │
│   + getValues(string $table, string $column): array         │
│       │                                                      │
│       ├─► Check Cache: {table}.{column}                     │
│       │   If cached: return cached array                    │
│       │                                                      │
│       └─► If not cached:                                    │
│           ├─ Execute: SHOW COLUMNS FROM {table}             │
│           │            WHERE Field = '{column}'             │
│           │                                                  │
│           ├─ Parse Type: enum('val1','val2','val3')         │
│           │   Regex: /^enum\((.*)\)$/                       │
│           │                                                  │
│           ├─ Extract values: ['val1', 'val2', 'val3']       │
│           │                                                  │
│           ├─ Format labels:                                 │
│           │   'pending_approval' → 'Pending Approval'       │
│           │   (replace _ with space, ucwords)               │
│           │                                                  │
│           ├─ Build result:                                  │
│           │   ['val1' => 'Val1', 'val2' => 'Val2']          │
│           │                                                  │
│           └─ Cache result and return                        │
└─────────────────────────────────────────────────────────────┘
         │
         │ uses
         ▼
┌─────────────────────────────────────────────────────────────┐
│ Laravel Facades                                              │
│   - DB::select()          (database queries)                │
│   - Cache::remember()     (caching with TTL)                │
│   - Arr::add()            (array manipulation)              │
└─────────────────────────────────────────────────────────────┘
```

### Helper Layer

```
┌─────────────────────────────────────────────────────────────┐
│ Global Helper Function (src/helpers.php)                     │
│                                                              │
│ function get_enum_values(string $table, string $column)     │
│     │                                                        │
│     └─► Delegates to Enum::getValues($table, $column)       │
│                                                              │
│ Loaded via: app()->booted() callback in service provider    │
└─────────────────────────────────────────────────────────────┘
```

## Data Flow Diagrams

### Enum Value Retrieval Flow

```
┌──────────────┐
│ Application  │
│ Code         │
└──────┬───────┘
       │ get_enum_values('users', 'status')
       │ or Enum::getValues('users', 'status')
       ▼
┌─────────────────────────────────────────────────────────┐
│ Enum::getValues()                                        │
└─────────────────┬───────────────────────────────────────┘
                  │
                  ▼
         ┌────────────────┐
         │ Check Cache    │ Cache key: "users.status"
         │ Laravel Cache  │
         └────────┬───────┘
                  │
        ┌─────────┴──────────┐
        │                    │
   [HIT]│                    │[MISS]
        │                    │
        ▼                    ▼
  ┌───────────┐      ┌──────────────────────────┐
  │  Return   │      │ Query Database           │
  │  Cached   │      │ SHOW COLUMNS FROM users  │
  │  Values   │      │ WHERE Field = 'status'   │
  └───────────┘      └──────────┬───────────────┘
                                │
                                ▼
                      ┌──────────────────────┐
                      │ Parse ENUM String    │
                      │ Type column contains:│
                      │ "enum('active',      │
                      │  'inactive','banned')│
                      └──────────┬───────────┘
                                │
                                ▼
                      ┌──────────────────────┐
                      │ Extract & Format     │
                      │ Regex Match          │
                      │ Split by comma       │
                      │ Trim quotes          │
                      └──────────┬───────────┘
                                │
                                ▼
                      ┌──────────────────────┐
                      │ Format Labels        │
                      │ 'active' → 'Active'  │
                      │ 'pending_approval' → │
                      │   'Pending Approval' │
                      └──────────┬───────────┘
                                │
                                ▼
                      ┌──────────────────────┐
                      │ Cache Result         │
                      │ TTL: config setting  │
                      └──────────┬───────────┘
                                │
                                ▼
                      ┌──────────────────────┐
                      │ Return Array         │
                      │ ['active' => 'Active'│
                      │  'inactive' =>       │
                      │    'Inactive', ...]  │
                      └──────────────────────┘
```

### Service Provider Boot Sequence

```
┌──────────────────────┐
│ Laravel Application  │
│ Bootstrapping        │
└──────────┬───────────┘
           │
           ▼
┌──────────────────────────────────────┐
│ Service Provider Registration Phase  │
│                                      │
│ EnumServiceProvider::register()     │
│   ├─ parent::register()             │
│   └─ Bind 'enum' to Enum::class     │
└──────────┬───────────────────────────┘
           │
           ▼
┌──────────────────────────────────────┐
│ Service Provider Boot Phase          │
│                                      │
│ EnumServiceProvider::boot()          │
│   ├─ parent::register() [BUG!]      │
│   └─ Register booted callback       │
└──────────┬───────────────────────────┘
           │
           ▼
┌──────────────────────────────────────┐
│ Application Booted Event             │
│                                      │
│ Callback Executes:                   │
│   └─ include __DIR__.'/helpers.php'  │
│       └─ Defines get_enum_values()   │
└──────────────────────────────────────┘
```

## Component Reference Tables

### Class: Enum (`Bongo\Enum\Services\Enum`)

| Method | Visibility | Return Type | Parameters | Description |
|--------|-----------|-------------|------------|-------------|
| `getValues()` | public static | `array` | `string $table`<br>`string $column` | Extracts ENUM values from database column, caches result, returns formatted array |

**Dependencies**:
- `Illuminate\Support\Facades\DB` - Database queries
- `Illuminate\Support\Facades\Cache` - Result caching
- `Illuminate\Support\Arr` - Array manipulation

**Cache Strategy**:
- Key format: `{table}.{column}` (e.g., "users.status")
- TTL: Uses `config('settings.cache_default')`
- No cache tags or invalidation strategy

### Class: EnumServiceProvider (`Bongo\Enum\EnumServiceProvider`)

| Method | Visibility | Return Type | Description |
|--------|-----------|-------------|-------------|
| `register()` | public | `void` | Binds 'enum' alias to Enum class |
| `boot()` | public | `void` | Registers helper file via booted callback |

**Properties**:
- `protected string $module = 'enum'` - Module identifier for AbstractServiceProvider

**Known Issues**:
- Line 24: Calls `parent::register()` instead of `parent::boot()`

### Helper Functions

| Function | Parameters | Return Type | Description |
|----------|-----------|-------------|-------------|
| `get_enum_values()` | `string $table`<br>`string $column` | `array` | Global helper that delegates to `Enum::getValues()` |

## Extension Points

### Adding Custom Value Formatting

Currently, values are formatted using `ucwords(str_replace('_', ' ', $value))`. To customize:

**Option 1: Extend the Enum class**
```php
namespace App\Services;

use Bongo\Enum\Services\Enum as BaseEnum;

class CustomEnum extends BaseEnum
{
    public static function getValues(string $table, string $column): array
    {
        $values = parent::getValues($table, $column);

        // Apply custom formatting
        return array_map(function ($label) {
            return __("enums.$label"); // Translate
        }, $values);
    }
}
```

**Option 2: Add a new method**
```php
// In src/Services/Enum.php
public static function getTranslatedValues(string $table, string $column): array
{
    $values = self::getValues($table, $column);

    return array_map(fn($label) => __("enums.$label"), $values);
}
```

### Adding Database Connection Support

Current implementation uses default database connection. To support multiple connections:

```php
public static function getValues(
    string $table,
    string $column,
    ?string $connection = null
): array {
    $db = $connection ? DB::connection($connection) : DB::getFacadeRoot();

    return Cache::remember(
        "$connection.$table.$column",
        config('settings.cache_default'),
        function () use ($db, $table, $column) {
            $type = $db->select("SHOW COLUMNS FROM $table WHERE Field = '$column'")[0]->Type;
            // ... rest of implementation
        }
    );
}
```

### Adding Cache Tag Support

To enable cache invalidation by table or tag:

```php
public static function getValues(string $table, string $column): array
{
    return Cache::tags(['enums', "enum:$table"])->remember(
        "$table.$column",
        config('settings.cache_default'),
        function () use ($table, $column) {
            // ... implementation
        }
    );
}

// Clear all enums for a table
Cache::tags("enum:users")->flush();

// Clear all enum caches
Cache::tags('enums')->flush();
```

## Design Decisions

### Why Static Methods?

The `Enum` service uses static methods rather than instance methods because:
1. **Stateless**: No instance state needs to be maintained
2. **Simplicity**: Easier to use without dependency injection
3. **Global Helper**: Matches the pattern of the global helper function
4. **Laravel Convention**: Common pattern for utility services

**Trade-off**: Static methods are harder to mock in tests. Consider adding an instance-based alternative if testability becomes an issue.

### Why Manual SQL Parsing?

The package uses raw SQL (`SHOW COLUMNS`) and regex parsing rather than Laravel Schema Builder because:
1. **Runtime Inspection**: Schema Builder doesn't provide ENUM value inspection
2. **No Migration Required**: Works with existing database schemas
3. **MySQL Native**: Leverages MySQL's native ENUM support

**Trade-off**: Only works with MySQL. PostgreSQL, SQLite, and other databases not supported.

### Why No Validation?

The service doesn't validate table/column existence or ENUM type because:
1. **Trust Internal Code**: Assumes developers know their schema
2. **Performance**: Avoids extra database queries
3. **Simplicity**: Lets database errors bubble up naturally

**Trade-off**: Cryptic errors if table/column doesn't exist or isn't an ENUM.

### Cache Configuration Dependency

Uses `config('settings.cache_default')` rather than a package-specific config because:
1. **Convention**: Aligns with other Bongo packages
2. **Centralized**: Single cache TTL setting across packages

**Trade-off**: Requires host application to have this config key. Should provide fallback.

## Adding New Features

### Feature: Add Support for Default Values

To return which ENUM value is the default:

1. **Update Enum::getValues() to return metadata**:
```php
public static function getValues(string $table, string $column): array
{
    return Cache::remember("$table.$column", config('settings.cache_default'),
        function () use ($table, $column) {
            $result = DB::select("SHOW COLUMNS FROM $table WHERE Field = '$column'")[0];

            preg_match('/^enum\((.*)\)$/', $result->Type, $matches);

            // Parse values (existing code)
            // ...

            return [
                'values' => $enumValues,
                'default' => $result->Default ?? null,
            ];
        }
    );
}
```

2. **Add a dedicated method for just the default**:
```php
public static function getDefaultValue(string $table, string $column): ?string
{
    $values = self::getValues($table, $column);
    return $values['default'] ?? null;
}
```

3. **Update helper function**:
```php
function get_enum_default(string $table, string $column): ?string
{
    return Enum::getDefaultValue($table, $column);
}
```

### Feature: Add Validation Helper

To generate validation rules automatically:

1. **Add method to Enum service**:
```php
public static function getValidationRule(string $table, string $column): string
{
    $values = self::getValues($table, $column);
    return 'in:' . implode(',', array_keys($values));
}
```

2. **Add helper function**:
```php
function enum_validation_rule(string $table, string $column): string
{
    return Enum::getValidationRule($table, $column);
}
```

3. **Usage**:
```php
$request->validate([
    'status' => ['required', enum_validation_rule('users', 'status')],
]);
```

### Feature: Add Testing Support

To create a testable version without database dependency:

1. **Add an interface**:
```php
namespace Bongo\Enum\Contracts;

interface EnumServiceInterface
{
    public function getValues(string $table, string $column): array;
}
```

2. **Implement in Enum service**:
```php
class Enum implements EnumServiceInterface
{
    public function getValues(string $table, string $column): array { /* ... */ }
}
```

3. **Create a fake implementation**:
```php
namespace Bongo\Enum\Testing;

class FakeEnum implements EnumServiceInterface
{
    private array $enums = [];

    public function fake(string $table, string $column, array $values): void
    {
        $this->enums["$table.$column"] = $values;
    }

    public function getValues(string $table, string $column): array
    {
        return $this->enums["$table.$column"] ?? [];
    }
}
```

4. **Use in tests**:
```php
$fake = new FakeEnum();
$fake->fake('users', 'status', ['active' => 'Active', 'banned' => 'Banned']);
$this->app->instance(EnumServiceInterface::class, $fake);
```

## Security Considerations

### SQL Injection Risk

**Location**: `src/Services/Enum.php:17-18`

```php
"SHOW COLUMNS FROM $table WHERE Field = '$column'"
```

**Risk**: String interpolation of table/column names could allow SQL injection if values come from user input.

**Mitigation**:
- **Current**: Only use with trusted internal table/column names
- **Future**: Add parameter binding or validation:

```php
// Option 1: Whitelist validation
private static function validateIdentifier(string $identifier): void
{
    if (!preg_match('/^[a-zA-Z0-9_]+$/', $identifier)) {
        throw new InvalidArgumentException("Invalid identifier: $identifier");
    }
}

// Option 2: Use information_schema (safer but slower)
$type = DB::select(
    "SELECT COLUMN_TYPE FROM information_schema.COLUMNS
     WHERE TABLE_SCHEMA = DATABASE()
     AND TABLE_NAME = ?
     AND COLUMN_NAME = ?",
    [$table, $column]
)[0]->COLUMN_TYPE;
```

### Cache Poisoning

If table/column names could be manipulated, attackers could poison the cache with arbitrary keys.

**Mitigation**: Validate identifiers before using as cache keys.

## Performance Characteristics

| Operation | Complexity | Notes |
|-----------|-----------|-------|
| First call (cache miss) | O(1) | Single database query |
| Subsequent calls (cache hit) | O(1) | Cache retrieval |
| Regex parsing | O(n) | n = number of ENUM values |
| Label formatting | O(n) | n = number of ENUM values |

**Bottlenecks**:
- Database query on cache miss
- Regex parsing for large ENUM definitions

**Optimization opportunities**:
- Use tagged cache for easier invalidation
- Pre-warm cache for frequently accessed enums
- Consider serializing parsed data structure rather than array

## Testing Strategy

### Unit Tests

```php
// Test regex parsing
public function test_parses_enum_string_correctly()
public function test_handles_single_value_enum()
public function test_handles_empty_enum()

// Test formatting
public function test_formats_underscored_values()
public function test_formats_mixed_case_values()
public function test_formats_special_characters()

// Test caching
public function test_caches_results()
public function test_uses_correct_cache_key()
public function test_respects_cache_ttl()
```

### Integration Tests

```php
// Test database interaction
public function test_retrieves_real_enum_values()
public function test_handles_nonexistent_table()
public function test_handles_nonexistent_column()
public function test_handles_non_enum_column()

// Test helper function
public function test_helper_function_works()
public function test_helper_function_matches_service()
```

### Feature Tests

```php
// Test in application context
public function test_enum_values_populate_select_dropdown()
public function test_enum_values_validate_form_input()
public function test_cache_persists_across_requests()
```
