# GitHub Copilot Instructions for bongo/enum

## Project Overview

Laravel package for extracting and caching MySQL ENUM column values. Provides a service class and global helper function for retrieving ENUM values formatted for dropdowns and validation.

**Namespace**: `Bongo\Enum`
**Dependencies**: bongo/framework ^3.0, illuminate/contracts ^10.0
**PHP**: 8.2+
**Laravel**: 10+

## Key Classes and Relationships

```
EnumServiceProvider (extends AbstractServiceProvider)
    ├── Binds: 'enum' → Enum::class
    └── Loads: helpers.php (on app booted)

Enum (Service Class)
    └── getValues(string $table, string $column): array
        ├── Queries database: SHOW COLUMNS FROM {table}
        ├── Parses: enum('val1','val2') → ['val1' => 'Val1', 'val2' => 'Val2']
        └── Caches: {table}.{column} key

Helper Function
    └── get_enum_values(string $table, string $column): array
```

## Code Style Templates

### Service Provider Pattern

```php
<?php

namespace Bongo\Enum;

use Bongo\Enum\Services\Enum;
use Bongo\Framework\Providers\AbstractServiceProvider;

class EnumServiceProvider extends AbstractServiceProvider
{
    protected string $module = 'enum';

    public function register(): void
    {
        parent::register();
        $this->app->bind('enum', Enum::class);
    }

    public function boot(): void
    {
        parent::boot(); // Note: should be boot(), not register()

        $this->app->booted(function () {
            include __DIR__.'/helpers.php';
        });
    }
}
```

### Static Service Class Pattern

```php
<?php

namespace Bongo\Enum\Services;

use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class Enum
{
    public static function getValues(string $table, string $column): array
    {
        return Cache::remember(
            $table.'.'.$column,
            config('settings.cache_default'),
            function () use ($table, $column) {
                // Query database
                $type = DB::select(
                    "SHOW COLUMNS FROM $table WHERE Field = '$column'"
                )[0]->Type;

                // Parse enum values with regex
                preg_match('/^enum\((.*)\)$/', $type, $matches);

                // Build formatted array
                $enums = [];
                foreach (explode(',', $matches[1]) as $value) {
                    $v = trim($value, "'");
                    $enums = Arr::add($enums, $v, $v);
                }

                // Format for display
                $enumValues = [];
                if (count($enums) > 0) {
                    foreach ($enums as $enum) {
                        $enumValues[$enum] = ucwords(
                            str_replace('_', ' ', $enum)
                        );
                    }
                }

                return $enumValues;
            }
        );
    }
}
```

### Global Helper Pattern

```php
<?php

use Bongo\Enum\Services\Enum;

if (! function_exists('get_enum_values')) {
    function get_enum_values(string $table, string $column): array
    {
        return Enum::getValues($table, $column);
    }
}
```

### Test Case Pattern (Orchestra Testbench)

```php
<?php

namespace Bongo\Enum\Tests;

use Bongo\Enum\EnumServiceProvider;
use Illuminate\Foundation\Application;
use Orchestra\Testbench\TestCase as BaseTestCase;

class TestCase extends BaseTestCase
{
    public function setUp(): void
    {
        parent::setUp();
        // Additional setup here
    }

    protected function getPackageProviders(Application $app): array
    {
        return [
            EnumServiceProvider::class,
        ];
    }

    protected function getEnvironmentSetUp(Application $app): void
    {
        // Configure test environment
    }
}
```

## Common Patterns

### Using the Enum Service

```php
use Bongo\Enum\Services\Enum;

// Get user statuses from database
$statuses = Enum::getValues('users', 'status');
// Returns: ['active' => 'Active', 'inactive' => 'Inactive', 'banned' => 'Banned']

// Use in a form
<select name="status">
    @foreach($statuses as $value => $label)
        <option value="{{ $value }}">{{ $label }}</option>
    @endforeach
</select>

// Use in validation
$request->validate([
    'status' => ['required', 'in:' . implode(',', array_keys($statuses))],
]);
```

### Using the Helper Function

```php
// In a controller
public function create()
{
    return view('users.create', [
        'statuses' => get_enum_values('users', 'status'),
        'roles' => get_enum_values('users', 'role'),
    ]);
}

// In a validation rule
public function rules()
{
    $statuses = array_keys(get_enum_values('users', 'status'));

    return [
        'status' => ['required', Rule::in($statuses)],
    ];
}
```

### Cache Management

```php
use Illuminate\Support\Facades\Cache;

// Clear cached enum values
Cache::forget('users.status');

// Or clear all enum cache (if using tagged cache)
// Note: Current implementation doesn't use tags
Cache::flush(); // Nuclear option
```

## Important Constraints

1. **MySQL Only**: Uses MySQL-specific `SHOW COLUMNS` syntax
2. **SQL Injection Risk**: Table/column names are interpolated (not bound parameters)
   - Only use with trusted internal table/column names
   - Never expose to user input
3. **No Validation**: Doesn't check if table/column exists or is an ENUM type
4. **Cache Dependency**: Requires `config('settings.cache_default')` to exist

## Testing Patterns

### Unit Test Example (when implementing)

```php
<?php

namespace Bongo\Enum\Tests\Unit;

use Bongo\Enum\Services\Enum;
use Bongo\Enum\Tests\TestCase;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class EnumTest extends TestCase
{
    public function test_it_extracts_enum_values_from_database(): void
    {
        // Mock DB response
        DB::shouldReceive('select')
            ->once()
            ->andReturn([
                (object) ['Type' => "enum('active','inactive','banned')"]
            ]);

        $result = Enum::getValues('users', 'status');

        $this->assertEquals([
            'active' => 'Active',
            'inactive' => 'Inactive',
            'banned' => 'Banned',
        ], $result);
    }

    public function test_it_caches_enum_values(): void
    {
        Cache::shouldReceive('remember')
            ->once()
            ->with('users.status', \Mockery::any(), \Mockery::any())
            ->andReturn(['active' => 'Active']);

        $result = Enum::getValues('users', 'status');

        $this->assertIsArray($result);
    }

    public function test_it_formats_enum_labels_correctly(): void
    {
        DB::shouldReceive('select')
            ->andReturn([
                (object) ['Type' => "enum('pending_approval','in_review')"]
            ]);

        $result = Enum::getValues('posts', 'status');

        $this->assertEquals([
            'pending_approval' => 'Pending Approval',
            'in_review' => 'In Review',
        ], $result);
    }
}
```

### Integration Test Example

```php
<?php

namespace Bongo\Enum\Tests\Integration;

use Bongo\Enum\Services\Enum;
use Bongo\Enum\Tests\TestCase;
use Illuminate\Support\Facades\Schema;

class EnumIntegrationTest extends TestCase
{
    protected function setUp(): void
    {
        parent::setUp();

        // Create test table with ENUM column
        Schema::create('test_table', function ($table) {
            $table->id();
            $table->enum('status', ['draft', 'published', 'archived']);
        });
    }

    public function test_it_retrieves_real_enum_values(): void
    {
        $values = Enum::getValues('test_table', 'status');

        $this->assertEquals([
            'draft' => 'Draft',
            'published' => 'Published',
            'archived' => 'Archived',
        ], $values);
    }

    protected function tearDown(): void
    {
        Schema::dropIfExists('test_table');
        parent::tearDown();
    }
}
```

## When to Use This Package

**Use when**:
- Working with legacy databases that use ENUM columns
- Need to populate form dropdowns with database-defined values
- Want to keep validation rules in sync with database schema
- Need formatted labels for ENUM values

**Don't use when**:
- Using PostgreSQL, SQLite, or other non-MySQL databases
- Starting a new project (prefer dedicated lookup tables)
- ENUM values need to be internationalized
- Need to accept user-provided table/column names
