# CLAUDE.md - bongo/enum

## Overview

Laravel package for extracting and caching MySQL ENUM column values. Provides a service class and global helper function for retrieving formatted ENUM values for use in forms and validation.

**Purpose**: Bridge database schema (ENUM columns) with application logic
**Pattern**: Static service with Laravel caching
**Database Support**: MySQL only

## Documentation Links

- **[ARCHITECTURE.md](./ARCHITECTURE.md)** - Detailed architecture, data flow diagrams, and extension points
- **[.cursorrules](./.cursorrules)** - Cursor AI instructions with coding conventions
- **[.github/copilot-instructions.md](./.github/copilot-instructions.md)** - GitHub Copilot code templates

## Quick Start

### Installation

```bash
composer require bongo/enum
```

### Usage

```php
// Using the service class
use Bongo\Enum\Services\Enum;

$statuses = Enum::getValues('users', 'status');
// Returns: ['active' => 'Active', 'inactive' => 'Inactive']

// Using the global helper
$statuses = get_enum_values('users', 'status');

// In a form
<select name="status">
    @foreach($statuses as $value => $label)
        <option value="{{ $value }}">{{ $label }}</option>
    @endforeach
</select>

// In validation
$request->validate([
    'status' => ['required', 'in:' . implode(',', array_keys($statuses))],
]);
```

## Available Commands

From `composer.json`:

```bash
# Run tests
composer test
# or
vendor/bin/phpunit

# Fix code style
composer format
# or
vendor/bin/pint

# Check code style without fixing
vendor/bin/pint --test

# Static analysis
vendor/bin/phpstan analyse
# or
composer analyse
```

## Architecture Quick Reference

### Directory Structure

```
src/
├── EnumServiceProvider.php    # Service provider (extends AbstractServiceProvider)
├── Services/
│   └── Enum.php               # Core ENUM extraction service
└── helpers.php                # Global helper: get_enum_values()

tests/
└── TestCase.php               # Orchestra Testbench base test case
```

### Key Classes

**`Bongo\Enum\EnumServiceProvider`**
- Extends: `Bongo\Framework\Providers\AbstractServiceProvider`
- Binds: `'enum'` to `Enum::class`
- Loads: `helpers.php` on application boot

**`Bongo\Enum\Services\Enum`**
- Method: `getValues(string $table, string $column): array`
- Queries: `SHOW COLUMNS FROM {table} WHERE Field = '{column}'`
- Parses: `enum('val1','val2')` → `['val1' => 'Val1', 'val2' => 'Val2']`
- Caches: Result with key `{table}.{column}`

### Data Flow

```
Application Code
    ↓
get_enum_values('users', 'status')
    ↓
Enum::getValues()
    ↓
Check Laravel Cache
    ├─ [HIT] → Return cached array
    └─ [MISS] → Query database
        ↓
    SHOW COLUMNS FROM users WHERE Field = 'status'
        ↓
    Parse ENUM string with regex
        ↓
    Format labels (underscore → space, ucwords)
        ↓
    Cache result
        ↓
    Return array
```

## Key Files

| File | Purpose | Key Features |
|------|---------|--------------|
| `src/EnumServiceProvider.php` | Service provider | Binds Enum service, loads helpers |
| `src/Services/Enum.php` | Core service | Static method for ENUM extraction |
| `src/helpers.php` | Global helper | `get_enum_values()` function |
| `tests/TestCase.php` | Test base | Orchestra Testbench configuration |
| `composer.json` | Dependencies | PHP 8.2+, Laravel 10+, bongo/framework ^3.0 |

## Code Style Summary

### PHP Standards

- **PHP Version**: 8.2+
- **Return Types**: Required on all methods
- **Visibility**: Always declare (public/protected/private)
- **Naming**:
  - Classes: PascalCase (`Enum`, `EnumServiceProvider`)
  - Methods: camelCase (`getValues`)
  - Helper functions: snake_case (`get_enum_values`)

### Laravel Pint

```bash
# Fix style issues
vendor/bin/pint

# Check without fixing
vendor/bin/pint --test
```

Configuration follows Laravel preset (see `.php_cs` or `pint.json` if present).

## How This Extends bongo/framework

This package extends `Bongo\Framework\Providers\AbstractServiceProvider` but uses minimal features:

**Uses**:
- `$module` property for identification
- `register()` and `boot()` lifecycle hooks
- Service container binding

**Does NOT use**:
- Automatic config loading (no config files)
- Automatic route loading (no routes)
- Automatic view loading (no views)
- Automatic migration loading (no migrations)
- Middleware, commands, composers, listeners, or subscribers

The package only needs basic service provider functionality to bind the Enum service and load the helper file.

## Important Constraints

1. **MySQL Only**: Uses MySQL-specific `SHOW COLUMNS` syntax
2. **SQL Injection Risk**: Table/column names are string-interpolated (not parameterized)
   - **IMPORTANT**: Only use with trusted internal table/column names
   - Never expose to user input
3. **No Validation**: Doesn't check if table/column exists or is an ENUM type
4. **Cache Dependency**: Requires `config('settings.cache_default')` to exist
5. **Manual Cache Clearing**: No automatic cache invalidation when ENUM values change

## Common Tasks

### Clear Cached ENUM Values

```php
use Illuminate\Support\Facades\Cache;

// Clear specific enum
Cache::forget('users.status');

// Clear all cache (nuclear option)
Cache::flush();
```

### Add Tests

Currently only base TestCase exists. To add tests:

1. Create test files in `tests/` directory
2. Extend `Bongo\Enum\Tests\TestCase`
3. Mock database responses for unit tests
4. Use Orchestra Testbench for integration tests

Example:
```php
namespace Bongo\Enum\Tests\Unit;

use Bongo\Enum\Services\Enum;
use Bongo\Enum\Tests\TestCase;
use Illuminate\Support\Facades\DB;

class EnumTest extends TestCase
{
    public function test_it_extracts_enum_values(): void
    {
        DB::shouldReceive('select')
            ->once()
            ->andReturn([
                (object) ['Type' => "enum('active','inactive')"]
            ]);

        $result = Enum::getValues('users', 'status');

        $this->assertEquals([
            'active' => 'Active',
            'inactive' => 'Inactive',
        ], $result);
    }
}
```

### Extend Formatting Logic

To customize how ENUM values are formatted:

```php
namespace App\Services;

use Bongo\Enum\Services\Enum as BaseEnum;

class CustomEnum extends BaseEnum
{
    public static function getValues(string $table, string $column): array
    {
        $values = parent::getValues($table, $column);

        // Apply custom formatting (e.g., translation)
        return array_map(fn($label) => __("enums.$label"), $values);
    }
}
```

### Add Validation Helper

Create a helper for generating validation rules:

```php
// In src/helpers.php
if (! function_exists('enum_validation_rule')) {
    function enum_validation_rule(string $table, string $column): string
    {
        $values = get_enum_values($table, $column);
        return 'in:' . implode(',', array_keys($values));
    }
}

// Usage
$request->validate([
    'status' => ['required', enum_validation_rule('users', 'status')],
]);
```

## Known Issues

1. **Bug in `EnumServiceProvider::boot()`** (line 24):
   ```php
   parent::register(); // Should be parent::boot()
   ```

2. **No Cache Configuration Fallback**:
   - Relies on `config('settings.cache_default')` without fallback
   - Will error if config key doesn't exist

3. **No Tests**: Package has TestCase setup but no actual tests

4. **SQL Injection Risk**: See constraints section above

## Development Workflow

1. **Make changes** to source files
2. **Run tests**: `vendor/bin/phpunit`
3. **Fix style**: `vendor/bin/pint`
4. **Run analysis**: `vendor/bin/phpstan analyse`
5. **Commit changes** using git

For batch operations across multiple Bongo packages, see the main monorepo CLAUDE.md.

## Related Packages

- **bongo/framework**: ^3.0 - Provides AbstractServiceProvider base class
- Part of Bongo monorepo: `default/enum`

## When to Use This Package

**Use when**:
- Working with legacy databases that use ENUM columns
- Need to populate form dropdowns from database schema
- Want validation rules synchronized with database
- Need formatted labels for ENUM values

**Don't use when**:
- Using PostgreSQL, SQLite, or other non-MySQL databases
- Starting a new project (prefer lookup tables)
- ENUM values need internationalization
- Accepting user-provided table/column names

## Questions?

For detailed architecture information, see [ARCHITECTURE.md](./ARCHITECTURE.md).
