# Enum

[![Latest Version on Packagist](https://img.shields.io/packagist/v/bongo/enum.svg?style=flat-square)](https://packagist.org/packages/bongo/enum)
[![Total Downloads](https://img.shields.io/packagist/dt/bongo/enum.svg?style=flat-square)](https://packagist.org/packages/bongo/enum)

Laravel package for extracting and caching MySQL ENUM column values. Provides a service class and global helper function for retrieving formatted ENUM values for use in forms, dropdowns, and validation.

## Features

- 🔍 **Automatic ENUM Extraction**: Queries database schema to extract ENUM values
- ⚡ **Smart Caching**: Caches results using Laravel Cache for optimal performance
- 🏷️ **Label Formatting**: Automatically formats ENUM values into human-readable labels
- 🎯 **Simple API**: Static service method and global helper function
- 🔧 **Zero Configuration**: Works out of the box with existing database schemas

## Requirements

- PHP 8.2 or higher
- Laravel 10.0 or higher
- MySQL database (uses MySQL-specific ENUM introspection)
- `bongo/framework` ^3.0

## Installation

Install via Composer:

```bash
composer require bongo/enum
```

### Laravel Auto-Discovery

The package uses Laravel's auto-discovery feature. The service provider will be automatically registered.

For Laravel versions without auto-discovery, add the service provider to `config/app.php`:

```php
'providers' => [
    // ...
    Bongo\Enum\EnumServiceProvider::class,
],
```

## Usage

### Basic Usage

Use the global helper function:

```php
$statuses = get_enum_values('users', 'status');
// Returns: ['active' => 'Active', 'inactive' => 'Inactive', 'banned' => 'Banned']
```

Or use the service class directly:

```php
use Bongo\Enum\Services\Enum;

$statuses = Enum::getValues('users', 'status');
```

### In Controllers

```php
use Bongo\Enum\Services\Enum;

class UserController extends Controller
{
    public function create()
    {
        return view('users.create', [
            'statuses' => get_enum_values('users', 'status'),
            'roles' => get_enum_values('users', 'role'),
        ]);
    }
}
```

### In Blade Views

```blade
<select name="status" class="form-control">
    <option value="">Select Status</option>
    @foreach($statuses as $value => $label)
        <option value="{{ $value }}" {{ old('status') == $value ? 'selected' : '' }}>
            {{ $label }}
        </option>
    @endforeach
</select>
```

### In Form Validation

```php
use Illuminate\Validation\Rule;

public function rules()
{
    $statuses = array_keys(get_enum_values('users', 'status'));

    return [
        'status' => ['required', Rule::in($statuses)],
    ];
}

// Or with simple in: validation
public function rules()
{
    return [
        'status' => 'required|in:' . implode(',', array_keys(get_enum_values('users', 'status'))),
    ];
}
```

### Label Formatting

The package automatically formats ENUM values into human-readable labels:

```php
// Database ENUM: enum('pending_approval', 'in_review', 'published')

$statuses = get_enum_values('posts', 'status');

// Returns:
[
    'pending_approval' => 'Pending Approval',
    'in_review' => 'In Review',
    'published' => 'Published',
]
```

Formatting rules:
- Underscores (`_`) are replaced with spaces
- Each word is capitalized using `ucwords()`

## How It Works

1. **Query Database**: Executes `SHOW COLUMNS FROM {table} WHERE Field = '{column}'`
2. **Parse ENUM**: Extracts ENUM values using regex: `/^enum\((.*)\)$/`
3. **Format Labels**: Converts `pending_approval` → `Pending Approval`
4. **Cache Results**: Stores in Laravel Cache with key `{table}.{column}`
5. **Return Array**: Returns `['value' => 'Formatted Label']`

## Caching

The package automatically caches ENUM values to avoid repeated database queries.

**Cache Key Format**: `{table}.{column}` (e.g., `users.status`)

**Cache TTL**: Uses `config('settings.cache_default')` from your application configuration.

### Clear Cache

```php
use Illuminate\Support\Facades\Cache;

// Clear specific ENUM cache
Cache::forget('users.status');

// Clear all cache (if needed)
Cache::flush();
```

## API Reference

### `Enum::getValues()`

```php
public static function getValues(string $table, string $column): array
```

Extracts ENUM values from a database column.

**Parameters**:
- `$table` (string): Database table name
- `$column` (string): Column name containing ENUM type

**Returns**: Array with ENUM values as keys and formatted labels as values

**Example**:
```php
Enum::getValues('users', 'status');
// ['active' => 'Active', 'inactive' => 'Inactive']
```

### `get_enum_values()`

```php
function get_enum_values(string $table, string $column): array
```

Global helper function that delegates to `Enum::getValues()`.

**Parameters**: Same as `Enum::getValues()`

**Returns**: Same as `Enum::getValues()`

**Example**:
```php
get_enum_values('users', 'status');
// ['active' => 'Active', 'inactive' => 'Inactive']
```

## Important Constraints

### MySQL Only

This package uses MySQL-specific syntax (`SHOW COLUMNS`) and only works with MySQL databases. It is not compatible with PostgreSQL, SQLite, or other database systems.

### SQL Injection Warning

**IMPORTANT**: The package uses string interpolation for table and column names. Only use with **trusted internal table/column names**. Never pass user input directly to these functions.

```php
// ✅ SAFE - Hardcoded table/column names
$statuses = get_enum_values('users', 'status');

// ❌ UNSAFE - User input could inject SQL
$table = $_GET['table']; // User controlled
$statuses = get_enum_values($table, 'status'); // SQL injection risk!
```

### No Validation

The service does not validate:
- Whether the table exists
- Whether the column exists
- Whether the column is actually an ENUM type

Database errors will bubble up if you provide invalid table/column names.

## Configuration

### Cache TTL

The package uses `config('settings.cache_default')` for cache duration. Ensure this configuration exists in your application:

```php
// config/settings.php
return [
    'cache_default' => 3600, // 1 hour in seconds
];
```

If this configuration doesn't exist, you may encounter errors. Consider adding a fallback in your application.

## Advanced Usage

### Custom Label Formatting

Extend the `Enum` class to customize label formatting:

```php
namespace App\Services;

use Bongo\Enum\Services\Enum as BaseEnum;

class CustomEnum extends BaseEnum
{
    public static function getValues(string $table, string $column): array
    {
        $values = parent::getValues($table, $column);

        // Apply translation
        return array_map(function ($label) {
            return __("enums.$label");
        }, $values);
    }
}
```

Then use your custom service:

```php
use App\Services\CustomEnum;

$statuses = CustomEnum::getValues('users', 'status');
```

### Create a Validation Helper

Add a helper for generating validation rules:

```php
// app/helpers.php
if (! function_exists('enum_validation_rule')) {
    function enum_validation_rule(string $table, string $column): string
    {
        $values = get_enum_values($table, $column);
        return 'in:' . implode(',', array_keys($values));
    }
}

// Usage in FormRequest
public function rules()
{
    return [
        'status' => ['required', enum_validation_rule('users', 'status')],
    ];
}
```

## Testing

Run the test suite:

```bash
composer test
```

Or using PHPUnit directly:

```bash
vendor/bin/phpunit
```

## Code Style

This package uses Laravel Pint for code style enforcement.

Fix code style issues:

```bash
composer format
```

Or manually:

```bash
vendor/bin/pint
```

Check style without fixing:

```bash
vendor/bin/pint --test
```

## Static Analysis

Run PHPStan for static analysis:

```bash
composer analyse
```

Or manually:

```bash
vendor/bin/phpstan analyse
```

## Documentation

- **[CLAUDE.md](./CLAUDE.md)** - Quick reference for Claude Code AI
- **[ARCHITECTURE.md](./ARCHITECTURE.md)** - Detailed architecture documentation with diagrams
- **[.cursorrules](./.cursorrules)** - Cursor AI development guidelines
- **[.github/copilot-instructions.md](./.github/copilot-instructions.md)** - GitHub Copilot code templates

## Changelog

Please see [CHANGELOG.md](CHANGELOG.md) for recent changes.

## Contributing

Contributions are welcome! Please feel free to submit a Pull Request.

### Development Guidelines

1. Follow PSR-12 coding standards
2. Write tests for new features
3. Run code style fixes before committing: `composer format`
4. Run static analysis: `composer analyse`
5. Ensure tests pass: `composer test`

## Security

If you discover any security issues, please email stuart.elliott@bespokeuk.com instead of using the issue tracker.

**Known Security Considerations**:
- Only use with trusted table/column names (SQL injection risk)
- Do not expose to user input without validation

## Credits

- [Stuart Elliott](https://github.com/stuartaelliott)
- [Bespoke UK](https://bespokeuk.com)

## License

The MIT Licence (MIT). Please see [Licence File](LICENSE) for more information.

## Support

For issues, questions, or contributions:
- **Issues**: [Bitbucket Repository](https://bitbucket.org/designtec/enum)
- **Email**: stuart.elliott@bespokeuk.com
- **Website**: [Bespoke UK](https://bespokeuk.com)

## Related Packages

Part of the Bongo framework monorepo:
- **bongo/framework** - Base framework functionality
- See other Bongo packages at [designtecpackages.co.uk](https://designtecpackages.co.uk)
