<?php

declare(strict_types=1);

namespace Bongo\Estimate\Calculators;

use Bongo\Estimate\Models\EstimateItem;
use Bongo\Estimate\Models\EstimateItemPrice;
use Bongo\Estimate\Models\EstimatePlan;
use Bongo\Estimate\Models\EstimatePlanItem;
use Bongo\Framework\Helpers\Tax;
use Illuminate\Contracts\Container\BindingResolutionException;

class EstimateItemPriceCalculator
{
    private EstimatePlan $estimatePlan;
    private EstimatePlanItem $estimatePlanItem;
    private EstimateItem $estimateItem;
    private EstimateItemPrice $estimateItemPrice;

    private float $planCostPerM2 = 0.00;
    private float $totalAreaM2 = 0.00;

    private float $costPerM2 = 0.00;
    private float $subtotal = 0.00;
    private float $taxRate = 0.00;
    private float $tax = 0.00;
    private float $total = 0.00;

    private int $chargeableTreatments = 0;
    private float $pricePerMonth = 0.00;

    public function setEstimateItemPrice(EstimateItemPrice $estimateItemPrice): self
    {
        $this->estimatePlan = $estimateItemPrice->plan;
        $this->estimatePlanItem = $estimateItemPrice->planItem;
        $this->estimateItem = $estimateItemPrice->item;
        $this->estimateItemPrice = $estimateItemPrice;

        return $this;
    }

    /** @throws BindingResolutionException */
    public function calculate(): self
    {
        $this->setTaxRate()
            ->setPlanCostPerM2()
            ->setTotalAreaM2()
            ->setCostPerM2()
            ->setSubtotal()
            ->setTax()
            ->setTotal()
            ->setChargeableTreatments()
            ->setPricePerMonth();

        return $this;
    }

    public function save(): EstimateItemPrice
    {
        $this->estimateItemPrice->cost_per_m2 = $this->costPerM2;
        $this->estimateItemPrice->subtotal = $this->subtotal;
        $this->estimateItemPrice->tax_rate = $this->taxRate;
        $this->estimateItemPrice->tax = $this->tax;
        $this->estimateItemPrice->total = $this->total;
        $this->estimateItemPrice->price_per_month = $this->pricePerMonth;
        $this->estimateItemPrice->save();

        return $this->estimateItemPrice;
    }

    /** @throws BindingResolutionException */
    private function setTaxRate(): self
    {
        $this->taxRate = setting('package::estimate.tax_rate', 20);

        return $this;
    }

    private function setPlanCostPerM2(): self
    {
        $this->planCostPerM2 = $this->estimatePlanItem->cost_per_m2;

        if (config('estimate.prices_include_tax')) {
            $this->planCostPerM2 = Tax::excluding($this->planCostPerM2, $this->taxRate);
        }

        // Handle the minimum price for the plan
        if ($this->planCostPerM2 < $this->estimatePlan->min_price) {
            $this->planCostPerM2 = $this->estimatePlan->min_price;
        }

        return $this;
    }

    private function setTotalAreaM2(): self
    {
        $this->totalAreaM2 = (float) $this->estimateItem->total_area_m2;

        return $this;
    }

    private function setCostPerM2(): self
    {
        $this->costPerM2 = ! empty($this->totalAreaM2)
            ? $this->planCostPerM2 / $this->totalAreaM2
            : 0;

        return $this;
    }

    private function setSubtotal(): self
    {
        $this->subtotal = ! empty($this->planCostPerM2)
            ? $this->planCostPerM2
            : 0;

        return $this;
    }

    private function setTax(): self
    {
        $this->tax = Tax::calculate($this->subtotal, $this->taxRate);

        return $this;
    }

    private function setTotal(): self
    {
        $this->total = $this->subtotal + $this->tax;

        return $this;
    }

    private function setChargeableTreatments(): self
    {
        $this->chargeableTreatments = (int) $this->estimatePlan->chargeable_treatments;

        return $this;
    }

    private function setPricePerMonth(): self
    {
        $monthsInYear = 12;

        $this->pricePerMonth = ! empty($this->total)
            ? ($this->total * $this->chargeableTreatments) / $monthsInYear
            : 0;

        return $this;
    }
}
