<?php

declare(strict_types=1);

namespace Bongo\Estimate\Commands;

use Bongo\Estimate\Exceptions\EstimateItemException;
use Bongo\Estimate\Exceptions\EstimateLocationException;
use Bongo\Estimate\Exceptions\EstimatePlanException;
use Bongo\Estimate\Models\Estimate;
use Bongo\Estimate\Models\EstimateItem;
use Bongo\Estimate\Models\EstimateLocation;
use Bongo\Estimate\Models\EstimatePlan;
use Bongo\Estimate\Models\EstimatePlanItem;
use Bongo\Estimate\Models\EstimateService;
use Bongo\Framework\Interfaces\StatusInterface;
use Bongo\Setting\Interfaces\Namespaces;
use Bongo\Setting\Models\Setting;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class UpgradeToV4Command extends Command
{
    /** @var string */
    protected $signature = 'estimate:upgrade_to_v4';

    /** @var string */
    protected $description = 'Migrate the existing data to the v4 structure.';

    public function handle(): void
    {
        try {
            $this->migrateExistingLocations();
            $this->migrateExistingPricePlans();
            $this->migrateExistingEstimates();

            $this->deleteAreasFromSettings();
            $this->removeUnusedColumns();

            console_print('Upgrade completed successfully.');

        } catch (Exception $e) {
            console_print('Upgrade failed: '.$e->getMessage(), 'error');

            return;
        }
    }

    /** @throws BindingResolutionException|Exception */
    protected function migrateExistingLocations(): void
    {
        console_print('-- Locations: Migrating existing data...');

        // Check if the setting manager exists and has the getEstimateAreas method
        $settingManager = app()->make('setting_manager');
        if (! class_exists($settingManager::class) || ! method_exists($settingManager, 'getEstimateAreas')) {
            throw new EstimateLocationException('The setting manager does not have the getEstimateAreas method.');
        }

        // Get the areas from the settings
        $areas = $settingManager->getEstimateAreas();
        if (empty($areas) || ! is_array($areas)) {
            throw new EstimateLocationException('No areas found in the settings.');
        }

        // Otherwise, we assume the areas are in the correct format
        foreach ($areas as $area) {

            if (empty($area['name'])
                || empty($area['latitude'])
                || empty($area['longitude'])
                || empty($area['radius'])
                || empty($area['fill'])
                || empty($area['stroke'])
            ) {
                console_print('Area is missing required fields.');

                continue;
            }

            EstimateLocation::updateOrCreate([
                'name' => $area['name'],
                'latitude' => $area['latitude'],
                'longitude' => $area['longitude'],
            ], [
                'radius' => $area['radius'],
                'fill' => $area['fill'],
                'stroke' => $area['stroke'],
                'status' => StatusInterface::ACTIVE,
            ]);
        }

        console_print('-- Locations: Existing data migrated.');
    }

    /** @throws Exception */
    protected function migrateExistingPricePlans(): void
    {
        console_print('-- Price Plans: Migrating existing data...');

        // Find or create the default service
        if (! $estimateService = EstimateService::firstWhere('default', 1)) {
            $estimateService = EstimateService::firstOrCreate(['name' => 'Lawn Care'], [
                'default' => 1,
                'status' => StatusInterface::ACTIVE,
            ]);
        }

        // If the basic plan does not exist then bail
        if (! $this->hasBasicPlanItems()) {
            throw new EstimatePlanException('The estimate.basic plan items array is empty.');
        }

        // Otherwise, create the plans
        $this->migrateBasicPlan($estimateService);
        $this->migrateEnhancedPlan($estimateService);
        $this->migratePremierPlan($estimateService);

        console_print('-- Price Plans: Existing data migrated.');
    }

    /** @throws Exception */
    protected function migrateExistingEstimates(): void
    {
        console_print('-- Estimates: Migrating existing data...');

        // Get the default service
        if (! $estimateService = EstimateService::firstWhere('default', 1)) {
            throw new EstimateItemException('No default estimate service found.');
        }

        // Get all estimates
        if (! $estimates = Estimate::all()) {
            console_print('- No existing estimate found, skipping.');

            return;
        }

        // Loop through each estimate
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        foreach ($estimates as $estimate) {

            // Create the estimate item
            console_print("- Checking estimate: {$estimate->number}");
            $estimateItem = EstimateItem::firstOrCreate([
                'estimate_id' => $estimate->id,
                'estimate_service_id' => $estimateService->id,
                'number' => $estimate->number.'-'.'1',
            ], [
                'step' => $estimate->step,
                'status' => $estimate->status,
                'areas' => $estimate->areas,
                'marker_moved' => $estimate->marker_moved,
                'total_area_m2' => round((float) $estimate->total_area_m2, 2),

                'sent_at' => ! empty($estimate->sent_at) ? $estimate->sent_at : null,
                'exported_at' => ! empty($estimate->exported_at) ? $estimate->exported_at : null,
                'export_error' => ! empty($estimate->export_error) ? $estimate->export_error : null,

                'created_at' => ! empty($estimate->created_at) ? $estimate->created_at : null,
                'created_by' => ! empty($estimate->created_by) ? $estimate->created_by : 1,

                'updated_at' => ! empty($estimate->updated_at) ? $estimate->updated_at : null,
                'updated_by' => ! empty($estimate->updated_by) ? $estimate->updated_by : 1,

                'deleted_at' => ! empty($estimate->deleted_at) ? $estimate->deleted_at : null,
                'deleted_by' => ! empty($estimate->deleted_by) ? $estimate->deleted_by : null,
            ]);

            // If the estimate is not at step 3, then bail
            if (intval($estimate->step) < 3) {
                console_print("-- Estimate: {{$estimate->number}} is not at step 3, skipping.");

                continue;
            }

            // If the estimate service does not have plans, then bail
            if (! $estimateService->hasPlans()) {
                console_print('-- Estimate service does not have plans, skipping.', 'error');

                continue;
            }

            // Loop through all the plans for the service
            foreach ($estimateService->plans as $estimatePlan) {
                console_print("--- Checking estimate plan: {$estimatePlan->name}");

                // If not in the plan items range, then skip
                if (! $estimatePlan->hasItemByAreaM2((float) $estimate->total_area_m2)) {
                    console_print(trim(sprintf('
                        --- Estimate plan: %s, max area M2: %s, is not within the total area: %s, skipping.',
                        $estimatePlan->name,
                        $estimatePlan->items()->max('area_m2'),
                        $estimate->total_area_m2,
                    )), 'error');

                    continue;
                }

                // Get the item price for range
                $estimatePlanItem = $estimatePlan->findItemByAreaM2((float) $estimate->total_area_m2);

                // Otherwise, create the estimate item prices
                console_print("--- Creating estimate price for item: {$estimatePlanItem->id}");
                $estimateItemPrice = $estimateItem->prices()->firstOrCreate([
                    'estimate_item_id' => $estimateItem->id,
                    'estimate_plan_id' => $estimatePlan->id,
                    'estimate_plan_item_id' => $estimatePlanItem->id,
                ], [
                    'created_at' => ! empty($estimate->created_at) ? $estimate->created_at : null,
                    'created_by' => ! empty($estimate->created_by) ? $estimate->created_by : 1,

                    'updated_at' => ! empty($estimate->updated_at) ? $estimate->updated_at : null,
                    'updated_by' => ! empty($estimate->updated_by) ? $estimate->updated_by : 1,

                    'deleted_at' => ! empty($estimate->deleted_at) ? $estimate->deleted_at : null,
                    'deleted_by' => ! empty($estimate->deleted_by) ? $estimate->deleted_by : null,
                ]);

                // Update the item totals
                $estimateItemPrice->updateTotals();
                console_print('--- Estimate item created.');
            }
        }
        DB::statement('SET FOREIGN_KEY_CHECKS=1;');

        console_print('-- Estimates: Existing data migrated.');
    }

    protected function deleteAreasFromSettings(): void
    {
        Setting::query()
            ->where('namespace', Namespaces::ESTIMATE_AREA)
            ->delete();
    }

    protected function removeUnusedColumns(): void
    {
        Schema::table('estimates', function (Blueprint $table) {
            $table->dropColumn([
                'step',
                'areas',
                'total_area_m2',
                'cost_per_m2',
                'subtotal',
                'tax_rate',
                'tax',
                'total',
            ]);
        });
    }

    private function hasBasicPlanItems(): bool
    {
        return config('estimate.basic') && count(config('estimate.basic')) > 0;
    }

    private function migrateBasicPlan(EstimateService $estimateService): void
    {
        $estimatePlan = $this->createPlanForService($estimateService, 'Basic');
        $this->createItemsForPlan($estimatePlan, config('estimate.basic'));

        console_print('-- Basic plan items migrated successfully.');
    }

    private function hasEnhancedPlanItems(): bool
    {
        return config('estimate.enhanced') && count(config('estimate.enhanced')) > 0;
    }

    private function migrateEnhancedPlan(EstimateService $estimateService): void
    {
        if (! $this->hasEnhancedPlanItems()) {
            console_print('-- Enhanced plan items are not set in the config. Skipping.');

            return;
        }

        $estimatePlan = $this->createPlanForService($estimateService, 'Enhanced');
        $this->createItemsForPlan($estimatePlan, config('estimate.enhanced'));

        console_print('-- Enhanced plan items migrated successfully.');
    }

    private function hasPremierPlanItems(): bool
    {
        return config('estimate.premier') && count(config('estimate.premier')) > 0;
    }

    private function migratePremierPlan(EstimateService $estimateService): void
    {
        if (! $this->hasPremierPlanItems()) {
            console_print('-- Premier plan items are not set in the config. Skipping.');

            return;
        }

        $estimatePlan = $this->createPlanForService($estimateService, 'Premier');
        $this->createItemsForPlan($estimatePlan, config('estimate.premier'));

        console_print('-- Premier plan items migrated successfully.');
    }

    private function createPlanForService(EstimateService $estimateService, string $planName): EstimatePlan
    {
        return EstimatePlan::firstOrCreate([
            'estimate_service_id' => $estimateService->id,
            'name' => $planName,
        ], [
            'default' => strtolower($planName) === 'basic',
            'status' => StatusInterface::ACTIVE,
            'min_price' => config('estimate.min_price'),
            'treatments' => config('estimate.'.strtolower($planName).'_treatments'),
            'chargeable_treatments' => config('estimate.chargeable_treatments_per_year'),
            'visits' => config('estimate.'.strtolower($planName).'_visits'),
        ]);
    }

    private function createItemsForPlan(EstimatePlan $estimatePlan, array $planItems): void
    {
        $sortOrder = 1;

        foreach ($planItems as $areaM2 => $costPerM2) {
            console_print("-- Creating {$estimatePlan->name} plan item for {$areaM2} m2 with cost {$costPerM2}...");

            EstimatePlanItem::updateOrCreate([
                'estimate_plan_id' => $estimatePlan->id,
                'area_m2' => $areaM2,
                'cost_per_m2' => $costPerM2,
            ], [
                'sort_order' => $sortOrder,
            ]);

            $sortOrder++;
        }
    }
}
