<?php

declare(strict_types=1);

namespace Bongo\Estimate\Http\Controllers\Frontend;

use Bongo\Estimate\Actions\FindEstimate;
use Bongo\Estimate\Exceptions\EstimateNotFoundException;
use Bongo\Estimate\Exceptions\GoogleGeoCodeException;
use Bongo\Estimate\Http\Requests\Frontend\SearchRequest;
use Bongo\Estimate\Services\GoogleGeoCode;
use Bongo\Framework\Http\Controllers\AbstractController;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Log;
use Spatie\Geocoder\Exceptions\CouldNotGeocode;

class SearchController extends AbstractController
{
    /**
     * @throws BindingResolutionException
     * @throws EstimateNotFoundException
     */
    public function index(SearchRequest $request): RedirectResponse
    {
        // Find the estimate or fail
        if (! $estimate = FindEstimate::byUuid(session('estimate_id'))) {
            if (config('app.debug')) {
                Log::error('Search - Estimate not found', [
                    'estimate_id' => session('estimate_id'),
                    'step' => 1,
                ]);
            }
            throw new EstimateNotFoundException();
        }

        // Validate the postcode
        $validated = $request->validated();

        try {
            // Call the Google locations api to get the lat and long
            $result = (new GoogleGeoCode())->search(
                $validated['line_1'],
                $validated['postcode']
            );

            // If no result was found, reset the address and redirect
            if (! $result) {
                throw new GoogleGeoCodeException(
                    "No results found for line 1: {$validated['line_1']}, postcode: {$validated['postcode']}"
                );
            }

            // If there was a result, but it is not within the coverage area, reset the address and redirect
            if (! $estimate->isWithinCoverageArea($result['latitude'], $result['longitude'])) {
                throw new GoogleGeoCodeException(
                    "Address not within coverage area line 1: {$validated['line_1']}, postcode: {$validated['postcode']}"
                );
            }

            // Otherwise store the address details and continue
            $estimate->line_1 = (string) $result['line_1'];
            $estimate->line_2 = (string) $result['line_2'];
            $estimate->line_3 = (string) $result['line_3'];
            $estimate->city = (string) $result['city'];
            $estimate->county = (string) $result['county'];
            $estimate->postcode = (string) $result['postcode'];
            $estimate->latitude = (string) $result['latitude'];
            $estimate->longitude = (string) $result['longitude'];
            $estimate->save();

            return redirect()->route('frontend.estimate.step_1.show');

        } catch (CouldNotGeocode|BindingResolutionException|GoogleGeoCodeException $e) {
            log_exception($e);
            $estimate->resetAddress();

            return redirect()->to(
                setting('package::estimate.more_assistance_url')
            );
        }
    }
}
