<?php

declare(strict_types=1);

namespace Bongo\Estimate\Http\Controllers\Frontend;

use Bongo\Estimate\Actions\CleanUpOldEstimates;
use Bongo\Estimate\Actions\FindEstimate;
use Bongo\Estimate\Actions\FindEstimateItem;
use Bongo\Estimate\Actions\GetDefaultService;
use Bongo\Estimate\Events\EstimateItemUpdated;
use Bongo\Estimate\Exceptions\EstimateItemNotFoundException;
use Bongo\Estimate\Exceptions\EstimateNotFoundException;
use Bongo\Estimate\Exceptions\NoServicesAvailableException;
use Bongo\Estimate\Http\Requests\Frontend\StoreStep1Request;
use Bongo\Estimate\Interfaces\StatusInterface;
use Bongo\Estimate\Models\Estimate;
use Bongo\Estimate\Models\EstimateService;
use Bongo\Framework\Http\Controllers\AbstractController;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class Step1Controller extends AbstractController
{
    public function index(Request $request): RedirectResponse
    {
        CleanUpOldEstimates::execute();

        return redirect()->route('frontend.estimate.step_1.show', $request->query() ?? []);
    }

    /** @throws NoServicesAvailableException */
    public function show(Request $request)
    {
        // Find or create the estimate
        if (! $estimate = FindEstimate::byUuid(session('estimate_id'))) {
            $estimate = Estimate::create();
            session(['estimate_id' => $estimate->uuid]);
        }

        // Find or create the estimate item
        if (! $estimateItem = FindEstimateItem::byUuid(session('estimate_item_id'))) {
            $estimateItem = $estimate->items()->create([
                'estimate_service_id' => null,
                'status' => StatusInterface::DRAFT,
            ]);
            session(['estimate_item_id' => $estimateItem->uuid]);
        }

        // Has a service been passed in the request?
        if ($estimateService = $this->paramsHasService($request)) {
            $estimateItem->estimate_service_id = $estimateService->id;
            $estimateItem->save();
        }

        // Get the available services
        $estimateServices = EstimateService::query()
            ->active()
            ->orderBy('name')
            ->get();

        // If the estimate item does not have a service, then bail
        if (! $estimateServices || $estimateServices->isEmpty()) {
            if (config('app.debug')) {
                Log::error('No Services Available', [
                    'estimate_id' => $estimate->id,
                    'estimate_item_id' => $estimateItem->id,
                    'step' => 1,
                ]);
            }
            throw new NoServicesAvailableException();
        }

        // If there is only one service, set it as the default
        if ($estimateServices->count() === 1) {
            $estimateItem->update([
                'estimate_service_id' => GetDefaultService::execute()?->id
            ]);
        }

        // Set the estimate item to step 1
        $estimateItem->setStepAs(1);

        return view('estimate::frontend.step_1.index', [
            'estimate' => $estimate,
            'estimateItem' => $estimateItem,
            'estimateServices' => $estimateServices,
        ]);
    }

    /**
     * @throws EstimateNotFoundException
     * @throws EstimateItemNotFoundException
     */
    public function store(StoreStep1Request $request): RedirectResponse
    {
        // Find the estimate or fail
        if (! $estimate = FindEstimate::byUuid(session('estimate_id'))) {
            if (config('app.debug')) {
                Log::error('Estimate not found', [
                    'estimate_id' => session('estimate_id'),
                    'step' => 1,
                ]);
            }
            throw new EstimateNotFoundException();
        }

        // Validate the input
        $validated = $request->validated();

        // Store the details and continue
        if (! empty($validated['estimate'])) {
            $estimate->fill($validated['estimate']);
            $estimate->status = StatusInterface::PENDING;
            $estimate->save();
        }

        // Find the estimate item or fail
        if (! $estimateItem = FindEstimateItem::byUuid(session('estimate_item_id'))) {
            if (config('app.debug')) {
                Log::error('Estimate Item not found', [
                    'estimate_id' => $estimate->id,
                    'estimate_item_id' => session('estimate_item_id'),
                    'step' => 1,
                ]);
            }
            throw new EstimateItemNotFoundException();
        }

        // Store the details and continue
        if (! empty($validated['estimate_item'])) {
            $estimateItem->fill($validated['estimate_item']);
        }
        $estimateItem->step = 2;
        $estimateItem->status = StatusInterface::PENDING;
        $estimateItem->save();

        // Fire the event for anyone who is listening
        event(new EstimateItemUpdated($estimateItem));

        return redirect()->route('frontend.estimate.step_2.show');
    }

    protected function paramsHasService(Request $request): ?EstimateService
    {
        if (! $request->filled('service')) {
            return null;
        }

        return EstimateService::firstWhere('key', make_key($request->get('service')));
    }
}
