<?php

declare(strict_types=1);

namespace Bongo\Estimate\Http\Controllers\Frontend;

use Bongo\Estimate\Actions\FindEstimate;
use Bongo\Estimate\Actions\FindEstimateItem;
use Bongo\Estimate\Events\EstimateItemUpdated;
use Bongo\Estimate\Exceptions\EstimateItemNotFoundException;
use Bongo\Estimate\Exceptions\EstimateNotFoundException;
use Bongo\Estimate\Exceptions\IncorrectStepException;
use Bongo\Estimate\Exceptions\NoServiceSelectedException;
use Bongo\Estimate\Http\Requests\Frontend\StoreStep2Request;
use Bongo\Framework\Http\Controllers\AbstractController;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Log;

class Step2Controller extends AbstractController
{
    /**
     * @throws EstimateNotFoundException
     * @throws EstimateItemNotFoundException
     * @throws NoServiceSelectedException
     * @throws IncorrectStepException
     */
    public function show()
    {
        // Find the estimate or fail
        if (! $estimate = FindEstimate::byUuid(session('estimate_id'))) {
            if (config('app.debug')) {
                Log::error('Estimate not found', [
                    'estimate_id' => session('estimate_id'),
                    'step' => 2,
                ]);
            }
            throw new EstimateNotFoundException();
        }

        // Find the estimate item or fail
        if (! $estimateItem = FindEstimateItem::byUuid(session('estimate_item_id'))) {
            if (config('app.debug')) {
                Log::error('Estimate Item not found', [
                    'estimate_id' => $estimate->id,
                    'estimate_item_id' => session('estimate_item_id'),
                    'step' => 2,
                ]);
            }
            throw new EstimateItemNotFoundException();
        }

        // Check if the estimate item is on the correct step
        if (intval($estimateItem->step) < 2) {
            if (config('app.debug')) {
                Log::error('Incorrect step', [
                    'estimate_id' => $estimate->id,
                    'estimate_item_id' => $estimateItem->id,
                    'step' => intval($estimateItem->step),
                ]);
            }
            throw new IncorrectStepException();
        }

        // Make sure we have a service selected
        if (empty($estimateItem->estimate_service_id)) {
            if (config('app.debug')) {
                Log::error('No service selected', [
                    'estimate_id' => $estimate->id,
                    'estimate_item_id' => $estimateItem->id,
                    'step' => 2,
                ]);
            }
            throw new NoServiceSelectedException();
        }

        return view('estimate::frontend.step_2.index', [
            'estimate' => $estimate,
            'estimateItem' => $estimateItem,
        ]);
    }

    /**
     * @throws IncorrectStepException
     * @throws NoServiceSelectedException
     * @throws EstimateNotFoundException
     * @throws EstimateItemNotFoundException
     */
    public function store(StoreStep2Request $request): RedirectResponse
    {
        // Find the estimate or fail
        if (! $estimate = FindEstimate::byUuid(session('estimate_id'))) {
            if (config('app.debug')) {
                Log::error('Estimate not found', [
                    'estimate_id' => session('estimate_id'),
                    'step' => 2,
                ]);
            }
            throw new EstimateNotFoundException();
        }

        // Find the estimate item or fail
        if (! $estimateItem = FindEstimateItem::byUuid(session('estimate_item_id'))) {
            if (config('app.debug')) {
                Log::error('Estimate Item not found', [
                    'estimate_id' => $estimate->id,
                    'estimate_item_id' => session('estimate_item_id'),
                    'step' => 2,
                ]);
            }
            throw new EstimateItemNotFoundException();
        }

        // Check if the estimate item is on the correct step
        if (intval($estimateItem->step) < 2) {
            if (config('app.debug')) {
                Log::error('Incorrect step', [
                    'estimate_id' => $estimate->id,
                    'estimate_item_id' => $estimateItem->id,
                    'step' => intval($estimateItem->step),
                ]);
            }
            throw new IncorrectStepException();
        }

        // Make sure we have a service selected
        if (empty($estimateItem->estimate_service_id)) {
            if (config('app.debug')) {
                Log::error('No service selected', [
                    'estimate_id' => $estimate->id,
                    'estimate_item_id' => $estimateItem->id,
                    'step' => 2,
                ]);
            }
            throw new NoServiceSelectedException();
        }

        // Validate the input
        $validated = $request->validated();

        // Set the latitude and longitude for the estimate
        $estimate->latitude = $validated['latitude'];
        $estimate->longitude = $validated['longitude'];
        $estimate->marker_moved = $validated['marker_moved'];
        $estimate->save();

        // Store the details and continue
        $estimateItem->areas = $validated['areas'];
        $estimateItem->marker_moved = $validated['marker_moved'];
        $estimateItem->total_area_m2 = $validated['total_area_m2'];
        $estimateItem->step = 3;
        $estimateItem->save();

        // Fire the event for anyone who is listening
        event(new EstimateItemUpdated($estimateItem));

        return redirect()->route('frontend.estimate.step_3.show');
    }
}
