<?php

declare(strict_types=1);

namespace Bongo\Estimate\Http\Controllers\Frontend;

use Bongo\Estimate\Actions\FindEstimate;
use Bongo\Estimate\Actions\FindEstimateItem;
use Bongo\Estimate\Events\EstimateItemUpdated;
use Bongo\Estimate\Exceptions\EstimateItemNotFoundException;
use Bongo\Estimate\Exceptions\EstimateNotFoundException;
use Bongo\Estimate\Exceptions\IncorrectStepException;
use Bongo\Estimate\Exceptions\NoPlansAvailableException;
use Bongo\Estimate\Exceptions\NoPriceFoundException;
use Bongo\Estimate\Exceptions\NoServiceSelectedException;
use Bongo\Estimate\Mailables\EstimateMailable;
use Bongo\Framework\Http\Controllers\AbstractController;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class Step3Controller extends AbstractController
{
    /**
     * @throws BindingResolutionException
     * @throws NoServiceSelectedException
     * @throws EstimateItemNotFoundException
     * @throws EstimateNotFoundException
     * @throws IncorrectStepException
     * @throws NoPlansAvailableException
     * @throws NoPriceFoundException
     */
    public function show()
    {
        // Find the estimate or fail
        if (! $estimate = FindEstimate::byUuid(session('estimate_id'))) {
            throw new EstimateNotFoundException();
        }

        // Find the estimate item or fail
        if (! $estimateItem = FindEstimateItem::byUuid(session('estimate_item_id'))) {
            if (config('app.debug')) {
                Log::error('EstimateItem not found', [
                    'estimate_id' => $estimate->id,
                    'estimate_item_id' => session('estimate_item_id'),
                ]);
            }
            throw new EstimateItemNotFoundException();
        }

        // Check if the estimate item is on the correct step
        if (intval($estimateItem->step) < 3) {
            if (config('app.debug')) {
                Log::error('Incorrect step', [
                    'estimate_id' => $estimate->id,
                    'estimate_item_id' => $estimateItem->id,
                    'step' => intval($estimateItem->step),
                ]);
            }
            throw new IncorrectStepException();
        }

        // Make sure we have a service selected
        if (! $estimateItem->hasService()) {
            if (config('app.debug')) {
                Log::error('No service selected', [
                    'estimate_id' => $estimate->id,
                    'estimate_item_id' => $estimateItem->id,
                    'estimate_service_id' => $estimateItem->estimate_service_id,
                    'step' => intval($estimateItem->step),
                ]);
            }
            throw new NoServiceSelectedException();
        }

        // If the service does not have any plans, then bail
        if (! $estimateItem->service->hasPlans()) {
            if (config('app.debug')) {
                Log::error('No plans available', [
                    'estimate_id' => $estimate->id,
                    'estimate_item_id' => $estimateItem->id,
                    'estimate_service_id' => $estimateItem->estimate_service_id,
                    'step' => intval($estimateItem->step),
                ]);
            }
            throw new NoPlansAvailableException();
        }

        // Destroy the session, so they have to start again
        if (! App::environment('local')) {
            session()->forget('estimate_item_id');
        }

        // Loop through all the plans for the service
        foreach ($estimateItem->service->plans as $estimatePlan) {

            // If not in the plan items range, then skip
            if (! $estimatePlan->hasItemByAreaM2((float) $estimateItem->total_area_m2)) {
                continue;
            }

            // Get the item price for range
            $estimatePlanItem = $estimatePlan->findItemByAreaM2((float) $estimateItem->total_area_m2);

            // Otherwise, create the estimate item prices
            $estimateItemPrice = $estimateItem->prices()->firstOrCreate([
                'estimate_item_id' => $estimateItem->id,
                'estimate_plan_id' => $estimatePlan->id,
                'estimate_plan_item_id' => $estimatePlanItem->id,
            ]);

            // Update the item totals
            $estimateItemPrice->updateTotals();
        }

        // If we have no prices, then bail
        if (! $estimateItem->hasPrices()) {
            if (config('app.debug')) {
                Log::error('EstimateItem has no prices', [
                    'estimate_id' => $estimate->id,
                    'estimate_item_id' => $estimateItem->id,
                    'estimate_service_id' => $estimateItem->estimate_service_id,
                    'total_area_m2' => $estimateItem->total_area_m2,
                    'prices' => $estimateItem->prices?->toArray(),
                    'step' => $estimateItem->step,
                ]);
            }
            throw new NoPriceFoundException();
        }

        // Send the estimate to the customer
        if (config('estimate.email_notifications.enabled') && empty($estimateItem->sent_at)) {
            Mail::to($estimate->email)->send(new EstimateMailable($estimate, $estimateItem));
            $estimateItem->sent_at = now();
            $estimateItem->save();
        }

        // Fire the event for anyone who is listening
        event(new EstimateItemUpdated($estimateItem));

        return view('estimate::frontend.step_3.index', [
            'estimate' => $estimate,
            'estimateItem' => $estimateItem,
            'viewPartials' => $this->getViewPartials($estimateItem->service->key),
        ]);
    }

    public function getViewPartials(?string $serviceKey): array
    {
        $viewPartials = [];
        $viewPartials['thank_you_text'] = $this->getPartial($serviceKey, 'thank_you_text');
        $viewPartials['contact_details'] = $this->getPartial($serviceKey, 'contact_details');
        $viewPartials['measurement_summary'] = $this->getPartial($serviceKey, 'measurement_summary');
        $viewPartials['single_price'] = $this->getPartial($serviceKey, 'single_price');
        $viewPartials['static_map'] = $this->getPartial($serviceKey, 'static_map');
        $viewPartials['pricing_table'] = $this->getPartial(
            $serviceKey, 'pricing_table_'.config('estimate.pricing_table_type')
        );
        $viewPartials['footer_text'] = $this->getPartial($serviceKey, 'footer_text');
        $viewPartials['create_another_button'] = $this->getPartial($serviceKey, 'create_another_button');
        $viewPartials['book_treatment_button'] = $this->getPartial($serviceKey, 'book_treatment_button');
        $viewPartials['back_to_home_button'] = $this->getPartial($serviceKey, 'back_to_home_button');

        return $viewPartials;
    }

    public function getPartial(string $serviceKey, string $partialName): string
    {
        $partialPath = 'estimate::frontend.step_3.partials';

        if (view()->exists("{$partialPath}.{$serviceKey}.{$partialName}")) {
            return "{$partialPath}.{$serviceKey}.{$partialName}";
        }

        return "{$partialPath}.{$partialName}";
    }
}
