<?php

declare(strict_types=1);

namespace Bongo\Estimate\Mailables;

use Bongo\Estimate\Models\Estimate;
use Bongo\Estimate\Models\EstimateItem;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Mail\Mailable;

class EstimateMailable extends Mailable
{
    protected Estimate $estimate;
    protected EstimateItem $estimateItem;

    public function __construct(Estimate $estimate, EstimateItem $estimateItem)
    {
        $this->estimate = $estimate;
        $this->estimateItem = $estimateItem;

        $this->estimateItem->loadMissing(['service', 'prices.plan']);
    }

    /**
     * @throws BindingResolutionException
     */
    public function build(): EstimateMailable
    {
        return $this
            ->from(
                config('settings.mail_from_address'),
                config('settings.mail_from_name')
            )
            ->replyTo(
                setting('package::estimate.sales_email'),
                setting('client::company.name'),
            )
            ->cc(setting('package::estimate.sales_email'))
            ->subject('Estimate:'.config('estimate.number_prefix').$this->estimateItem->number)
            ->view('estimate::mail.estimate', [
                'estimate' => $this->estimate,
                'estimateItem' => $this->estimateItem,
                'htmlViewPartials' => $this->getHtmlViewPartials($this->estimateItem->service->key),
            ])
            ->text('estimate::mail.estimate_plain', [
                'estimate' => $this->estimate,
                'estimateItem' => $this->estimateItem,
                'textViewPartials' => $this->getTextViewPartials($this->estimateItem->service->key),
            ]);
    }

    public function getHtmlViewPartials(?string $serviceKey): array
    {
        $viewPartials = [];
        $viewPartials['thank_you_text'] = $this->getPartial($serviceKey, 'thank_you_text');
        $viewPartials['contact_details'] = $this->getPartial($serviceKey, 'contact_details');
        $viewPartials['call_to_action'] = $this->getPartial($serviceKey, 'call_to_action');
        $viewPartials['measurement_summary'] = $this->getPartial($serviceKey, 'measurement_summary');
        $viewPartials['single_price'] = $this->getPartial($serviceKey, 'single_price');
        $viewPartials['pricing_table'] = $this->getPartial(
            $serviceKey, 'pricing_table_'.config('estimate.pricing_table_type')
        );
        $viewPartials['footer_text'] = $this->getPartial($serviceKey, 'footer_text');

        return $viewPartials;
    }

    public function getTextViewPartials(?string $serviceKey): array
    {
        $viewPartials = [];
        $viewPartials['thank_you_text'] = $this->getPartial($serviceKey, 'thank_you_text_plain');
        $viewPartials['contact_details'] = $this->getPartial($serviceKey, 'contact_details_plain');
        $viewPartials['call_to_action'] = $this->getPartial($serviceKey, 'call_to_action_plain');
        $viewPartials['measurement_summary'] = $this->getPartial($serviceKey, 'measurement_summary_plain');
        $viewPartials['single_price'] = $this->getPartial($serviceKey, 'single_price_plain');
        $viewPartials['pricing_table'] = $this->getPartial(
            $serviceKey, 'pricing_table_'.config('estimate.pricing_table_type').'_plain'
        );
        $viewPartials['footer_text'] = $this->getPartial($serviceKey, 'footer_text_plain');

        return $viewPartials;
    }

    public function getPartial(string $serviceKey, string $partialName): string
    {
        $partialPath = 'estimate::mail.partials';

        if (view()->exists("{$partialPath}.{$serviceKey}.{$partialName}")) {
            return "{$partialPath}.{$serviceKey}.{$partialName}";
        }

        return "{$partialPath}.{$partialName}";
    }
}
