<?php

declare(strict_types=1);

namespace Bongo\Estimate\Models;

use Bongo\Estimate\Concerns\CanBeExported;
use Bongo\Estimate\Interfaces\StatusInterface;
use Bongo\Estimate\Maps\Marker;
use Bongo\Estimate\Maps\Path;
use Bongo\Estimate\Maps\StaticMap;
use Bongo\Estimate\Traits\HasAreas;
use Bongo\Estimate\Traits\HasItemNumber;
use Bongo\Estimate\Traits\HasMapMarker;
use Bongo\Estimate\Traits\HasStatus;
use Bongo\Estimate\Traits\HasSteps;
use Bongo\Framework\Models\AbstractModel;
use Bongo\Framework\Traits\HasUUID;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class EstimateItem extends AbstractModel implements StatusInterface
{
    use CanBeExported;
    use HasAreas;
    use HasItemNumber;
    use HasMapMarker;
    use HasStatus;
    use HasSteps;
    use HasUUID;
    use SoftDeletes;

    /** @var array */
    protected $fillable = [
        'estimate_id',
        'estimate_service_id',
        'estimate_plan_id',
        'sent_at',
    ];

    /** @var array */
    protected $casts = [];

    /**
     * @throws BindingResolutionException
     * @throws Exception
     */
    public function getStaticMap($asImage = false): string
    {
        if (count($this->getAreas()) < 1 || ! $this->hasEstimate()) {
            return 'https://via.placeholder.com/600x300';
        }

        // Create the map
        $staticMap = (new StaticMap())
            ->setKey(setting('system::credentials.google_maps_api_key'))
            ->setCenter([
                $this->estimate->latitude,
                $this->estimate->longitude
            ])
            ->setZoom(config('estimate.static_map.zoom_level', 20));

        // Add the marker
        $marker = (new Marker())->addLocation([
            $this->estimate->latitude,
            $this->estimate->longitude
        ]);
        $staticMap->addMarker($marker);

        // Add the areas (Overlays)
        foreach (json_decode($this->areas, true) as $area) {
            $path = (new Path())->setCoordinates($area['coordinates']);
            $staticMap->addPath($path);
        }

        // Generate the map
        return $asImage
            ? $staticMap->generateImage("{$this->uuid}.png", 'estimates')
            : $staticMap->generateUrl();
    }

    public function estimate(): BelongsTo
    {
        return $this->belongsTo(Estimate::class, 'estimate_id');
    }

    public function hasEstimate(): bool
    {
        $this->loadMissing('estimate');

        return ! empty($this->estimate_id) && ! is_null($this->estimate);
    }

    public function service(): BelongsTo
    {
        return $this->belongsTo(EstimateService::class, 'estimate_service_id');
    }

    public function hasService(): bool
    {
        $this->loadMissing('service');

        return ! empty($this->estimate_service_id) && ! is_null($this->service);
    }

    public function prices(): HasMany
    {
        return $this->hasMany(EstimateItemPrice::class, 'estimate_item_id');
    }

    public function hasPrices(): bool
    {
        $this->load('prices');

        return ! is_null($this->prices) && count($this->prices) > 0;
    }

    public function hasSinglePrice(): bool
    {
        $this->loadMissing('prices');

        return ! is_null($this->prices) && count($this->prices) === 1;
    }

    public function getSinglePrice(): ?EstimateItemPrice
    {
        return $this->prices?->first() ?? null;
    }

    public function hasMultiplePrices(): bool
    {
        $this->loadMissing('prices');

        return ! is_null($this->prices) && count($this->prices) > 1;
    }
}
