<?php

declare(strict_types=1);

namespace Bongo\Estimate\Models;

use Bongo\Estimate\Traits\HasDefault;
use Bongo\Framework\Casts\Pence;
use Bongo\Framework\Interfaces\StatusInterface;
use Bongo\Framework\Models\AbstractModel;
use Bongo\Framework\Traits\HasKey;
use Bongo\Framework\Traits\HasStatus;
use Bongo\Framework\Traits\HasUUID;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class EstimatePlan extends AbstractModel implements StatusInterface
{
    use HasDefault;
    use HasKey;
    use HasStatus;
    use HasUUID;
    use SoftDeletes;

    /** @var array */
    protected $fillable = [
        'estimate_service_id',
        'name',
        'min_price',
        'treatments',
        'chargeable_treatments',
        'visits',
    ];

    protected $casts = [
        'min_price' => Pence::class,
        'treatments' => 'int',
        'chargeable_treatments' => 'int',
        'visits' => 'int',
    ];

    public function service(): BelongsTo
    {
        return $this->belongsTo(EstimateService::class, 'estimate_service_id');
    }

    public function hasService(): bool
    {
        $this->loadMissing('service');

        return ! empty($this->estimate_service_id) && ! is_null($this->service);
    }

    public function items(): HasMany
    {
        return $this->hasMany(EstimatePlanItem::class, 'estimate_plan_id')
            ->orderBy('sort_order', 'ASC');
    }

    public function hasItems(): bool
    {
        $this->loadMissing('items');

        return ! is_null($this->items) && count($this->items);
    }

    /**
     * Check if the highest item by area_m2 is greater
     * than or equal to the total_area_m2.
     */
    public function hasItemByAreaM2(int|float $totalAreaM2): bool
    {
        return $this->items()
            ->where('area_m2', '>=', $totalAreaM2)
            ->orderBy('area_m2', 'ASC')
            ->orderBy('sort_order', 'ASC')
            ->exists();
    }

    /**
     * Get the first item by area_m2 that is greater
     * than or equal to the total_area_m2.
     */
    public function findItemByAreaM2(int|float $totalAreaM2): ?EstimatePlanItem
    {
        return $this->items()
            ->where('area_m2', '>=', $totalAreaM2)
            ->orderBy('area_m2', 'ASC')
            ->orderBy('sort_order', 'ASC')
            ->first();
    }

    public function hasFeature(string $key): bool
    {
        $feature = config('estimate.features.'.$key);

        return ! empty($feature['plans']) && in_array($this->key, $feature['plans']);
    }

    public function duplicate(): self
    {
        // Copy attributes and update unique fields
        $newEstimatePlan = $this->replicate();
        $newEstimatePlan->uuid = (string) Str::uuid();
        $newEstimatePlan->name = $this->name.' '.Str::random(6);
        $newEstimatePlan->key = $newEstimatePlan->formatKey($newEstimatePlan->name);
        $newEstimatePlan->default = 0;
        $newEstimatePlan->status = self::PENDING;
        $newEstimatePlan->save();

        // Reset & re-load the relations
        $this->relations = [];
        $this->load('items');

        // Clone items relations
        if (isset($this->items) && count($this->items) > 0) {

            $sortOrder = 1;
            foreach ($this->items as $item) {
                $newItem = $item->replicate();
                $newItem->uuid = (string) Str::uuid();
                $newItem->estimate_plan_id = $newEstimatePlan->id;
                $newItem->sort_order = $sortOrder;
                $newItem->save();

                $sortOrder++;
            }
        }

        return $newEstimatePlan;
    }
}
