<?php

declare(strict_types=1);

namespace Bongo\Estimate\Traits;

use Bongo\Estimate\Actions\CalculateDistance;
use Bongo\Estimate\Models\EstimateLocation;
use Bongo\Framework\Traits\Address\HasCity;
use Bongo\Framework\Traits\Address\HasCounty;
use Bongo\Framework\Traits\Address\HasLatitude;
use Bongo\Framework\Traits\Address\HasLine1;
use Bongo\Framework\Traits\Address\HasLine2;
use Bongo\Framework\Traits\Address\HasLine3;
use Bongo\Framework\Traits\Address\HasLongitude;
use Bongo\Framework\Traits\Address\HasPostcode;
use Illuminate\Contracts\Container\BindingResolutionException;

trait HasAddress
{
    use HasCity;
    use HasCounty;
    use HasLatitude;
    use HasLine1;
    use HasLine2;
    use HasLine3;
    use HasLongitude;
    use HasPostcode;

    public function getAddressAttribute(): string
    {
        return implode(', ', array_filter([
            $this->line_1,
            $this->line_2,
            $this->line_3,
            $this->city,
            $this->county,
            $this->postcode,
        ]));
    }

    public function resetAddress(): void
    {
        $this->line_1 = null;
        $this->line_2 = null;
        $this->line_3 = null;
        $this->city = null;
        $this->county = null;
        $this->postcode = null;
        $this->latitude = null;
        $this->longitude = null;
        $this->save();
    }

    /** @throws BindingResolutionException */
    public function isWithinCoverageArea(?float $toLatitude, ?float $toLongitude): bool
    {
        if (is_null($toLatitude) || is_null($toLongitude)) {
            return false;
        }

        $areasCovered = EstimateLocation::query()
            ->active()
            ->get(['latitude', 'longitude', 'radius'])
            ->toArray();

        foreach ($areasCovered as $area) {
            $radiusInMiles = floatval($area['radius']);
            $fromLatitude = floatval($area['latitude']);
            $fromLongitude = floatval($area['longitude']);

            $distance = CalculateDistance::inMiles($fromLatitude, $fromLongitude, $toLatitude, $toLongitude);
            if ($distance <= $radiusInMiles) {
                return true;
            }
        }

        return false;
    }
}
