# Architecture Documentation - bongo/estimate

## Table of Contents

1. [Overview](#overview)
2. [Directory Structure](#directory-structure)
3. [Domain Model Architecture](#domain-model-architecture)
4. [Database Schema](#database-schema)
5. [Trait Composition System](#trait-composition-system)
6. [Action Pattern](#action-pattern)
7. [Calculator Pattern](#calculator-pattern)
8. [Event System](#event-system)
9. [Service Provider Integration](#service-provider-integration)
10. [Frontend Workflow](#frontend-workflow)
11. [Backend Workflow](#backend-workflow)
12. [Maps Integration](#maps-integration)
13. [Price Calculation Flow](#price-calculation-flow)
14. [Validation and Security](#validation-and-security)
15. [Testing Architecture](#testing-architecture)
16. [Extension Points](#extension-points)
17. [How to Add Features](#how-to-add-features)

---

## Overview

The `bongo/estimate` package is a Laravel package that provides a comprehensive estimation/quoting system for service-based businesses. It manages geographic service areas, tiered pricing plans, and automated calculations for recurring service costs.

**Key Architectural Principles**:
- **Trait Composition over Inheritance**: Models use multiple traits for cross-cutting concerns
- **Action Classes for Business Logic**: Discrete operations encapsulated in static utility classes
- **Calculator Pattern for Complex Logic**: Fluent builder pattern for multi-step calculations
- **Event-Driven Architecture**: Domain events fired for external integrations
- **Multi-Step Frontend Workflow**: Session-based state management across steps
- **Database-Driven Configuration**: Services, plans, and pricing stored in database

---

## Directory Structure

```
src/
├── Actions/                    # Business logic operations
│   ├── CalculateDistance.php   # Geographic distance calculations
│   ├── GenerateNumber.php      # Reference number generation
│   ├── GetPrice.php            # Price retrieval from config matrix
│   ├── GetM2Price.php          # Price lookup by area
│   ├── FindEstimate.php        # Estimate retrieval
│   ├── FindEstimateItem.php    # EstimateItem retrieval
│   ├── GetDefaultService.php   # Default service lookup
│   └── CleanUpOldEstimates.php # Draft estimate cleanup
│
├── Calculators/
│   └── EstimateItemPriceCalculator.php  # Complex price calculation logic
│
├── Commands/
│   └── UpgradeToV4Command.php  # Data migration from v3 to v4
│
├── Components/
│   └── Backend/
│       ├── RadiusMap.php       # Blade component for coverage map
│       └── ServiceDropdown.php # Service selector component
│
├── Concerns/
│   └── CanBeExported.php       # Export tracking trait
│
├── Config/
│   └── estimate.php            # Package configuration
│
├── Events/                     # Domain events
│   ├── EstimateUpdated.php
│   ├── EstimateItem*.php       # Created/Updated/Deleted
│   ├── Service*.php            # Created/Updated/Deleted
│   ├── Plan*.php               # Created/Updated/Deleted
│   └── Location*.php           # Created/Updated/Deleted
│
├── Exceptions/                 # Custom exceptions
│   ├── EstimateNotFoundException.php
│   ├── EstimateItemNotFoundException.php
│   ├── EstimateLocationException.php
│   ├── GoogleGeoCodeException.php
│   ├── IncorrectStepException.php
│   ├── NoPlansAvailableException.php
│   ├── NoPriceFoundException.php
│   └── NoServicesAvailableException.php
│
├── Http/
│   ├── Controllers/
│   │   ├── Backend/            # Admin CRUD controllers
│   │   │   ├── EstimateController.php
│   │   │   ├── EstimateItemController.php
│   │   │   ├── ServiceController.php
│   │   │   ├── LocationController.php
│   │   │   ├── PlanController.php
│   │   │   └── *DatatableController.php  # DataTables API
│   │   └── Frontend/           # Public-facing controllers
│   │       ├── Step1Controller.php  # Contact & service selection
│   │       ├── Step2Controller.php  # Area measurement
│   │       ├── Step3Controller.php  # Quote display
│   │       ├── SearchController.php # Address lookup
│   │       └── ResetController.php  # Session reset
│   ├── Requests/               # Form validation
│   └── Resources/              # API transformers
│
├── Interfaces/
│   ├── StatusInterface.php     # Status constants
│   ├── PriceTypeInterface.php  # Price type constants
│   └── GeoLookup.php           # Geocoding contract
│
├── Mailables/                  # Email notifications
│
├── Maps/                       # Google Maps integration
│   ├── Circle.php              # Radius circle generation
│   ├── ImageGenerator.php      # Static image generation
│   ├── Map.php                 # Abstract base map
│   ├── Marker.php              # Map marker
│   ├── Path.php                # Line/polygon path
│   ├── Point.php               # Single coordinate
│   ├── PolylineEncoder.php     # Google polyline encoding
│   ├── RadiusMap.php           # Interactive radius map
│   ├── RadiusMapUrl.php        # Radius map URL generator
│   ├── StaticMap.php           # Static map builder
│   ├── StaticMapUrl.php        # Static map URL generator
│   └── URLGenerator.php        # Abstract URL generation
│
├── Migrations/                 # Database schema migrations
│   ├── 2021_01_01_000001_create_estimates_table.php
│   ├── 2024_01_01_000006_create_estimate_services_table.php
│   ├── 2024_01_01_000007_create_estimate_locations_table.php
│   ├── 2024_01_01_000008_create_estimate_plans_table.php
│   ├── 2024_01_01_000009_create_estimate_plan_items_table.php
│   ├── 2024_01_01_000010_create_estimate_items_table.php
│   └── 2024_01_01_000011_create_estimate_item_prices_table.php
│
├── Models/                     # Eloquent models
│   ├── Estimate.php            # Customer quote
│   ├── EstimateItem.php        # Quote item with areas
│   ├── EstimateItemPrice.php   # Calculated pricing
│   ├── EstimateLocation.php    # Coverage area
│   ├── EstimatePlan.php        # Pricing tier
│   ├── EstimatePlanItem.php    # Area pricing breakpoint
│   └── EstimateService.php     # Service category
│
├── Routes/
│   ├── backend.php             # Admin routes (auth + employee middleware)
│   └── frontend.php            # Public routes (frontend.* named routes)
│
├── Services/
│   └── GoogleGeoCode.php       # Address geocoding service
│
├── Traits/                     # Model composition traits
│   ├── HasAddress.php          # Address fields + geocoding
│   ├── HasAreas.php            # JSON polygon storage
│   ├── HasContact.php          # Contact information
│   ├── HasDate.php             # Date tracking
│   ├── HasItemNumber.php       # Item number generation
│   ├── HasMapMarker.php        # Marker position tracking
│   ├── HasNumber.php           # Reference number generation
│   ├── HasPriceType.php        # Price type field
│   ├── HasStatus.php           # Status tracking
│   ├── HasSteps.php            # Multi-step workflow
│   ├── HasTotals.php           # Price calculation fields
│   └── HasVoucherCode.php      # Voucher code field
│
├── Translations/               # i18n files
│
├── Views/                      # Blade templates
│   ├── backend/                # Admin views
│   └── frontend/               # Public views
│       ├── step_1/             # Contact & service selection
│       ├── step_2/             # Area measurement with map
│       └── step_3/             # Quote display with pricing
│
└── EstimateServiceProvider.php # Service provider

database/
├── factories/                  # Model factories for testing
│   ├── EstimateFactory.php
│   ├── EstimateItemFactory.php
│   ├── EstimateItemPriceFactory.php
│   ├── EstimateLocationFactory.php
│   ├── EstimatePlanFactory.php
│   ├── EstimatePlanItemFactory.php
│   └── EstimateServiceFactory.php
└── seeders/                    # Database seeders

tests/
├── Unit/                       # Unit tests
│   ├── Actions/                # Action tests
│   └── Models/                 # Model tests
├── Feature/                    # Feature tests
└── TestCase.php                # Base test class
```

---

## Domain Model Architecture

### Model Hierarchy

```
┌─────────────────────┐
│  EstimateService    │  (Service category, e.g., "Lawn Care")
│  - id: uuid          │
│  - name: string      │
│  - key: string       │
│  - status: string    │
│  - default: boolean  │
└──────────┬──────────┘
           │
           │ HasMany
           ↓
┌─────────────────────┐
│   EstimatePlan      │  (Pricing tier, e.g., "Basic", "Enhanced")
│  - id: uuid          │
│  - service_id: FK    │
│  - name: string      │
│  - key: string       │
│  - min_price: int    │
│  - treatments: int   │
│  - chargeable: int   │
│  - price_type: enum  │
└──────────┬──────────┘
           │
           │ HasMany
           ↓
┌─────────────────────┐
│  EstimatePlanItem   │  (Area pricing, e.g., "50m² = £28")
│  - id: uuid          │
│  - plan_id: FK       │
│  - area_m2: decimal  │
│  - cost_per_m2: int  │
│  - sort_order: int   │
└─────────────────────┘


┌─────────────────────┐
│    Estimate         │  (Customer quote)
│  - id: uuid          │
│  - number: string    │
│  - date: date        │
│  - step: int         │
│  - status: string    │
│  - address fields    │
│  - contact fields    │
│  - voucher_code      │
└──────────┬──────────┘
           │
           │ HasMany
           ↓
┌─────────────────────┐
│   EstimateItem      │  (Service selection for property)
│  - id: uuid          │
│  - estimate_id: FK   │
│  - service_id: FK    │
│  - plan_id: FK       │
│  - number: string    │
│  - step: int         │
│  - areas: json       │ ← Polygon coordinates from map
│  - total_area_m2    │
└──────────┬──────────┘
           │
           │ HasMany
           ↓
┌─────────────────────┐
│ EstimateItemPrice   │  (Calculated pricing breakdown)
│  - id: uuid          │
│  - item_id: FK       │
│  - plan_id: FK       │
│  - plan_item_id: FK  │
│  - cost_per_m2: int  │
│  - subtotal: int     │
│  - tax_rate: decimal │
│  - tax: int          │
│  - total: int        │
│  - per_month: int    │
└─────────────────────┘


┌─────────────────────┐
│ EstimateLocation    │  (Coverage area - independent)
│  - id: uuid          │
│  - name: string      │
│  - key: string       │
│  - radius: int (km)  │
│  - latitude: decimal │
│  - longitude: decimal│
│  - fill: string      │
│  - stroke: string    │
└─────────────────────┘
```

### Key Relationships

```php
// EstimateService relationships
EstimateService::plans() → HasMany(EstimatePlan)

// EstimatePlan relationships
EstimatePlan::service() → BelongsTo(EstimateService)
EstimatePlan::items() → HasMany(EstimatePlanItem)

// EstimatePlanItem relationships
EstimatePlanItem::plan() → BelongsTo(EstimatePlan)

// Estimate relationships
Estimate::items() → HasMany(EstimateItem)

// EstimateItem relationships
EstimateItem::estimate() → BelongsTo(Estimate)
EstimateItem::service() → BelongsTo(EstimateService)
EstimateItem::plan() → BelongsTo(EstimatePlan)
EstimateItem::prices() → HasMany(EstimateItemPrice)

// EstimateItemPrice relationships
EstimateItemPrice::item() → BelongsTo(EstimateItem)
EstimateItemPrice::plan() → BelongsTo(EstimatePlan)
EstimateItemPrice::planItem() → BelongsTo(EstimatePlanItem)
```

---

## Database Schema

### estimate_services

```sql
CREATE TABLE estimate_services (
    id BIGINT UNSIGNED PRIMARY KEY,
    uuid UUID UNIQUE NOT NULL,
    name VARCHAR(255) NOT NULL,
    key VARCHAR(255) UNIQUE NOT NULL,
    status VARCHAR(50) DEFAULT 'pending',
    default BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    deleted_at TIMESTAMP NULL
);
```

### estimate_plans

```sql
CREATE TABLE estimate_plans (
    id BIGINT UNSIGNED PRIMARY KEY,
    uuid UUID UNIQUE NOT NULL,
    estimate_service_id BIGINT UNSIGNED NOT NULL,
    name VARCHAR(255) NOT NULL,
    key VARCHAR(255) NOT NULL,
    min_price INTEGER DEFAULT 0 COMMENT 'Stored in pence',
    treatments INTEGER DEFAULT 11,
    chargeable_treatments INTEGER DEFAULT 7,
    visits INTEGER DEFAULT 11,
    price_type VARCHAR(50) DEFAULT 'per_treatment_month',
    status VARCHAR(50) DEFAULT 'pending',
    default BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    deleted_at TIMESTAMP NULL,
    FOREIGN KEY (estimate_service_id) REFERENCES estimate_services(id)
);
```

### estimate_plan_items

```sql
CREATE TABLE estimate_plan_items (
    id BIGINT UNSIGNED PRIMARY KEY,
    uuid UUID UNIQUE NOT NULL,
    estimate_plan_id BIGINT UNSIGNED NOT NULL,
    area_m2 DECIMAL(10,2) NOT NULL COMMENT 'Area in square metres',
    cost_per_m2 INTEGER NOT NULL COMMENT 'Cost in pence',
    sort_order INTEGER DEFAULT 0,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    deleted_at TIMESTAMP NULL,
    FOREIGN KEY (estimate_plan_id) REFERENCES estimate_plans(id)
);
```

### estimate_locations

```sql
CREATE TABLE estimate_locations (
    id BIGINT UNSIGNED PRIMARY KEY,
    uuid UUID UNIQUE NOT NULL,
    name VARCHAR(255) NOT NULL,
    key VARCHAR(255) UNIQUE NOT NULL,
    radius INTEGER NOT NULL COMMENT 'Radius in kilometres',
    latitude DECIMAL(10,7) NOT NULL,
    longitude DECIMAL(10,7) NOT NULL,
    fill VARCHAR(50) DEFAULT '#FF0000',
    stroke VARCHAR(50) DEFAULT '#FF0000',
    status VARCHAR(50) DEFAULT 'pending',
    default BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    deleted_at TIMESTAMP NULL
);
```

### estimates

```sql
CREATE TABLE estimates (
    id BIGINT UNSIGNED PRIMARY KEY,
    uuid UUID UNIQUE NOT NULL,
    number VARCHAR(255) UNIQUE NOT NULL COMMENT 'e.g., KLC-0001',
    date DATE NOT NULL,
    step INTEGER DEFAULT 1,
    status VARCHAR(50) DEFAULT 'draft',

    -- Address fields
    line_1 VARCHAR(255),
    line_2 VARCHAR(255),
    line_3 VARCHAR(255),
    city VARCHAR(255),
    county VARCHAR(255),
    postcode VARCHAR(255),
    latitude DECIMAL(10,7),
    longitude DECIMAL(10,7),

    -- Contact fields
    first_name VARCHAR(255),
    last_name VARCHAR(255),
    email VARCHAR(255),
    phone VARCHAR(255),
    accepted_terms BOOLEAN DEFAULT FALSE,
    marketing_emails BOOLEAN DEFAULT FALSE,

    -- Additional fields
    voucher_code VARCHAR(255),
    sent_at TIMESTAMP NULL,
    exported_at TIMESTAMP NULL,
    export_error TEXT,

    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    deleted_at TIMESTAMP NULL
);
```

### estimate_items

```sql
CREATE TABLE estimate_items (
    id BIGINT UNSIGNED PRIMARY KEY,
    uuid UUID UNIQUE NOT NULL,
    estimate_id BIGINT UNSIGNED NOT NULL,
    estimate_service_id BIGINT UNSIGNED NOT NULL,
    estimate_plan_id BIGINT UNSIGNED NOT NULL,
    number VARCHAR(255) NOT NULL COMMENT 'e.g., KLC-0001-1',
    step INTEGER DEFAULT 1,
    status VARCHAR(50) DEFAULT 'draft',

    -- Area data
    areas JSON COMMENT 'Polygon coordinates from map drawing',
    marker_moved BOOLEAN DEFAULT FALSE,
    total_area_m2 DECIMAL(10,2) DEFAULT 0,

    -- Export tracking
    sent_at TIMESTAMP NULL,
    exported_at TIMESTAMP NULL,
    export_error TEXT,

    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    deleted_at TIMESTAMP NULL,

    FOREIGN KEY (estimate_id) REFERENCES estimates(id),
    FOREIGN KEY (estimate_service_id) REFERENCES estimate_services(id),
    FOREIGN KEY (estimate_plan_id) REFERENCES estimate_plans(id)
);
```

### estimate_item_prices

```sql
CREATE TABLE estimate_item_prices (
    id BIGINT UNSIGNED PRIMARY KEY,
    uuid UUID UNIQUE NOT NULL,
    estimate_item_id BIGINT UNSIGNED NOT NULL,
    estimate_plan_id BIGINT UNSIGNED NOT NULL,
    estimate_plan_item_id BIGINT UNSIGNED NOT NULL,

    -- Price breakdown (all stored in pence)
    cost_per_m2 INTEGER DEFAULT 0,
    subtotal INTEGER DEFAULT 0,
    tax_rate DECIMAL(5,2) DEFAULT 20.00,
    tax INTEGER DEFAULT 0,
    total INTEGER DEFAULT 0,
    price_per_month INTEGER DEFAULT 0,

    created_at TIMESTAMP,
    updated_at TIMESTAMP,
    deleted_at TIMESTAMP NULL,

    FOREIGN KEY (estimate_item_id) REFERENCES estimate_items(id),
    FOREIGN KEY (estimate_plan_id) REFERENCES estimate_plans(id),
    FOREIGN KEY (estimate_plan_item_id) REFERENCES estimate_plan_items(id)
);
```

---

## Trait Composition System

Models use trait composition instead of inheritance to add functionality. This provides flexibility and code reuse.

### Trait Reference Table

| Trait | Fields Added | Methods Added | Purpose |
|-------|--------------|---------------|---------|
| `HasUUID` | uuid | - | UUID primary key generation |
| `HasStatus` | status | scopePending(), scopeAccepted(), scopeRejected(), isPending(), isAccepted(), isRejected() | Status tracking with scopes |
| `HasDefault` | default | bootHasDefault() | Ensure only one record marked as default |
| `HasKey` | key | - | Slug-like identifier field |
| `HasAddress` | line_1, line_2, line_3, city, county, postcode, latitude, longitude | getAddressAttribute(), isWithinCoverageArea() | Address storage and validation |
| `HasContact` | first_name, last_name, email, phone, accepted_terms, marketing_emails | getNameAttribute() | Contact information |
| `HasAreas` | areas, total_area_m2 | getAreas() | JSON polygon storage |
| `HasDate` | date | - | Auto-set date to today |
| `HasNumber` | number | bootHasNumber() | Auto-generate reference number |
| `HasItemNumber` | number | bootHasItemNumber() | Generate item number (estimate-count) |
| `HasVoucherCode` | voucher_code | - | Voucher code field |
| `HasMapMarker` | marker_moved | - | Track marker movement |
| `HasSteps` | step | stepIs(), stepIsGt(), stepIsGte(), setStepAs() | Multi-step workflow |
| `HasPriceType` | price_type | scopePerTreatmentAndPerMonth(), scopePerTreatment(), scopePerMonth() | Price calculation type |
| `HasTotals` | cost_per_m2, subtotal, tax_rate, tax, total, price_per_month | updateTotals() | Price calculation fields |
| `CanBeExported` | exported_at, export_error | hasExportedAt(), hasExportError() | Export tracking |

### Trait Composition Example

```php
class Estimate extends AbstractModel implements StatusInterface
{
    use CanBeExported;      // exported_at, export_error
    use HasAddress;         // line_1, city, postcode, lat/lng
    use HasContact;         // first_name, last_name, email, phone
    use HasDate;            // date (auto-set)
    use HasNumber;          // number (auto-generated: KLC-0001)
    use HasStatus;          // status (draft/pending/accepted/rejected)
    use HasVoucherCode;     // voucher_code
    use HasUUID;            // uuid primary key
    use SoftDeletes;        // deleted_at

    // All fillable fields merged from traits via initializeHas*() methods
}
```

### Trait Initialization Pattern

Each trait implements an `initializeHas*()` method to merge fillable fields:

```php
trait HasStatus
{
    public function initializeHasStatus(): void
    {
        $this->mergeFillable(['status']);
    }

    // Scopes
    public function scopePending($query)
    {
        return $query->where('status', self::PENDING);
    }

    // Helpers
    public function isPending(): bool
    {
        return $this->status === self::PENDING;
    }
}
```

---

## Action Pattern

Action classes encapsulate discrete business logic operations. They use static methods for simple, stateless operations.

### Action Architecture

```
┌──────────────────────────────────────────┐
│          Action Classes                   │
│  (Static utility methods)                 │
├──────────────────────────────────────────┤
│  CalculateDistance                        │
│  - inMiles($lat1, $lng1, $lat2, $lng2)   │
│  - inKilometers($lat1, $lng1, $lat2, $lng2)│
├──────────────────────────────────────────┤
│  GenerateNumber                           │
│  - forEstimate()                          │
├──────────────────────────────────────────┤
│  GetPrice                                 │
│  - forBasicPlan($totalAreaM2)            │
│  - forEnhancedPlan($totalAreaM2)         │
│  - forPremierPlan($totalAreaM2)          │
├──────────────────────────────────────────┤
│  FindEstimate                             │
│  - byUuid(?string $uuid)                 │
├──────────────────────────────────────────┤
│  FindEstimateItem                         │
│  - byUuid(?string $uuid)                 │
├──────────────────────────────────────────┤
│  GetDefaultService                        │
│  - execute()                              │
├──────────────────────────────────────────┤
│  CleanUpOldEstimates                      │
│  - execute()                              │
└──────────────────────────────────────────┘
```

### Action Example: CalculateDistance

```php
namespace Bongo\Estimate\Actions;

class CalculateDistance
{
    /**
     * Calculate distance between two coordinates in miles.
     */
    public static function inMiles(
        float $lat1,
        float $lng1,
        float $lat2,
        float $lng2
    ): float {
        $distance = self::calculate($lat1, $lng1, $lat2, $lng2);
        return $distance * 0.621371; // Convert km to miles
    }

    /**
     * Calculate distance between two coordinates in kilometres.
     */
    public static function inKilometers(
        float $lat1,
        float $lng1,
        float $lat2,
        float $lng2
    ): float {
        return self::calculate($lat1, $lng1, $lat2, $lng2);
    }

    /**
     * Spherical law of cosines calculation.
     */
    private static function calculate(
        float $lat1,
        float $lng1,
        float $lat2,
        float $lng2
    ): float {
        $theta = $lng1 - $lng2;
        $dist = sin(deg2rad($lat1)) * sin(deg2rad($lat2))
              + cos(deg2rad($lat1)) * cos(deg2rad($lat2))
              * cos(deg2rad($theta));
        $dist = acos($dist);
        $dist = rad2deg($dist);
        return $dist * 60 * 1.1515 * 1.609344; // km
    }
}

// Usage:
$distance = CalculateDistance::inMiles(52.6, 1.2, 52.7, 1.3);
```

---

## Calculator Pattern

Calculator classes use a fluent builder pattern for complex, multi-step calculations.

### Calculator Flow Diagram

```
┌─────────────────────────────────────────────────────────┐
│         EstimateItemPriceCalculator Flow                │
└─────────────────────────────────────────────────────────┘
                          │
                          ↓
        ┌─────────────────────────────────┐
        │ setEstimateItemPrice($price)    │
        │ - Store EstimateItemPrice       │
        │ - Store EstimateItem            │
        │ - Store EstimatePlan            │
        │ - Store EstimatePlanItem        │
        └────────────┬────────────────────┘
                     ↓
        ┌─────────────────────────────────┐
        │ calculate()                      │
        │ - Execute all calculation steps │
        └────────────┬────────────────────┘
                     ↓
        ┌─────────────────────────────────┐
        │ Step 1: setTaxRate()             │
        │ - Get from settings (default 20%)│
        └────────────┬────────────────────┘
                     ↓
        ┌─────────────────────────────────┐
        │ Step 2: getPlanCost()            │
        │ - Find EstimatePlanItem by area  │
        │ - Get cost_per_m2                │
        └────────────┬────────────────────┘
                     ↓
        ┌─────────────────────────────────┐
        │ Step 3: applyMinimumPrice()      │
        │ - Check plan min_price           │
        │ - Use higher of: cost or min     │
        └────────────┬────────────────────┘
                     ↓
        ┌─────────────────────────────────┐
        │ Step 4: calculateCostPerM2()     │
        │ - cost_per_m2 = cost ÷ area     │
        └────────────┬────────────────────┘
                     ↓
        ┌─────────────────────────────────┐
        │ Step 5: calculateSubtotal()      │
        │ - subtotal = planCost            │
        └────────────┬────────────────────┘
                     ↓
        ┌─────────────────────────────────┐
        │ Step 6: calculateTax()           │
        │ - tax = subtotal × tax_rate / 100│
        └────────────┬────────────────────┘
                     ↓
        ┌─────────────────────────────────┐
        │ Step 7: calculateTotal()         │
        │ - total = subtotal + tax         │
        └────────────┬────────────────────┘
                     ↓
        ┌─────────────────────────────────┐
        │ Step 8: calculatePricePerMonth() │
        │ - per_month = (total ×          │
        │   chargeable_treatments) ÷ 12   │
        └────────────┬────────────────────┘
                     ↓
        ┌─────────────────────────────────┐
        │ save()                           │
        │ - Persist to database            │
        └─────────────────────────────────┘
```

### Calculator Implementation

```php
namespace Bongo\Estimate\Calculators;

class EstimateItemPriceCalculator
{
    private EstimateItemPrice $estimateItemPrice;
    private EstimateItem $estimateItem;
    private EstimatePlan $estimatePlan;
    private EstimatePlanItem $estimatePlanItem;

    private int $taxRate;
    private int $planCost;

    public function setEstimateItemPrice(EstimateItemPrice $estimateItemPrice): self
    {
        $this->estimateItemPrice = $estimateItemPrice;
        $this->estimateItem = $estimateItemPrice->item;
        $this->estimatePlan = $estimateItemPrice->plan;
        $this->estimatePlanItem = $estimateItemPrice->planItem;

        return $this;
    }

    public function calculate(): self
    {
        $this->setTaxRate();
        $this->getPlanCost();
        $this->applyMinimumPrice();
        $this->calculateCostPerM2();
        $this->calculateSubtotal();
        $this->calculateTax();
        $this->calculateTotal();
        $this->calculatePricePerMonth();

        return $this;
    }

    public function save(): self
    {
        $this->estimateItemPrice->save();
        return $this;
    }

    // Private calculation methods...
}

// Usage:
(new EstimateItemPriceCalculator())
    ->setEstimateItemPrice($estimateItemPrice)
    ->calculate()
    ->save();
```

---

## Event System

The package fires domain events for all major model lifecycle operations.

### Event Architecture

```
┌──────────────────────────────────────────┐
│           Domain Events                   │
├──────────────────────────────────────────┤
│  Estimate                                 │
│  - EstimateUpdated                        │
├──────────────────────────────────────────┤
│  EstimateItem                             │
│  - EstimateItemCreated                    │
│  - EstimateItemUpdated                    │
│  - EstimateItemDeleted                    │
├──────────────────────────────────────────┤
│  EstimateService                          │
│  - ServiceCreated                         │
│  - ServiceUpdated                         │
│  - ServiceDeleted                         │
├──────────────────────────────────────────┤
│  EstimatePlan                             │
│  - PlanCreated                            │
│  - PlanUpdated                            │
│  - PlanDeleted                            │
├──────────────────────────────────────────┤
│  EstimateLocation                         │
│  - LocationCreated                        │
│  - LocationUpdated                        │
│  - LocationDeleted                        │
└──────────────────────────────────────────┘
```

### Event Example

```php
namespace Bongo\Estimate\Events;

use Bongo\Estimate\Models\EstimateItem;
use Illuminate\Broadcasting\InteractsWithSockets;
use Illuminate\Foundation\Events\Dispatchable;
use Illuminate\Queue\SerializesModels;

class EstimateItemCreated
{
    use Dispatchable, InteractsWithSockets, SerializesModels;

    public EstimateItem $estimateItem;

    public function __construct(EstimateItem $estimateItem)
    {
        $this->estimateItem = $estimateItem;
    }
}

// Firing events in controllers:
$item = EstimateItem::create($data);
event(new EstimateItemCreated($item));
```

---

## Service Provider Integration

The package extends `AbstractServiceProvider` from `bongo/framework` for automatic bootstrapping.

### Service Provider Architecture

```
┌──────────────────────────────────────────────────────────┐
│  EstimateServiceProvider extends AbstractServiceProvider │
└──────────────────────────────────────────────────────────┘
                           │
                           ↓
            ┌──────────────────────────┐
            │  protected $module = 'estimate' │
            └──────────────────────────┘
                           │
    ┌──────────────────────┼──────────────────────┐
    ↓                      ↓                      ↓
┌─────────┐          ┌─────────┐           ┌─────────┐
│ Config  │          │ Routes  │           │  Views  │
│ Auto-   │          │ Auto-   │           │  Auto-  │
│ loaded  │          │ loaded  │           │  loaded │
└─────────┘          └─────────┘           └─────────┘
    │                      │                      │
    ↓                      ↓                      ↓
src/Config/         src/Routes/            src/Views/
estimate.php        - backend.php          estimate/
                    - frontend.php
                           │
    ┌──────────────────────┼──────────────────────┐
    ↓                      ↓                      ↓
┌─────────┐          ┌─────────┐           ┌─────────┐
│Migrations│         │Commands │           │Components│
│ Auto-   │          │Registered│          │Registered│
│ loaded  │          │         │           │         │
└─────────┘          └─────────┘           └─────────┘
    │                      │                      │
    ↓                      ↓                      ↓
src/Migrations/     $commands array       bootComponents()
                    - UpgradeToV4         - x-radius-map
                                           - x-estimate-service-dropdown
```

### Service Provider Implementation

```php
namespace Bongo\Estimate;

use Bongo\Estimate\Commands\UpgradeToV4Command;
use Bongo\Estimate\Components\Backend\RadiusMap;
use Bongo\Estimate\Components\Backend\ServiceDropdown;
use Bongo\Framework\Providers\AbstractServiceProvider;
use Illuminate\Support\Facades\Blade;

class EstimateServiceProvider extends AbstractServiceProvider
{
    /**
     * Module name for auto-loading.
     */
    protected string $module = 'estimate';

    /**
     * Artisan commands to register.
     */
    public array $commands = [
        UpgradeToV4Command::class,
    ];

    /**
     * Bootstrap package services.
     */
    public function boot(): void
    {
        parent::boot(); // Auto-loads config, routes, views, migrations

        $this->bootAssets();
        $this->bootComponents();
    }

    /**
     * Publish package assets.
     */
    protected function bootAssets(): void
    {
        $this->publishes([
            __DIR__.'/../public' => public_path('vendor/estimate'),
        ], 'estimate-assets');
    }

    /**
     * Register Blade components.
     */
    protected function bootComponents(): void
    {
        Blade::component('radius-map', RadiusMap::class);
        Blade::component('estimate-service-dropdown', ServiceDropdown::class);
    }
}
```

### Route Middleware Patterns

```php
// backend.php routes (auto-prefixed with 'backend.*')
Route::middleware(['auth', 'employee'])
    ->group(function () {
        Route::get('/estimates', [EstimateController::class, 'index'])
            ->name('backend.estimates.index');
    });

// frontend.php routes (auto-named with 'frontend.*')
Route::middleware(['web'])
    ->group(function () {
        Route::get('/estimate/step-1', [Step1Controller::class, 'show'])
            ->name('frontend.estimate.step-1');

        Route::post('/estimate/step-1/store', [Step1Controller::class, 'store'])
            ->middleware('honeypot')
            ->name('frontend.estimate.step-1.store');
    });
```

---

## Frontend Workflow

The frontend uses a multi-step workflow with session-based state management.

### Frontend Flow Diagram

```
┌──────────────────────────────────────────────────────────────────┐
│                    FRONTEND WORKFLOW                              │
└──────────────────────────────────────────────────────────────────┘

STEP 1: Contact & Service Selection
┌─────────────────────────────────────┐
│ GET /estimate/step-1                │
│ - Display contact form              │
│ - Postcode input with validation    │
│ - Service selection dropdown        │
│ - Terms & marketing consent         │
└────────────┬────────────────────────┘
             ↓
┌─────────────────────────────────────┐
│ POST /estimate/step-1/store         │
│ - Validate postcode (GoogleGeoCode) │
│ - Check coverage area               │
│ - Create/update Estimate            │
│ - Create EstimateItem               │
│ - Store in session:                 │
│   * estimate_uuid                   │
│   * estimate_item_uuid              │
│ - Set step = 1                      │
└────────────┬────────────────────────┘
             ↓
           STEP 2

STEP 2: Area Measurement
┌─────────────────────────────────────┐
│ GET /estimate/step-2                │
│ - Validate step progression         │
│ - Display Google Maps               │
│ - Drawing tools enabled             │
│ - User draws property areas         │
│ - Calculate total area (m²)         │
└────────────┬────────────────────────┘
             ↓
┌─────────────────────────────────────┐
│ POST /estimate/step-2/store         │
│ - Store areas (JSON coordinates)    │
│ - Store total_area_m2               │
│ - Update EstimateItem               │
│ - Set step = 2                      │
└────────────┬────────────────────────┘
             ↓
           STEP 3

STEP 3: Quote Display
┌─────────────────────────────────────┐
│ GET /estimate/step-3                │
│ - Validate step progression         │
│ - Load all EstimatePlans            │
│ - For each plan:                    │
│   * Find EstimatePlanItem by area   │
│   * Create EstimateItemPrice        │
│   * Run EstimateItemPriceCalculator │
│   * Calculate totals                │
│ - Display pricing table             │
│   * Based on pricing_table_type     │
│   * Show all plan options           │
│   * Highlight recommended plan      │
│ - Set step = 3                      │
└─────────────────────────────────────┘
```

### Session Data Flow

```
Session Storage:
┌──────────────────────────────────────┐
│ estimate_uuid: "550e8400-..."        │
│ estimate_item_uuid: "6ba7b810-..."   │
│                                      │
│ estimate_address: [...]              │
│ estimate_contact: [...]              │
│ estimate_item: [...]                 │
└──────────────────────────────────────┘
         ↓
    Database Storage:
┌──────────────────────────────────────┐
│ Estimate (uuid: 550e8400-...)        │
│ ├─ status: draft                     │
│ ├─ step: 1 → 2 → 3                  │
│ ├─ line_1, postcode, lat/lng        │
│ ├─ first_name, email, phone         │
│ └─ EstimateItem (uuid: 6ba7b810-...) │
│     ├─ step: 1 → 2 → 3              │
│     ├─ areas: JSON                   │
│     ├─ total_area_m2: 75.50         │
│     └─ EstimateItemPrice (×3 plans) │
│         ├─ cost_per_m2               │
│         ├─ subtotal                  │
│         ├─ tax                       │
│         ├─ total                     │
│         └─ price_per_month           │
└──────────────────────────────────────┘
```

### Step Validation

```php
// In Step2Controller:
public function show(): View
{
    $item = FindEstimateItem::byUuid(session('estimate_item_uuid'));

    // Validate step progression
    if (!$item->stepIsGte(1)) {
        return redirect()->route('frontend.estimate.step-1');
    }

    return view('estimate::frontend.step_2.show', compact('item'));
}
```

---

## Backend Workflow

The backend provides CRUD interfaces for managing services, plans, locations, and viewing estimates.

### Backend Flow Diagram

```
┌──────────────────────────────────────────────────────────────────┐
│                    BACKEND WORKFLOW                               │
└──────────────────────────────────────────────────────────────────┘

SERVICE CONFIGURATION
┌─────────────────────────────────────┐
│ 1. Create EstimateService           │
│    - Name: "Lawn Care"              │
│    - Key: "lawn-care"               │
│    - Status: active                 │
│    - Default: true/false            │
└────────────┬────────────────────────┘
             ↓
┌─────────────────────────────────────┐
│ 2. Create EstimatePlan (×3)         │
│    - Basic Plan                     │
│      * min_price: 2800              │
│      * treatments: 11               │
│      * chargeable_treatments: 7     │
│    - Enhanced Plan                  │
│    - Premier Plan                   │
└────────────┬────────────────────────┘
             ↓
┌─────────────────────────────────────┐
│ 3. Create EstimatePlanItem (×5 each)│
│    Basic Plan:                      │
│    - 50m² = £28.00 (2800 pence)    │
│    - 100m² = £35.00                │
│    - 150m² = £42.00                │
│    - 200m² = £49.00                │
│    - 250m² = £56.00                │
└────────────┬────────────────────────┘
             ↓
┌─────────────────────────────────────┐
│ 4. Configure Features (config file) │
│    features => [                    │
│      'key1' => ['basic', 'enhanced']│
│      'key2' => ['premier']          │
│    ]                                │
└─────────────────────────────────────┘

COVERAGE MANAGEMENT
┌─────────────────────────────────────┐
│ 1. Create EstimateLocation          │
│    - Name: "Norwich"                │
│    - Radius: 25 (km)                │
│    - Latitude: 52.6308              │
│    - Longitude: 1.2973              │
│    - Fill: "#FF0000"                │
│    - Stroke: "#FF0000"              │
└─────────────────────────────────────┘

ESTIMATE REVIEW
┌─────────────────────────────────────┐
│ GET /estimates                      │
│ - DataTable with filters            │
│ - Status: draft/pending/accepted    │
│ - Date range filtering              │
│ - Postcode search                   │
└────────────┬────────────────────────┘
             ↓
┌─────────────────────────────────────┐
│ GET /estimates/{estimate}           │
│ - View estimate details             │
│ - Contact information               │
│ - Address with static map           │
│ - List of EstimateItems             │
│ - Export status                     │
└────────────┬────────────────────────┘
             ↓
┌─────────────────────────────────────┐
│ POST /estimates/{estimate}/update   │
│ - Update status                     │
│ - Add notes                         │
│ - Mark as exported                  │
│ - Fire EstimateUpdated event        │
└─────────────────────────────────────┘
```

---

## Maps Integration

The package includes comprehensive Google Maps integration for both static and interactive maps.

### Maps Architecture

```
┌──────────────────────────────────────────────────────────────────┐
│                    MAPS SYSTEM ARCHITECTURE                       │
└──────────────────────────────────────────────────────────────────┘

Base Classes:
┌─────────────────┐
│    Map          │ (Abstract base)
│  - key          │
│  - center       │
│  - zoom         │
│  - markers[]    │
│  - paths[]      │
└────────┬────────┘
         │
    ┌────┴────┐
    ↓         ↓
┌─────────┐ ┌─────────┐
│StaticMap│ │RadiusMap│
└─────────┘ └─────────┘

URL Generators:
┌─────────────────┐
│  URLGenerator   │ (Abstract base)
└────────┬────────┘
         │
    ┌────┴────┐
    ↓         ↓
┌──────────────┐ ┌──────────────┐
│StaticMapUrl  │ │RadiusMapUrl  │
└──────────────┘ └──────────────┘

Geometry Classes:
┌─────────┐
│  Point  │ (Single coordinate)
│  - lat  │
│  - lng  │
└─────────┘

┌─────────┐
│ Marker  │ (Map marker)
│  - locations[]│
│  - color      │
│  - size       │
│  - label      │
└─────────┘

┌─────────┐
│  Path   │ (Line/polygon)
│  - coordinates[]│
│  - color        │
│  - weight       │
│  - fillcolor    │
└─────────┘

┌─────────┐
│ Circle  │ (Radius circle)
│  - center      │
│  - radius      │
│  - fillColor   │
│  - strokeColor │
└─────────┘

Utilities:
┌─────────────────┐
│PolylineEncoder  │ (Google polyline encoding)
└─────────────────┘

┌─────────────────┐
│ImageGenerator   │ (Static image generation)
└─────────────────┘
```

### Static Map Usage

```php
use Bongo\Estimate\Maps\StaticMap;
use Bongo\Estimate\Maps\Marker;
use Bongo\Estimate\Maps\Path;

$staticMap = (new StaticMap())
    ->setKey(setting('system::credentials.google_maps_api_key'))
    ->setCenter([52.6308, 1.2973])
    ->setZoom(20);

// Add marker
$marker = (new Marker())
    ->addLocation([52.6308, 1.2973])
    ->setColor('red')
    ->setLabel('A');
$staticMap->addMarker($marker);

// Add path (polygon)
$coordinates = [
    [52.6308, 1.2973],
    [52.6309, 1.2974],
    [52.6310, 1.2975],
    [52.6308, 1.2973], // Close polygon
];
$path = (new Path())
    ->setCoordinates($coordinates)
    ->setColor('blue')
    ->setWeight(2)
    ->setFillColor('blue');
$staticMap->addPath($path);

// Generate URL
$url = $staticMap->generateUrl();

// Or save as image
$imagePath = $staticMap->generateImage('estimate-map.png', 'estimates');
```

### Interactive Map (Blade Component)

```blade
{{-- Display radius map in backend --}}
<x-radius-map />

{{-- Renders all active EstimateLocations as circles --}}
```

---

## Price Calculation Flow

Detailed flow of price calculation from area to final monthly price.

### Price Calculation Sequence Diagram

```
User                Step2Controller    EstimateItem    Step3Controller    EstimateItemPriceCalculator
 │                       │                  │                 │                       │
 │ Draw areas on map     │                  │                 │                       │
 │──────────────────────>│                  │                 │                       │
 │                       │                  │                 │                       │
 │                       │ Store areas JSON │                 │                       │
 │                       │─────────────────>│                 │                       │
 │                       │                  │                 │                       │
 │                       │ Calculate total_area_m2           │                       │
 │                       │─────────────────>│                 │                       │
 │                       │                  │                 │                       │
 │ View pricing          │                  │                 │                       │
 │──────────────────────────────────────────────────────────>│                       │
 │                       │                  │                 │                       │
 │                       │                  │                 │ For each plan:        │
 │                       │                  │                 │                       │
 │                       │                  │                 │ 1. Find EstimatePlanItem by area
 │                       │                  │                 │──────────────────────>│
 │                       │                  │                 │                       │
 │                       │                  │                 │ 2. Create EstimateItemPrice
 │                       │                  │                 │──────────────────────>│
 │                       │                  │                 │                       │
 │                       │                  │                 │ 3. setTaxRate()       │
 │                       │                  │                 │                       │─┐
 │                       │                  │                 │                       │ │
 │                       │                  │                 │                       │<┘
 │                       │                  │                 │                       │
 │                       │                  │                 │ 4. getPlanCost()      │
 │                       │                  │                 │                       │─┐
 │                       │                  │                 │                       │ │
 │                       │                  │                 │                       │<┘
 │                       │                  │                 │                       │
 │                       │                  │                 │ 5. applyMinimumPrice()│
 │                       │                  │                 │                       │─┐
 │                       │                  │                 │                       │ │
 │                       │                  │                 │                       │<┘
 │                       │                  │                 │                       │
 │                       │                  │                 │ 6. calculateCostPerM2()│
 │                       │                  │                 │                       │─┐
 │                       │                  │                 │                       │ │
 │                       │                  │                 │                       │<┘
 │                       │                  │                 │                       │
 │                       │                  │                 │ 7. calculateSubtotal()│
 │                       │                  │                 │                       │─┐
 │                       │                  │                 │                       │ │
 │                       │                  │                 │                       │<┘
 │                       │                  │                 │                       │
 │                       │                  │                 │ 8. calculateTax()     │
 │                       │                  │                 │                       │─┐
 │                       │                  │                 │                       │ │
 │                       │                  │                 │                       │<┘
 │                       │                  │                 │                       │
 │                       │                  │                 │ 9. calculateTotal()   │
 │                       │                  │                 │                       │─┐
 │                       │                  │                 │                       │ │
 │                       │                  │                 │                       │<┘
 │                       │                  │                 │                       │
 │                       │                  │                 │ 10. calculatePricePerMonth()
 │                       │                  │                 │                       │─┐
 │                       │                  │                 │                       │ │
 │                       │                  │                 │                       │<┘
 │                       │                  │                 │                       │
 │                       │                  │                 │ 11. save()            │
 │                       │                  │                 │                       │─┐
 │                       │                  │                 │                       │ │
 │                       │                  │                 │                       │<┘
 │                       │                  │                 │<──────────────────────│
 │                       │                  │                 │                       │
 │<──────────────────────────────────────────────────────────│                       │
 │                       │                  │                 │                       │
 │ Display pricing table │                  │                 │                       │
```

### Price Calculation Example

```
Input:
- Total Area: 75.50 m²
- EstimatePlanItem: 100m² = £35.00 (3500 pence)
- Plan Min Price: £28.00 (2800 pence)
- Tax Rate: 20%
- Chargeable Treatments: 7

Calculation Steps:
1. setTaxRate() → 20%
2. getPlanCost() → 3500 pence (from EstimatePlanItem for 100m²)
3. applyMinimumPrice() → max(3500, 2800) = 3500 pence
4. calculateCostPerM2() → 3500 ÷ 75.50 = 46.35 pence per m²
5. calculateSubtotal() → 3500 pence = £35.00
6. calculateTax() → 3500 × 20 ÷ 100 = 700 pence = £7.00
7. calculateTotal() → 3500 + 700 = 4200 pence = £42.00
8. calculatePricePerMonth() → (4200 × 7) ÷ 12 = 2450 pence = £24.50

Output (EstimateItemPrice):
- cost_per_m2: 46.35 pence
- subtotal: 3500 pence (£35.00)
- tax_rate: 20.00%
- tax: 700 pence (£7.00)
- total: 4200 pence (£42.00)
- price_per_month: 2450 pence (£24.50)
```

---

## Validation and Security

### Form Validation

```php
// src/Http/Requests/StoreStep1Request.php
class StoreStep1Request extends FormRequest
{
    public function rules(): array
    {
        return [
            'postcode' => ['required', 'string', 'max:10'],
            'first_name' => ['required', 'string', 'max:255'],
            'last_name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'email', 'max:255'],
            'phone' => ['required', 'string', 'max:20'],
            'accepted_terms' => ['required', 'accepted'],
            'marketing_emails' => ['nullable', 'boolean'],
            'estimate_service_id' => ['required', 'exists:estimate_services,id'],
        ];
    }
}
```

### Security Features

1. **Honeypot Protection**: Spam protection on all frontend POST routes
   ```php
   Route::post('/estimate/step-1/store', [Step1Controller::class, 'store'])
       ->middleware('honeypot');
   ```

2. **Step Validation**: Prevent skipping steps in workflow
   ```php
   if (!$item->stepIs(2)) {
       return redirect()->route('frontend.estimate.step-1');
   }
   ```

3. **Coverage Area Validation**: Ensure address is within service area
   ```php
   if (!$estimate->isWithinCoverageArea($estimate->latitude, $estimate->longitude)) {
       throw new EstimateLocationException('Address outside coverage area.');
   }
   ```

4. **UUID-Based Lookups**: No sequential IDs exposed
   ```php
   $estimate = FindEstimate::byUuid(session('estimate_uuid'));
   ```

5. **Status-Based Filtering**: Only show draft/pending estimates in frontend
   ```php
   public static function byUuid(?string $uuid): ?Estimate
   {
       return Estimate::where('uuid', $uuid)
           ->whereIn('status', [Estimate::DRAFT, Estimate::PENDING])
           ->first();
   }
   ```

---

## Testing Architecture

### Test Structure

```
tests/
├── Unit/
│   ├── Actions/
│   │   ├── CalculateDistanceTest.php
│   │   └── GenerateNumberTest.php
│   └── Models/
│       ├── EstimateTest.php
│       ├── EstimateLocationTest.php
│       ├── EstimateServiceTest.php
│       └── EstimatePlanTest.php
├── Feature/
│   └── (Feature tests)
├── TestCase.php
└── ExampleTest.php
```

### Base Test Class

```php
namespace Bongo\Estimate\Tests;

use Orchestra\Testbench\TestCase as Orchestra;

abstract class TestCase extends Orchestra
{
    protected function setUp(): void
    {
        parent::setUp();

        $this->setUpDatabase();
        $this->seedSettings();
    }

    protected function getPackageProviders($app): array
    {
        return [
            EstimateServiceProvider::class,
        ];
    }

    protected function setUpDatabase(): void
    {
        // Create in-memory SQLite database
        // Define table schemas manually
    }

    protected function seedSettings(): void
    {
        // Set default settings for tests
        setting()->set('package::estimate.tax_rate', 20);
        setting()->set('package::estimate.chargeable_treatments', 7);
    }
}
```

### Factory Usage in Tests

```php
public function test_can_create_estimate_with_items(): void
{
    $estimate = EstimateFactory::new()->create([
        'postcode' => 'NR1 1AA',
        'status' => Estimate::PENDING,
    ]);

    $item = EstimateItemFactory::new()
        ->for($estimate)
        ->create([
            'total_area_m2' => 75.50,
        ]);

    $this->assertTrue($estimate->hasItems());
    $this->assertEquals('NR1 1AA', $estimate->postcode);
    $this->assertEquals(75.50, $item->total_area_m2);
}
```

---

## Extension Points

### Adding a New Model

1. Create model class extending `AbstractModel`
2. Add appropriate traits (HasUUID, HasStatus, etc.)
3. Define relationships
4. Create migration
5. Create factory
6. Create tests
7. Create events (Created, Updated, Deleted)
8. Add to service provider if needed

### Adding a New Action

1. Create class in `src/Actions/`
2. Implement static methods with clear return types
3. Add tests in `tests/Unit/Actions/`
4. Document in this file

### Adding a New Calculator

1. Create class in `src/Calculators/`
2. Implement fluent builder pattern
3. Add `calculate()` and `save()` methods
4. Keep calculation logic in private methods
5. Add tests
6. Document calculation flow

### Adding a New Trait

1. Create in `src/Traits/`
2. Implement `initializeHas*()` for fillable fields
3. Add accessors, scopes, helper methods
4. Add to trait reference table in this document
5. Add tests

### Adding a New Frontend Step

1. Create controller in `src/Http/Controllers/Frontend/`
2. Add routes in `src/Routes/frontend.php`
3. Create form request for validation
4. Add honeypot to POST route
5. Validate step progression
6. Update session data
7. Create view in `src/Views/estimate/frontend/step_X/`
8. Add tests

### Adding a New Backend Feature

1. Create controller in `src/Http/Controllers/Backend/`
2. Add routes in `src/Routes/backend.php`
3. Create form requests
4. Fire domain events
5. Create views
6. Add datatable controller if needed
7. Add tests

---

## How to Add Features

### Example: Adding a New Service Type

```php
// 1. Backend: Create service via ServiceController
$service = EstimateService::create([
    'name' => 'Pest Control',
    'key' => 'pest-control',
    'status' => EstimateService::ACTIVE,
    'default' => false,
]);

// 2. Create pricing tiers
$basicPlan = EstimatePlan::create([
    'estimate_service_id' => $service->id,
    'name' => 'Basic',
    'key' => 'basic',
    'min_price' => 2500,
    'treatments' => 4,
    'chargeable_treatments' => 4,
    'price_type' => 'per_treatment',
]);

// 3. Add pricing breakpoints
EstimatePlanItem::create([
    'estimate_plan_id' => $basicPlan->id,
    'area_m2' => 100,
    'cost_per_m2' => 30,
    'sort_order' => 1,
]);

// 4. Frontend: Service now available in dropdown
<x-estimate-service-dropdown />
```

### Example: Adding a Custom Price Calculation

```php
// 1. Create new calculator class
namespace Bongo\Estimate\Calculators;

class CustomPriceCalculator
{
    private EstimateItemPrice $price;

    public function setPrice(EstimateItemPrice $price): self
    {
        $this->price = $price;
        return $this;
    }

    public function calculate(): self
    {
        // Custom calculation logic
        return $this;
    }

    public function save(): self
    {
        $this->price->save();
        return $this;
    }
}

// 2. Use in controller
(new CustomPriceCalculator())
    ->setPrice($estimateItemPrice)
    ->calculate()
    ->save();
```

### Example: Adding a New Coverage Area

```php
// Backend: Create location via LocationController
$location = EstimateLocation::create([
    'name' => 'Cambridge',
    'key' => 'cambridge',
    'radius' => 20, // km
    'latitude' => 52.2053,
    'longitude' => 0.1218,
    'fill' => '#0000FF',
    'stroke' => '#0000FF',
    'status' => EstimateLocation::ACTIVE,
]);

// Frontend: Validation automatically checks coverage
$estimate->isWithinCoverageArea($latitude, $longitude);
```

---

## Summary

The `bongo/estimate` package follows a well-structured architecture with:

- **Clear separation of concerns**: Models, Controllers, Actions, Calculators, Services
- **Trait-based composition**: Reusable functionality across models
- **Event-driven design**: Integration points for external systems
- **Type-safe PHP**: Strict types and return type declarations
- **Comprehensive testing**: Unit and feature tests with factories
- **Database-driven configuration**: Flexible service/plan/pricing management
- **Multi-step workflow**: Session-based state management
- **Maps integration**: Google Maps for area measurement and display
- **Security-first**: Honeypot protection, step validation, coverage checks

For code style guidelines, see `.cursorrules`.
For AI assistant instructions, see `CLAUDE.md` and `.github/copilot-instructions.md`.
For installation and usage, see `README.md`.
