<?php

declare(strict_types=1);

namespace Bongo\Estimate\Calculators;

use Bongo\Estimate\Models\EstimateItem;
use Bongo\Estimate\Models\EstimateItemPrice;
use Bongo\Estimate\Models\EstimatePlan;
use Bongo\Estimate\Models\EstimatePlanItem;
use Bongo\Framework\Helpers\Tax;
use Illuminate\Contracts\Container\BindingResolutionException;

class EstimateItemPriceCalculator
{
    private EstimatePlan $estimatePlan;

    private EstimatePlanItem $estimatePlanItem;

    private EstimateItem $estimateItem;

    private EstimateItemPrice $estimateItemPrice;

    private float $planPrice = 0.00;

    private float $totalAreaM2 = 0.00;

    private float $costPerM2 = 0.00;

    private float $subtotal = 0.00;

    private float $taxRate = 0.00;

    private float $tax = 0.00;

    private float $total = 0.00;

    private int $chargeableTreatments = 0;

    private float $pricePerMonth = 0.00;

    public function setEstimateItemPrice(EstimateItemPrice $estimateItemPrice): self
    {
        $this->estimatePlan = $estimateItemPrice->plan;
        $this->estimatePlanItem = $estimateItemPrice->planItem;
        $this->estimateItem = $estimateItemPrice->item;
        $this->estimateItemPrice = $estimateItemPrice;

        return $this;
    }

    /** @throws BindingResolutionException */
    public function calculate(): self
    {
        $this->setTaxRate()
            ->setPlanPrice()
            ->setTotalAreaM2()
            ->setCostPerM2()
            ->setSubtotal()
            ->setTax()
            ->setTotal()
            ->setChargeableTreatments()
            ->setPricePerMonth();

        return $this;
    }

    public function save(): EstimateItemPrice
    {
        $this->estimateItemPrice->cost_per_m2 = $this->costPerM2;
        $this->estimateItemPrice->subtotal = $this->subtotal;
        $this->estimateItemPrice->tax_rate = $this->taxRate;
        $this->estimateItemPrice->tax = $this->tax;
        $this->estimateItemPrice->total = $this->total;
        $this->estimateItemPrice->price_per_month = $this->pricePerMonth;
        $this->estimateItemPrice->save();

        return $this->estimateItemPrice;
    }

    /** @throws BindingResolutionException */
    private function setTaxRate(): self
    {
        $this->taxRate = setting('package::estimate.tax_rate', 20);

        return $this;
    }

    private function setPlanPrice(): self
    {
        // Column cost_per_m2 in estimate_plan_items needs to be renamed to price
        $this->planPrice = $this->estimatePlanItem->cost_per_m2;

        // Handle the minimum price for the plan
        if ($this->estimatePlan->isFlatPrice() && $this->planPrice < $this->estimatePlan->min_price) {
            $this->planPrice = $this->estimatePlan->min_price;
        }

        // Handle the case where prices include tax
        if (config('estimate.prices_include_tax')) {
            $this->planPrice = Tax::excluding($this->planPrice, $this->taxRate);
        }

        return $this;
    }

    private function setTotalAreaM2(): self
    {
        $this->totalAreaM2 = (float) $this->estimateItem->total_area_m2;

        return $this;
    }

    private function setCostPerM2(): self
    {
        // If this a "Price Per M2" plan
        if ($this->estimatePlan->isPricePerM2()) {
            $this->costPerM2 = $this->planPrice;

            return $this;
        }

        // Otherwise this is a "Flat Price" plan so calculate cost per m2
        $this->costPerM2 = ! empty($this->totalAreaM2)
            ? $this->planPrice / $this->totalAreaM2
            : 0;

        return $this;
    }

    private function setSubtotal(): self
    {
        // If this a "Price Per M2" plan
        if ($this->estimatePlan->isPricePerM2()) {
            $this->subtotal = $this->planPrice * $this->totalAreaM2;

            // Handle the minimum price for the plan
            if ($this->subtotal < $this->estimatePlan->min_price) {
                $this->subtotal = $this->estimatePlan->min_price;
            }

            return $this;
        }

        // Otherwise this is a "Flat Price" plan
        $this->subtotal = ! empty($this->planPrice)
            ? $this->planPrice
            : 0;

        return $this;
    }

    private function setTax(): self
    {
        $this->tax = Tax::calculate($this->subtotal, $this->taxRate);

        return $this;
    }

    private function setTotal(): self
    {
        $this->total = $this->subtotal + $this->tax;

        return $this;
    }

    private function setChargeableTreatments(): self
    {
        $this->chargeableTreatments = (int) $this->estimatePlan->chargeable_treatments;

        return $this;
    }

    private function setPricePerMonth(): self
    {
        $monthsInYear = 12;

        $this->pricePerMonth = ! empty($this->total)
            ? ($this->total * $this->chargeableTreatments) / $monthsInYear
            : 0;

        return $this;
    }
}
