<?php

declare(strict_types=1);

namespace Bongo\Estimate\Http\Controllers\Backend;

use Bongo\Estimate\Data\SqueegeeData;
use Bongo\Estimate\Exports\SqueegeeExport;
use Bongo\Estimate\Models\EstimateItem;
use Bongo\Framework\Http\Controllers\AbstractController;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Facades\Excel;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\StreamedResponse;

class ExportController extends AbstractController
{
    public function index(): BinaryFileResponse|StreamedResponse|RedirectResponse
    {
        // Shared query for all exports
        $rows = EstimateItem::query()
            ->select([
                'estimate_items.id',
                'estimate_items.uuid',

                'estimate_items.estimate_id',
                'estimates.date as estimate_date',
                'estimates.number as estimate_number',

                'estimate_items.number',
                'estimate_items.step',
                'estimate_items.status',

                'estimates.first_name',
                'estimates.last_name',
                'estimates.email',
                'estimates.phone',

                'estimates.line_1',
                'estimates.line_2',
                'estimates.line_3',
                'estimates.city',
                'estimates.county',
                'estimates.postcode',
                'estimates.latitude',
                'estimates.longitude',

                'estimates.accepted_terms',
                'estimates.marketing_emails',
                'estimates.voucher_code',

                'estimate_items.total_area_m2',
                'estimate_services.name as service_name',

                'estimate_items.created_at',
                'estimate_items.updated_at',
                'estimate_items.sent_at',
                'estimate_items.deleted_at',
            ])
            ->leftJoin('estimates', 'estimate_items.estimate_id', '=', 'estimates.id')
            ->leftJoin('estimate_services', 'estimate_items.estimate_service_id', '=', 'estimate_services.id')
            ->whereNot('estimate_items.status', 'draft')
            ->orderBy('estimate_items.estimate_id')
            ->get();

        try {
            return Excel::download(
                $this->getExportDataForSqueegee($rows),
                'estimates_'.now()->format('Y-m-d_H-i').'.xlsx'
            );

        } catch (Exception $e) {
            return redirect()->back()->error($e->getMessage());
        }
    }

    private function getExportDataForSqueegee(Collection $rows): SqueegeeExport
    {
        $rows = $rows->map(function (Model $row) {
            return (new SqueegeeData())->toArray($row);
        });

        return new SqueegeeExport($rows);
    }
}
