<?php

declare(strict_types=1);

namespace Bongo\Estimate\Http\Controllers\Backend;

use Bongo\Estimate\Events\LocationCreated;
use Bongo\Estimate\Events\LocationDeleted;
use Bongo\Estimate\Events\LocationUpdated;
use Bongo\Estimate\Http\Requests\Backend\StoreLocationRequest;
use Bongo\Estimate\Http\Requests\Backend\UpdateLocationRequest;
use Bongo\Estimate\Models\EstimateLocation;
use Bongo\Framework\Http\Controllers\AbstractController;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class LocationController extends AbstractController
{
    protected EstimateLocation $estimateLocation;

    public function __construct(EstimateLocation $estimateLocation)
    {
        $this->estimateLocation = $estimateLocation;
    }

    public function index(): View
    {
        return view('estimate::backend.location.index');
    }

    public function create(): View
    {
        $estimateLocation = $this->estimateLocation;

        return view('estimate::backend.location.create', compact('estimateLocation'));
    }

    public function store(StoreLocationRequest $request): RedirectResponse
    {
        $estimateLocation = $this->estimateLocation->create($request->all());
        event(new LocationCreated($estimateLocation));

        return redirect()
            ->route('backend.estimate_location.show', $estimateLocation->id)
            ->success(trans('estimate::backend.location_store_success'));
    }

    public function show(EstimateLocation $estimateLocation): View
    {
        return view('estimate::backend.location.show', compact('estimateLocation'));
    }

    public function edit(EstimateLocation $estimateLocation): View
    {
        return view('estimate::backend.location.edit', compact('estimateLocation'));
    }

    public function update(UpdateLocationRequest $request, EstimateLocation $estimateLocation): RedirectResponse
    {
        $estimateLocation->update($request->all());
        event(new LocationUpdated($estimateLocation));

        return redirect()
            ->route('backend.estimate_location.show', $estimateLocation->id)
            ->success(trans('estimate::backend.location_update_success'));
    }

    public function destroy(EstimateLocation $estimateLocation)
    {
        if ($estimateLocation->delete()) {
            event(new LocationDeleted($estimateLocation));
            $redirect = back();

            // If the entity was deleted from their detail view,
            // we'll redirect back to the entity list view.
            if (route('backend.estimate_location.show', $estimateLocation) == $redirect->getTargetUrl()) {
                $redirect = redirect()->route('backend.estimate_location.index');
            }

            return $redirect
                ->success(trans('estimate::backend.location_delete_success'));
        }

        return back()
            ->error(trans('estimate::backend.location_delete_failed'));
    }
}
