<?php

declare(strict_types=1);

namespace Bongo\Estimate\Http\Controllers\Backend;

use Bongo\Estimate\Events\ServiceCreated;
use Bongo\Estimate\Events\ServiceDeleted;
use Bongo\Estimate\Events\ServiceUpdated;
use Bongo\Estimate\Http\Requests\Backend\StoreServiceRequest;
use Bongo\Estimate\Http\Requests\Backend\UpdateServiceRequest;
use Bongo\Estimate\Models\EstimateService;
use Bongo\Framework\Http\Controllers\AbstractController;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class ServiceController extends AbstractController
{
    protected EstimateService $estimateService;

    public function __construct(EstimateService $estimateService)
    {
        $this->estimateService = $estimateService;
    }

    public function index(): View
    {
        return view('estimate::backend.service.index');
    }

    public function create(): View
    {
        $estimateService = $this->estimateService;

        return view('estimate::backend.service.create', compact('estimateService'));
    }

    public function store(StoreServiceRequest $request): RedirectResponse
    {
        $estimateService = $this->estimateService->create($request->all());
        event(new ServiceCreated($estimateService));

        return redirect()
            ->route('backend.estimate_service.show', $estimateService->id)
            ->success(trans('estimate::backend.service_store_success'));
    }

    public function show(EstimateService $estimateService): View
    {
        return view('estimate::backend.service.show', compact('estimateService'));
    }

    public function edit(EstimateService $estimateService): View
    {
        return view('estimate::backend.service.edit', compact('estimateService'));
    }

    public function update(UpdateServiceRequest $request, EstimateService $estimateService): RedirectResponse
    {
        $estimateService->update($request->all());
        event(new ServiceUpdated($estimateService));

        return redirect()
            ->route('backend.estimate_service.show', $estimateService->id)
            ->success(trans('estimate::backend.service_update_success'));
    }

    public function destroy(EstimateService $estimateService)
    {
        if ($estimateService->delete()) {
            event(new ServiceDeleted($estimateService));
            $redirect = back();

            // If the entity was deleted from their detail view,
            // we'll redirect back to the entity list view.
            if (route('backend.estimate_service.show', $estimateService) == $redirect->getTargetUrl()) {
                $redirect = redirect()->route('backend.estimate_service.index');
            }

            return $redirect
                ->success(trans('estimate::backend.service_delete_success'));
        }

        return back()
            ->error(trans('estimate::backend.service_delete_failed'));
    }
}
