# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Quick Links to Documentation

- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Comprehensive architecture documentation with diagrams and detailed explanations
- **[.cursorrules](.cursorrules)** - Cursor AI instructions with coding conventions and common tasks
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)** - GitHub Copilot instructions with code templates
- **[README.md](README.md)** - Installation, usage, and getting started guide
- **[CHANGELOG.md](CHANGELOG.md)** - Version history and changes

## Project Overview

This is a Laravel package (`bongo/estimate`) that provides an estimation/quoting system for service-based businesses. The package manages estimates with geographic service areas, tiered pricing plans, and automated calculations for recurring service costs. It includes Google Maps integration and supports multi-step estimate workflows.

**For detailed architecture documentation, see [ARCHITECTURE.md](ARCHITECTURE.md).**

## Requirements

- PHP 8.2+
- Laravel 10+ or Laravel 11+
- Bongo Framework 3.0+
- Google Maps API key (for map features)
- Spatie Laravel Honeypot 4.0+ (spam protection)
- Spatie Geocoder 3.12+ (address geocoding)

## Common Commands

### Testing
```bash
# Run all tests (no coverage)
composer test

# Run tests with coverage (requires Xdebug)
composer test:coverage

# Run a single test
vendor/bin/phpunit --filter TestMethodName
```

### Code Quality
```bash
# Run static analysis (PHPStan)
composer analyse

# Format code (Laravel Pint)
composer format
```

### Development
```bash
# Build the package
composer build

# Start test server
composer start

# Clear testbench cache
composer clear
```

## Quick Architecture Reference

> **See [ARCHITECTURE.md](ARCHITECTURE.md) for comprehensive details with diagrams and flow charts.**

### Core Domain Models

The package revolves around a hierarchical structure:

**EstimateService** (top-level category, e.g., "Lawn Care")
  └─ **EstimatePlan** (pricing tiers, e.g., "Basic", "Enhanced", "Premier")
      └─ **EstimatePlanItem** (area-based pricing rules, e.g., "50m² = £28", "100m² = £35")

**Estimate** (customer quote)
  └─ **EstimateItem** (specific service/plan selection with location)
      └─ **EstimateItemPrice** (calculated pricing for selected plan)

**EstimateLocation** (service coverage areas with radius)

### Key Patterns

- **Trait Composition**: Models use multiple traits instead of inheritance (e.g., `HasStatus`, `HasAddress`, `HasContact`)
- **Action Classes**: Static utility classes for business logic (e.g., `CalculateDistance::inMiles()`)
- **Calculator Pattern**: Fluent builder pattern for complex calculations (e.g., `EstimateItemPriceCalculator`)
- **Event-Driven**: Domain events fired for all major operations (e.g., `EstimateItemCreated`)
- **Multi-Step Workflow**: Session-based frontend workflow (Step 1 → Step 2 → Step 3)

### Configuration

The package uses `src/Config/estimate.php` for settings:
- Route prefixes (`prefix`, `backend_prefix`)
- Number format prefix (`number_prefix`: "KLC-")
- Map settings (`map_center`, zoom levels)
- Tax configuration (`prices_include_tax`, default 20%)
- Treatment schedules (`treatments_per_year`: 11, `chargeable_treatments_per_year`: 7)
- Pricing table display (`pricing_table_type`: default/with_vat/with_features)
- Plan features mapping (`features` array)
- Cleanup settings (`cleanup_days`: 7)

## Key Files Reference

| File | Purpose | Line Reference |
|------|---------|----------------|
| `src/EstimateServiceProvider.php` | Service provider registration | - |
| `src/Config/estimate.php` | Package configuration | - |
| `src/Models/Estimate.php` | Customer quote model | Line ~30 |
| `src/Models/EstimateItem.php` | Quote item with areas | Line ~35 |
| `src/Models/EstimateService.php` | Service category model | Line ~25 |
| `src/Models/EstimatePlan.php` | Pricing tier model | Line ~30 |
| `src/Models/EstimatePlanItem.php` | Area pricing breakpoint | Line ~20 |
| `src/Models/EstimateLocation.php` | Coverage area model | Line ~25 |
| `src/Calculators/EstimateItemPriceCalculator.php` | Price calculation logic | Line ~15 |
| `src/Actions/CalculateDistance.php` | Distance calculations | Line ~10 |
| `src/Actions/GenerateNumber.php` | Reference number generation | Line ~10 |
| `src/Actions/GetPrice.php` | Price retrieval | Line ~15 |
| `src/Http/Controllers/Frontend/Step1Controller.php` | Contact/service selection | Line ~20 |
| `src/Http/Controllers/Frontend/Step2Controller.php` | Area measurement | Line ~20 |
| `src/Http/Controllers/Frontend/Step3Controller.php` | Quote display | Line ~20 |
| `src/Http/Controllers/Backend/EstimateController.php` | Admin estimate management | Line ~20 |
| `src/Services/GoogleGeoCode.php` | Address geocoding | Line ~15 |
| `src/Maps/StaticMap.php` | Static map generation | Line ~20 |
| `tests/TestCase.php` | Base test class | Line ~15 |

## Code Style Summary

> **See [.cursorrules](.cursorrules) for complete coding conventions.**

- **Strict types**: All files use `declare(strict_types=1)`
- **Return types**: Always declare return types on methods
- **Laravel Pint** with Laravel preset (run: `composer format`)
- **PHPStan** level 1 (run: `composer analyse`)
- **UUID Primary Keys**: All models use `HasUUID` trait
- **Status Constants**: Use interface constants (e.g., `Estimate::PENDING`) not strings
- **Trait Composition**: Prefer traits over inheritance for cross-cutting concerns

## Package Integration with Bongo Framework

- **Service Provider**: `EstimateServiceProvider` extends `AbstractServiceProvider` from `bongo/framework`
- **Auto-loaded by AbstractServiceProvider**:
  - Config: `src/Config/estimate.php` (accessed via `config('estimate.*')`)
  - Routes: `src/Routes/backend.php`, `src/Routes/frontend.php`
  - Views: `src/Views/estimate/` (accessed via `view('estimate::...')`)
  - Migrations: `src/Migrations/` (auto-run on migration)
  - Translations: `src/Translations/`
- **Route Middleware**:
  - `backend.php` → `backend.*` named routes with `auth` + `employee` middleware
  - `frontend.php` → `frontend.*` named routes with standard web middleware
- **Blade Components**:
  - `<x-radius-map />` - Backend radius map for coverage areas
  - `<x-estimate-service-dropdown />` - Service selector dropdown
- **Commands**:
  - `php artisan estimate:upgrade-to-v4` - Migrate v3 data to v4 structure

## Important Notes

- **Version 4.0**: New database-driven architecture (upgraded from config-based v3)
- **UUID-based**: All models use UUID primary keys instead of auto-increment
- **Soft Deletes**: Major models use soft deletes (can be restored)
- **Event-Driven**: Events fired for all major operations (external integration points)
- **Google Maps Required**: Map features require API key in settings (`setting('system::credentials.google_maps_api_key')`)
- **Tax Handling**: Prices can be tax-inclusive or tax-exclusive (config: `prices_include_tax`)
- **Honeypot Protection**: Frontend forms protected from spam
- **Coverage Validation**: Addresses validated against `EstimateLocation` radius
- **Session-Based**: Frontend workflow uses session to track progress across steps
- **Step Validation**: Users cannot skip steps (enforced via `stepIs()`, `stepIsGt()`, etc.)
- **Cleanup Job**: Draft estimates auto-deleted after X days (config: `cleanup_days`)

## Related Documentation

For more detailed information, see:
- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Complete architecture with diagrams, flow charts, and extension guides
- **[.cursorrules](.cursorrules)** - Coding conventions, common tasks, and best practices
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)** - Code templates and patterns
- **[README.md](README.md)** - Installation, configuration, and usage instructions
