# Bongo Estimate Package

A comprehensive Laravel package for managing service estimates and quotes with geographic coverage areas, tiered pricing plans, and automated cost calculations. Perfect for service-based businesses that need location-based pricing with recurring service schedules.

## Documentation

- **[ARCHITECTURE.md](ARCHITECTURE.md)** - Complete architecture documentation with diagrams, flows, and extension guides
- **[CLAUDE.md](CLAUDE.md)** - AI assistant guidance for Claude Code
- **[.cursorrules](.cursorrules)** - Cursor AI instructions with coding conventions
- **[.github/copilot-instructions.md](.github/copilot-instructions.md)** - GitHub Copilot instructions with code templates

## Features

- **Multi-tier Pricing Plans** - Create multiple service tiers (e.g., Basic, Enhanced, Premier) with different pricing structures
- **Area-based Pricing** - Calculate costs based on measured area (m²) with configurable pricing breakpoints
- **Geographic Service Areas** - Define service coverage areas with radius-based location validation
- **Google Maps Integration** - Interactive map drawing, geocoding, and static map generation
- **Multi-step Estimate Workflow** - Guided customer journey from postcode lookup to final quote
- **Automated Calculations** - Tax calculations, monthly pricing, per-treatment costs
- **Reference Number Generation** - Auto-generated unique estimate references (e.g., "KLC-0001")
- **Export Tracking** - Track when estimates are exported to external systems
- **Email Notifications** - Automated email delivery with customizable templates
- **Anti-spam Protection** - Built-in honeypot and reCAPTCHA integration

## Requirements

- PHP 8.2+
- Laravel 10+ or Laravel 11+
- Bongo Framework 3.0+
- Google Maps API key (for map functionality)

## Installation

### 1. Install via Composer

```bash
composer require bongo/estimate
```

The package uses Laravel auto-discovery, so the service provider will be registered automatically.

### 2. Publish Configuration

```bash
php artisan vendor:publish --tag=estimate:config
```

### 3. Run Migrations

```bash
php artisan migrate
```

This will create the following tables:
- `estimates` - Customer quotes
- `estimate_items` - Individual services/plans selected
- `estimate_item_prices` - Calculated pricing for each item
- `estimate_services` - Top-level service categories
- `estimate_plans` - Pricing tier definitions
- `estimate_plan_items` - Area-based pricing rules
- `estimate_locations` - Geographic service coverage areas

### 4. Publish Assets (Optional)

```bash
php artisan vendor:publish --tag=estimate:assets
```

### 5. Configure Google Maps

Add your Google Maps API key to your application's settings or environment configuration. The package uses the Spatie Geocoder package for geocoding functionality.

## Configuration

The main configuration file is located at `config/estimate.php` after publishing. Key settings include:

```php
return [
    // Route prefixes
    'prefix' => 'estimate',
    'backend_prefix' => 'estimates',

    // Reference number prefix
    'number_prefix' => 'KLC-',

    // Map settings
    'map_center' => [52.600850, 1.175440], // [Latitude, Longitude]

    // Tax configuration
    'prices_include_tax' => false,

    // Treatment schedule
    'treatments_per_year' => 11,
    'chargeable_treatments_per_year' => 7,

    // Cleanup old estimates after X days
    'cleanup_days' => 7,

    // reCAPTCHA
    'recaptcha' => [
        'enabled' => true,
        'min_score' => 0.3,
    ],
];
```

## Basic Usage

### Setting Up Services and Plans

1. **Create a Service** (e.g., "Lawn Care")
```php
use Bongo\Estimate\Models\EstimateService;

$service = EstimateService::create([
    'name' => 'Lawn Care',
    'key' => 'lawn-care',
    'status' => EstimateService::ACTIVE,
    'default' => true,
]);
```

2. **Create Plans for the Service** (e.g., "Basic", "Enhanced", "Premier")
```php
use Bongo\Estimate\Models\EstimatePlan;

$plan = EstimatePlan::create([
    'estimate_service_id' => $service->id,
    'name' => 'Basic',
    'key' => 'basic',
    'min_price' => 2800, // £28.00 in pence
    'treatments' => 11,
    'chargeable_treatments' => 7,
    'visits' => 11,
    'status' => EstimatePlan::ACTIVE,
]);
```

3. **Add Pricing Breakpoints to Plans**
```php
use Bongo\Estimate\Models\EstimatePlanItem;

// £28 for lawns up to 50m²
EstimatePlanItem::create([
    'estimate_plan_id' => $plan->id,
    'area_m2' => 50,
    'cost_per_m2' => 2800, // £28.00 in pence
    'sort_order' => 1,
]);

// £35 for lawns up to 100m²
EstimatePlanItem::create([
    'estimate_plan_id' => $plan->id,
    'area_m2' => 100,
    'cost_per_m2' => 3500, // £35.00 in pence
    'sort_order' => 2,
]);
```

4. **Define Service Coverage Areas**
```php
use Bongo\Estimate\Models\EstimateLocation;

EstimateLocation::create([
    'name' => 'Norwich',
    'key' => 'norwich',
    'radius' => 15, // kilometres
    'latitude' => 52.6309,
    'longitude' => 1.2974,
    'fill' => '#FF0000',
    'stroke' => '#FF0000',
    'status' => EstimateLocation::ACTIVE,
]);
```

### Creating an Estimate

```php
use Bongo\Estimate\Models\Estimate;
use Bongo\Estimate\Models\EstimateItem;

// Create the estimate
$estimate = Estimate::create([
    'first_name' => 'John',
    'last_name' => 'Smith',
    'email' => 'john@example.com',
    'phone' => '01234567890',
    'line_1' => '123 High Street',
    'city' => 'Norwich',
    'postcode' => 'NR1 1AA',
    'latitude' => 52.6309,
    'longitude' => 1.2974,
    'status' => Estimate::PENDING,
]);

// Add items to the estimate
$item = EstimateItem::create([
    'estimate_id' => $estimate->id,
    'estimate_service_id' => $service->id,
    'estimate_plan_id' => $plan->id,
    'areas' => json_encode($drawnAreas), // Polygon coordinates from map
    'total_area_m2' => 75.5,
    'status' => EstimateItem::PENDING,
]);
```

### Calculating Prices

```php
use Bongo\Estimate\Calculators\EstimateItemPriceCalculator;
use Bongo\Estimate\Models\EstimateItemPrice;

// Find the appropriate plan item for the area
$planItem = $plan->findItemByAreaM2($item->total_area_m2);

// Create the price record
$estimateItemPrice = EstimateItemPrice::create([
    'estimate_item_id' => $item->id,
    'estimate_plan_id' => $plan->id,
    'estimate_plan_item_id' => $planItem->id,
]);

// Calculate pricing
(new EstimateItemPriceCalculator())
    ->setEstimateItemPrice($estimateItemPrice)
    ->calculate()
    ->save();

// Access calculated values (stored in pence):
// $estimateItemPrice->cost_per_m2
// $estimateItemPrice->subtotal
// $estimateItemPrice->tax
// $estimateItemPrice->total
// $estimateItemPrice->price_per_month
```

### Actions

The package provides several action classes for common operations:

```php
use Bongo\Estimate\Actions\CalculateDistance;
use Bongo\Estimate\Actions\GenerateNumber;
use Bongo\Estimate\Actions\FindEstimate;

// Calculate distance between two points
$distanceMiles = CalculateDistance::inMiles($lat1, $lng1, $lat2, $lng2);
$distanceKm = CalculateDistance::inKilometers($lat1, $lng1, $lat2, $lng2);

// Generate estimate reference number
$number = GenerateNumber::forEstimate(); // Returns "KLC-0001"

// Find estimate by UUID
$estimate = FindEstimate::byUuid($uuid);
```

## Architecture Overview

> **For comprehensive architecture documentation, see [ARCHITECTURE.md](ARCHITECTURE.md)**

### Core Models

- **EstimateService** - Top-level service category (e.g., "Lawn Care", "Pest Control")
  - HasMany: EstimatePlan
- **EstimatePlan** - Pricing tier within a service (e.g., "Basic", "Enhanced", "Premier")
  - BelongsTo: EstimateService
  - HasMany: EstimatePlanItem
- **EstimatePlanItem** - Area-based pricing rule (e.g., "50m² = £28")
  - BelongsTo: EstimatePlan
- **EstimateLocation** - Geographic service coverage area with radius validation
- **Estimate** - Customer quote with contact details
  - HasMany: EstimateItem
- **EstimateItem** - Specific service/plan selection with drawn areas
  - BelongsTo: Estimate, EstimateService, EstimatePlan
  - HasMany: EstimateItemPrice
- **EstimateItemPrice** - Calculated pricing breakdown
  - BelongsTo: EstimateItem, EstimatePlan, EstimatePlanItem

### Key Patterns

- **Trait Composition**: Models use multiple traits for cross-cutting concerns
  - `HasUUID` - UUID primary keys
  - `HasStatus` - Status field with draft/pending/accepted/rejected states
  - `HasNumber` - Auto-generated reference numbers (e.g., "KLC-0001")
  - `HasAddress` - Full address with geocoding
  - `HasContact` - Contact information fields
  - `HasAreas` - JSON storage for map polygons
  - `HasSteps` - Multi-step workflow tracking
  - `CanBeExported` - Export status tracking

- **Action Classes**: Static utility classes for business logic
  - `CalculateDistance::inMiles()`, `CalculateDistance::inKilometers()`
  - `GenerateNumber::forEstimate()`
  - `FindEstimate::byUuid()`, `FindEstimateItem::byUuid()`
  - `GetDefaultService::execute()`

- **Calculator Pattern**: Fluent builder for complex calculations
  - `EstimateItemPriceCalculator` - Handles tax, minimum prices, monthly pricing

- **Event-Driven**: Events fired for all major operations
  - `EstimateUpdated`
  - `EstimateItemCreated`, `EstimateItemUpdated`, `EstimateItemDeleted`
  - `ServiceCreated`, `ServiceUpdated`, `ServiceDeleted`
  - `PlanCreated`, `PlanUpdated`, `PlanDeleted`
  - `LocationCreated`, `LocationUpdated`, `LocationDeleted`

## Development

### Setup Development Environment

```bash
# Install dependencies
composer install

# Run migrations
composer prepare

# Build the workbench
composer build
```

### Running Tests

```bash
# Run all tests
composer test

# Run tests with coverage
composer test:coverage

# Run a specific test
vendor/bin/phpunit --filter TestMethodName

# Run tests in parallel
composer test:parallel
```

### Code Quality

```bash
# Run PHPStan static analysis
composer analyse

# Format code with Laravel Pint
composer format
```

### Start Development Server

```bash
composer start
```

This starts the Orchestra Testbench server for development and testing.

## Upgrading from V3 to V4

If you're upgrading from version 3.x to 4.x, run the upgrade command:

```bash
php artisan estimate:upgrade-to-v4
```

This migrates the pricing structure from the old config-based system to the new database-driven model.

## Blade Components

### Backend Components

```blade
{{-- Service dropdown selector --}}
<x-estimate-service-dropdown />

{{-- Radius map for location management --}}
<x-radius-map />
```

## Routes

The package registers the following route groups:

### Frontend Routes (prefix: `/estimate`)
- `GET /estimate` - Start estimate process (cleanup + redirect)
- `GET /estimate/step-1` - Contact & service selection
- `POST /estimate/step-1/store` - Submit step 1
- `GET /estimate/step-2` - Area measurement (map drawing)
- `POST /estimate/step-2/store` - Submit step 2
- `GET /estimate/step-3` - Quote display with pricing
- `POST /estimate/search` - Postcode search & validation
- `GET /estimate/reset-address` - Reset address session
- `GET /estimate/reset-contact` - Reset contact session
- `GET /estimate/reset-item` - Reset item session

### Backend Routes (prefix: `/admin/estimates`)
- `GET /estimates` - Estimate listing (DataTable)
- `GET /estimates/{estimate}` - View estimate details
- `POST /estimates/{estimate}/update` - Update estimate
- `GET /estimates/{estimate}/items` - List estimate items
- `GET /estimates/{estimate}/items/{item}` - View item details
- Resource controllers for Services, Plans, Locations
- DataTable endpoints for all listings

## License

MIT License - See LICENSE file for details

## Support

For issues, feature requests, or contributions, please refer to the project repository.
